namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class UserManagerController(MyDbContext context) : ODataController
{
    private readonly MyDbContext db = context;

    [Permission("User Manager", PermissionType.View)]
    [Route("/odata/GetUserItems")]
    public async Task<IActionResult> GetUserItems(ODataQueryOptions<UserInfoItem> queryOptions, bool isExport)
    {
        var itemsQueryable = GetUserItemsInternal();
        itemsQueryable = queryOptions.ApplyTo(itemsQueryable) as IQueryable<UserInfoItem>;
        var items = itemsQueryable == null ? null : await itemsQueryable.ToListAsync();

        if (isExport)
            return File(Util.Excel.GetExportFileStream(items), "application/octet-stream");
        else
            return Ok(items);
    }

    private IQueryable<UserInfoItem> GetUserItemsInternal()
    {
        IQueryable<UserInfoItem>? itemsQueryable = null;

        var currentTime = DateTimeOffset.UtcNow;

        itemsQueryable = (
            from u in db.AppUsers
            join uoi in db.VwUserOverviewInfos on u.Id equals uoi.UserInfoId into uoiGroup
            from uoi in uoiGroup.DefaultIfEmpty()
            orderby u.DisplayName
            select new UserInfoItem
            {
                Id = u.Id,
                DisplayName = u.DisplayName,
                LegalName = u.LegalName ?? "",
                Title = u.Title,
                Email = u.Email,
                UserName = u.UserName ?? "",
                IsLockedOut = u.LockoutEnd == null ? "no" : u.LockoutEnd.Value < currentTime ? "no" : "yes",
                GroupNames = uoi.GroupNames,
            }
        );

        return itemsQueryable;
    }

    [Permission("User Manager", PermissionType.View)]
    [Route("GetGroupItems")]
    public async Task<IActionResult> GetGroupItems()
    {
        var groupItems = await (
            from sgi in db.SecurityGroups
            orderby sgi.Name
            select new SecurityGroupItem
            {
                Id = sgi.Id,
                Name = sgi.Name,
            }
            ).ToListAsync();

        return Ok(groupItems);
    }
}
