using Microsoft.AspNetCore.Identity;
using Microsoft.CodeAnalysis.CSharp.Syntax;
namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class UserDetailController : ODataController
{
    private readonly MyDbContext db;
    private readonly AuthorizationHelper authHelper;
    private readonly UserManager<AppUser> userManager;

    public UserDetailController(MyDbContext context, UserManager<AppUser> userManager)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
        this.userManager = userManager;
    }

    [Permission("User Manager", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData()
    {

        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "User Manager", PermissionType.Modify);
        var securityGroups = (from q in db.SecurityGroups orderby q.Name select new { q.Id, q.Name }).ToList();
        var result = new { hasModifyPermission, securityGroups };

        return Ok(result);
    }

    [Permission("User Manager", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetDetail(int id)
    {
        var currentTime = DateTimeOffset.UtcNow;

        var detail = await (
            from u in db.AppUsers
            orderby u.DisplayName
            where u.Id == id
            select new UserDetailItem
            {
                Id = u.Id,
                DisplayName = u.DisplayName,
                LegalName = u.LegalName,
                Title = u.Title,
                Email = u.Email,
                UserName = u.UserName ?? "",
                TempPass = null,
                IsLockedOut = u.LockoutEnd.GetValueOrDefault() > currentTime,
                GroupNames = string.Join(" | ", u.SecurityUserGroups.OrderBy(x => x.SecurityGroup.Name).Select(x => x.SecurityGroup.Name)),
                GroupIds = u.SecurityUserGroups.OrderBy(x => x.SecurityGroup.Name).Select(x => x.SecurityGroup.Id)
            }
        ).FirstAsync();

        return Ok(detail);
    }

    [Permission("User Manager", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> SaveDetail(UserDetailItem detail, Enums.SaveType saveType)
    {
        int resultId = 0;

        await db.Database.CreateExecutionStrategy().Execute(async () =>
        {
            using var dbContextTransaction = await db.Database.BeginTransactionAsync();

            //since appUserMgrItem comes from the injected UserManager,
            // it does not include the SecurityUserGroups navigation property
            AppUser? appUserMgrItem = null;

            //  If the saveType is not new, we attempt to get the existing AppUser item from database, and the ApplicationUser item from the built-in UserManager function.
            if (saveType != Enums.SaveType.New)
                appUserMgrItem = await userManager.FindByIdAsync(detail.Id.ToString());

            bool isCreating = false;
            if (appUserMgrItem == null)
            {
                isCreating = true;
                appUserMgrItem = new AppUser();
                appUserMgrItem.IsUsingTempPassword = true;
                appUserMgrItem.Theme = "light";
            }

            appUserMgrItem.UserName = detail.UserName;
            appUserMgrItem.LockoutEnd = detail.IsLockedOut ? new DateTime(9000, 1, 1).ToUniversalTime() : null;
            appUserMgrItem.DisplayName = detail.DisplayName;
            appUserMgrItem.LegalName = detail.LegalName;
            appUserMgrItem.Title = detail.Title;
            appUserMgrItem.Email = detail.Email;

            if (!string.IsNullOrWhiteSpace(detail.TempPass))
            {
                appUserMgrItem.IsUsingTempPassword = true;
                var validationResult = await userManager.PasswordValidators.First().ValidateAsync(userManager, appUserMgrItem, detail.TempPass);
                if (validationResult.Succeeded)
                    appUserMgrItem.PasswordHash = userManager.PasswordHasher.HashPassword(appUserMgrItem, detail.TempPass);
                else
                    throw new Exception(validationResult.Errors.First().Description);
            }

            IdentityResult? result;
            if (isCreating)
                result = await userManager.CreateAsync(appUserMgrItem, detail.TempPass ?? GetNewTempPass());
            else
                result = await userManager.UpdateAsync(appUserMgrItem);

            if (!result.Succeeded)
                throw new Exception(result.Errors.First().Description);

            resultId = appUserMgrItem.Id;

            //unlike appUserMgrItem, appUser includes the SecurityUserGroups navigation property
            var appUser = await db.AppUsers.Include(x => x.SecurityUserGroups).FirstAsync(x => x.Id == resultId);
            //remove existing groups so that they get completely re-inserted
            db.SecurityUserGroups.RemoveRange(appUser.SecurityUserGroups);
            //insert new groups
            appUser.SecurityUserGroups = [.. detail.GroupIds.Select(x => new SecurityUserGroup { UserId = detail.Id, SecurityGroupId = x })];
            await db.SaveChangesAsync();

            await dbContextTransaction.CommitAsync();
        });
        return Ok(resultId);
    }

    [Permission("User Manager", PermissionType.Modify)]
    [Route("[action]/{id}")]
    public async Task<IActionResult> DeleteDetail(int id)
    {
        AppUser? appUser = await userManager.FindByIdAsync(id.ToString());
        if (appUser == null)
            return NotFound();

        if (appUser != null)
            await userManager.DeleteAsync(appUser);

        return Ok();
    }

    [Permission("User Manager", PermissionType.View)]
    [Route("[action]")]
    public IActionResult GeneratePassword()
    {
        string tempPass = GetNewTempPass();
        return Ok(tempPass);
    }

    private static string GetNewTempPass()
    {
        PronounceablePasswordGenerator generator = new();
        string tempPass = generator.Generate(1, 7).First();
        return tempPass;
    }
}
