[Authorize]
[ApiController]
[Route("api/[controller]")]
public class GroupDetailController : ODataController
{
    private readonly MyDbContext db;
    private readonly AuthorizationHelper authHelper;

    public GroupDetailController(MyDbContext context)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
    }

    [Permission("User Manager", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData()
    {

        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "User Manager", PermissionType.Modify);
        var securityActions = (from q in db.SecurityActions orderby q.DisplayOrder descending, q.Name ascending select new { q.Id, q.Name, q.IsView }).ToList();
        var result = new { hasModifyPermission, securityActions };

        return Ok(result);
    }

    [Permission("User Manager", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetDetail(int id)
    {
        var detail = await (
            from sgi in db.SecurityGroups
                    .Include(x => x.SecurityGroupPermissions)
                    .ThenInclude(x => x.SecurityAction)
            orderby sgi.Name
            where sgi.Id == id
            select new SecurityGroupDetail
            {
                Id = sgi.Id,
                Name = sgi.Name,
                SecurityGroupPermissions = sgi.SecurityGroupPermissions.Select(sgp => new SecurityGroupPermissionDetail
                {
                    SecurityActionId = sgp.SecurityActionId,
                    IsAllowed = sgp.IsAllowed,
                    IsViewAllowed = sgp.IsViewAllowed,
                    IsModifyAllowed = sgp.IsModifyAllowed,
                })
            }
        ).FirstAsync();

        return Ok(detail);
    }

    public enum SaveType
    {
        New = 1,
        Normal = 2
    }

    [Permission("User Manager", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> SaveDetail(SecurityGroupDetail detail, SaveType saveType)
    {
        int resultId = 0;

        await db.Database.CreateExecutionStrategy().Execute(async () =>
        {
            using var dbContextTransaction = await db.Database.BeginTransactionAsync();
            SecurityGroup? securityGroup = null;

            // find an existing item, otherwise create a new item and add it to the db.
            if (saveType != SaveType.New)
            {
                securityGroup = (
                    from q in db.SecurityGroups
                    .Include(x => x.SecurityGroupPermissions)
                    where q.Id == detail.Id
                    select q
                ).FirstOrDefault();
            }

            if (securityGroup == null)
            {
                securityGroup = new SecurityGroup();
                db.SecurityGroups.Add(securityGroup);
            }
            else
            {
                //remove existing items so that they get completely re-inserted
                db.SecurityGroupPermissions.RemoveRange(securityGroup.SecurityGroupPermissions);
            }

            var d = detail;
            securityGroup.Name = d.Name;

            // Clear the existing Permissions, then re-insert the new ones from the detail,
            // this is so that we can remove any that were removed in the UI and add any new ones.
            securityGroup.SecurityGroupPermissions.Clear();
            var newPerms = d.SecurityGroupPermissions.Select(x => new SecurityGroupPermission
            {
                SecurityActionId = x.SecurityActionId,
                IsAllowed = x.IsAllowed,
                IsViewAllowed = x.IsViewAllowed,
                IsModifyAllowed = x.IsModifyAllowed,
            }).ToList() ?? new List<SecurityGroupPermission>();
            foreach (var perm in newPerms)
                securityGroup.SecurityGroupPermissions.Add(perm);
            await db.SaveChangesAsync();
            resultId = securityGroup.Id;

            await dbContextTransaction.CommitAsync();
        });

        return Ok(resultId);
    }

    [Permission("User Manager", PermissionType.View)]
    [Route("[action]/{id}")]
    public async Task<IActionResult> DeleteDetail(int id)
    {
        SecurityGroup securityGroup = await db.SecurityGroups.Where(x => x.Id == id).FirstAsync();
        db.Entry(securityGroup).State = EntityState.Deleted;

        await db.SaveChangesAsync();

        return Ok();
    }
}
