import { ChangeDetectionStrategy, Component, ElementRef, inject, ViewChild } from '@angular/core';
import { tap, catchError, switchMap, shareReplay, retry } from 'rxjs/operators';
import * as util from '../_shared/utils/util';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { GroupItem, UserManagerService } from './user-manager.service';
import { Title } from '@angular/platform-browser';
import { ActivatedRoute } from '@angular/router';
import { DialogService } from '@progress/kendo-angular-dialog';
import { State } from '@progress/kendo-data-query';
import { BehaviorSubject, Observable, of, Subject } from 'rxjs';
import { GridDataResult, SelectableSettings } from '@progress/kendo-angular-grid';
import { MessageService } from '../_shared/services/message.service';
import { NotifyService } from '../_shared/services/notify.service';
import { UserDetailComponent } from "../user-detail/user-detail.component";
import { GroupDetailComponent } from "../group-detail/group-detail.component";
import { saveAs } from '@progress/kendo-file-saver';
import { FastButtonComponent } from "../_shared/elements/fast-button.component";
import { FastHeaderComponent } from "../_shared/elements/fast-header.component";
import { FastGridComponent } from "../_shared/elements/fast-grid.component";

@Component({
  selector: 'app-user-manager',
  templateUrl: './user-manager.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
  standalone: true,
  imports: [FAST_KENDO_COMMON, FAST_PAGE_COMMON, UserDetailComponent, GroupDetailComponent, FastButtonComponent, FastHeaderComponent, FastGridComponent]
})
export class UserManagerComponent {
  @ViewChild("grid", { read: ElementRef }) kendoGridEl: ElementRef;

  util = util;
  icons = util.icons;
  gridScrollPosition: util.GridScrollPosition = { topPos: 0, leftPos: 0 };
  mySelection: number[] = [];

  userDetailId = 0;  // reference number for the user detail
  groupDetailId = 0; // reference number for the group detail

  userItems$: Observable<GridDataResult>;
  groupItems$: Observable<GroupItem[]>;
  title$: Observable<string>;

  gridLoading$ = new BehaviorSubject<boolean>(false);
  refreshUserItems$ = new BehaviorSubject<string>(null);
  refreshGroupItems$ = new BehaviorSubject<string>(null);

  userDetailOpened$ = new BehaviorSubject<boolean>(false);
  groupDetailOpened$ = new BehaviorSubject<boolean>(false);

  exporting$ = new BehaviorSubject<boolean>(false);
  exportClicked$ = new Subject();
  exportAction$: Observable<{ fileBlob: Blob; fileName: string; }>;

  selectableSettings: SelectableSettings = {
    checkboxOnly: false,
    mode: 'single',
    enabled: true
  }

  userState: State = {
    filter: null,
    group: null,
    skip: 0,
    sort: [{ field: 'UserName', dir: 'asc' }],
    take: 25
  };

  groupsState: State = {
    filter: null,
    group: null,
    skip: 0,
    sort: [{ field: 'Name', dir: 'asc' }],
    take: 50
  };

  service = inject(UserManagerService);
  messageService = inject(MessageService);
  titleService = inject(Title);
  dialogService = inject(DialogService);
  notify = inject(NotifyService);
  activatedRoute = inject(ActivatedRoute);

  constructor() {
    this.userItems$ = this.refreshUserItems$.pipe(
      tap(() => {
        this.gridLoading$.next(true);
      }),
      switchMap(() => {
        return this.service.getUserItems(this.userState);
      }),
      tap(() => {
        this.gridLoading$.next(false);
        util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.groupItems$ = this.refreshGroupItems$.pipe(
      switchMap(() => {
        return this.service.getGroupItems();
      }),
      tap(() => {
        util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.title$ = of('User Manager').pipe(
      tap((title) => util.trySetTitle(this.titleService, title))
    )

    this.exportAction$ = this.exportClicked$.pipe(
      switchMap(() => {
        this.exporting$.next(true);
        return this.service.exportItems(this.userState, 'Users.xlsx');
      }),
      tap(res => {
        saveAs(res.fileBlob, res.fileName);
        this.exporting$.next(false);
      }),
      shareReplay(1),
      catchError(err => {
        this.exporting$.next(false);
        return util.handleError(err, this.messageService)
      }), retry(10)
    )
  }

  userDataStateChange(state: State): void {
    this.gridScrollPosition = { topPos: 0, leftPos: 0 };
    util.fixKendoQueryFilter(state.filter);
    this.userState = state;
    this.refreshUserItems$.next(null);
  }

  groupDataStateChange(state: State): void {
    this.gridScrollPosition = { topPos: 0, leftPos: 0 };
    util.fixKendoQueryFilter(state.filter);
    this.groupsState = state;
    this.refreshGroupItems$.next(null);
  }

  addUserDetail(): void {
    this.openUserDetail(0);
  }

  openUserDetail(id: number): void {
    this.userDetailId = id;
    this.userDetailOpened$.next(true);
  }

  closeUserDetail() {
    this.userDetailOpened$.next(false);
    this.refreshUserItems$.next(null);
  }

  addGroupDetail(): void {
    this.openGroupDetail(0);
  }

  openGroupDetail(id: number): void {
    this.groupDetailId = id;
    this.groupDetailOpened$.next(true);
  }

  closeGroupDetail() {
    this.groupDetailOpened$.next(false);
    this.refreshGroupItems$.next(null);
  }

  export(): void {
    this.exportClicked$.next(null);
  }
}
