import { ChangeDetectionStrategy, Component, inject, input, output, ViewChild, ViewContainerRef } from '@angular/core';
import * as util from '../_shared/utils/util';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { Detail, RequiredData, UserWindowService } from './user-detail.service';
import { BehaviorSubject, catchError, combineLatest, map, Observable, of, retry, shareReplay, Subject, switchMap, take, tap } from 'rxjs';
import { DialogService, DialogSettings } from '@progress/kendo-angular-dialog';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';
import { Title } from '@angular/platform-browser';
import { Validators } from '@angular/forms';
import { ActivatedRoute } from '@angular/router';
import { MessageService } from '../_shared/services/message.service';
import { NotifyService } from '../_shared/services/notify.service';
import { FastButtonComponent } from "../_shared/elements/fast-button.component";
import { FastWindowComponent } from "../_shared/elements/fast-window.component";
import { FastToggleComponent } from "../_shared/elements/fast-toggle.component";
import { FastHRComponent } from "../_shared/elements/fast-hr.component";
import { FastMultiselectComponent } from "../_shared/elements/fast-multiselect.component";
import { FastLabelComponent } from "../_shared/elements/fast-label.component";
import { FastTextboxComponent } from "../_shared/elements/fast-textbox.component";

@Component({
  selector: 'app-user-detail',
  standalone: true,
  imports: [FAST_KENDO_COMMON, FAST_PAGE_COMMON, FastButtonComponent, FastWindowComponent, FastToggleComponent, FastHRComponent, FastMultiselectComponent, FastLabelComponent, FastTextboxComponent],
  templateUrl: './user-detail.component.html',
  styles: `:host { @apply absolute w-full h-full; }`,
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class UserDetailComponent {
  @ViewChild('button1', { read: ViewContainerRef }) button1: ViewContainerRef;
  @ViewChild('button2', { read: ViewContainerRef }) button2: ViewContainerRef;

  //generalized component variables and inputs
  util = util;
  icons = util.icons;
  hasModifyPermission = false;
  userId = input.required<number>();
  closed = output();
  detailForm: util.FormModel<Detail>;
  detailInitialValues: Detail;
  mySelection: number[] = [];

  //observables, behaviorsubjects, etc.
  detail$: Observable<Detail>;
  refreshDetail$ = new BehaviorSubject<void>(null)
  detailLoading$ = new BehaviorSubject<boolean>(true)
  requiredData$: Observable<RequiredData>;
  refreshRequiredData$ = new BehaviorSubject<util.RefreshType>(null);
  tempPass$: Observable<void>;
  refreshTempPass$ = new Subject<void>();
  securityGroups$: Observable<util.IdName[]>;
  filterSecurityGroups$ = new BehaviorSubject<string>(null);
  save$ = new Subject<[util.SaveType, util.AfterCompletedAction]>()
  saveResult$: Observable<[number, util.AfterCompletedAction]>;
  deleteResult$: Observable<object>;
  delete$ = new Subject()

  //injects
  service = inject(UserWindowService);
  fb = inject(CustomFormBuilder);
  messageService = inject(MessageService);
  titleService = inject(Title);
  dialogService = inject(DialogService);
  notify = inject(NotifyService);
  activatedRoute = inject(ActivatedRoute);

  //constructor containing observable definition
  constructor() {
    this.detailForm = this.getDetailForm();
    this.detailInitialValues = this.detailForm.getRawValue() as Detail;

    this.requiredData$ = this.refreshRequiredData$.pipe(
      tap(() => this.detailLoading$.next(true)),
      switchMap(refreshType => {
        return combineLatest([this.service.getRequiredData(), of(refreshType)]);
      }),
      map(([requiredData, refreshType]) => {
        if (refreshType === util.RefreshType.SelfOnly)
          this.detailLoading$.next(false);
        return requiredData;
      }),
      tap(requiredData => {
        this.hasModifyPermission = requiredData.hasModifyPermission;
        util.focusInputTarget();
      }),
      shareReplay(1),
      catchError(err => {
        this.detailLoading$.next(false);
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.detail$ = this.refreshDetail$.pipe(
      switchMap(() => {
        this.detailLoading$.next(true);
        this.detailForm.reset();
        if (this.userId() == 0)
          return this.service.generatePassword();
        else
          return of(null);
      }),
      switchMap(tempPass => {
        if (this.userId() === 0) {
          this.detailInitialValues.tempPass = tempPass;
          return of(this.detailInitialValues);
        }
        else
          return this.service.getDetail(this.userId());
      }),
      map(detail => {
        this.detailForm.setValue(detail);
        return detail;
      }),
      tap(() => {
        this.detailLoading$.next(false);
        util.focusInputTarget();
      }),
      shareReplay(1),
      catchError(err => {
        this.detailLoading$.next(false);
        return util.handleError(err, this.messageService);
      }), retry(10)
    )

    this.tempPass$ = this.refreshTempPass$.pipe(
      switchMap(() => {
        this.detailLoading$.next(true);
        return this.service.generatePassword();
      }),
      map(result => {
        this.detailForm.patchValue({ tempPass: result });
      }),
      tap(() => {
        this.detailLoading$.next(false);
      }),
      shareReplay(1),
      catchError(err => {
        this.detailLoading$.next(false);
        return util.handleError(err, this.messageService);
      }), retry(10)
    )

    this.saveResult$ = this.save$.pipe(
      switchMap(([saveType, completedAction]) => {
        this.detailLoading$.next(true);
        const itemToSave: Detail = this.detailForm.value as Detail;
        return combineLatest([this.service.saveDetail(itemToSave, saveType), of(completedAction)]);
      }),
      tap(([saveResult, action]) => {
        this.notify.success('save successful');

        if (action === util.AfterCompletedAction.CloseDetail) {
          this.mySelection = [saveResult];
          this.detailLoading$.next(false);
          this.closed.emit();
        }
        else
          this.refreshDetail$.next();
      }),
      shareReplay(1),
      catchError(err => {
        this.detailLoading$.next(false);
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.deleteResult$ = this.delete$.pipe(
      switchMap(() => {
        this.detailLoading$.next(true);
        const itemToDelete: Detail = this.detailForm.getRawValue();
        return this.service.deleteDetail(itemToDelete.id);
      }),
      tap(() => {
        this.notify.success('delete successful');
        this.detailLoading$.next(false);
        this.closed.emit();
      }),
      shareReplay(1),
      catchError(err => {
        this.detailLoading$.next(false);
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.securityGroups$ = this.filterSecurityGroups$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'securityGroups'));
  }

  //methods
  getDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<Detail> = this.fb.group({
      id: fb.ctr(0, Validators.required),
      displayName: fb.ctr(null, Validators.required),
      legalName: fb.ctr(null, Validators.required),
      title: fb.ctr(null),
      email: fb.ctr(null, Validators.email),
      userName: fb.ctr(null, Validators.required),
      tempPass: fb.ctr(null),
      isLockedOut: fb.ctr(false),
      groupNames: fb.ctr(false),
      groupIds: fb.ctr([])
    })

    return fg;
  }

  generatePassword() {
    this.refreshTempPass$.next(null);
  }

  closeClicked() {
    this.detailLoading$.next(false);
    this.closed.emit();
  }

  save = (saveType: util.SaveType, action: util.AfterCompletedAction) => {
    this.detailForm.markAllAsTouched();
    console.log(this.detailForm);
    if (this.detailForm.valid)
      this.save$.next([saveType, action]);
    else
      this.notify.error("validation failed");
  }

  delete(): void {
    const deleteConfirmSettings: DialogSettings = {
      title: "Please confirm",
      content: "Are you sure you want to delete this item?",
      actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
      cssClass: 'utilPrompt'
    }

    this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
      if (util.getDialogAction(result) === util.dialogAction.Yes)
        this.delete$.next(null);
    });
  }
}
