import { ChangeDetectionStrategy, Component, inject, input, output, ViewChild } from '@angular/core';
import * as util from '../_shared/utils/util';
import { Detail, GroupDetailService, RequiredData, SecurityPermission } from './group-detail.service';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';
import { MessageService } from '../_shared/services/message.service';
import { Title } from '@angular/platform-browser';
import { DialogService, DialogSettings } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../_shared/services/notify.service';
import { ActivatedRoute } from '@angular/router';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { BehaviorSubject, catchError, combineLatest, map, Observable, of, retry, shareReplay, Subject, switchMap, take, tap } from 'rxjs';
import { UntypedFormArray, Validators } from '@angular/forms';
import { ComboBoxComponent } from '@progress/kendo-angular-dropdowns';
import { FastButtonComponent } from "../_shared/elements/fast-button.component";
import { FastWindowComponent } from "../_shared/elements/fast-window.component";
import { FastToggleComponent } from "../_shared/elements/fast-toggle.component";
import { FastHRComponent } from "../_shared/elements/fast-hr.component";
import { FastComboboxComponent } from "../_shared/elements/fast-combobox.component";
import { FastLabelComponent } from "../_shared/elements/fast-label.component";
import { FastTextboxComponent } from "../_shared/elements/fast-textbox.component";

@Component({
  selector: 'app-group-detail',
  standalone: true,
  imports: [FAST_KENDO_COMMON, FAST_PAGE_COMMON, FastButtonComponent, FastWindowComponent, FastToggleComponent, FastHRComponent, FastComboboxComponent, FastLabelComponent, FastTextboxComponent],
  templateUrl: './group-detail.component.html',
  styles: `:host { @apply absolute w-full h-full; }`,
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class GroupDetailComponent {
  @ViewChild("addSecurityPermBox") addSecurityPermBox: ComboBoxComponent;

  //generalized component variables and inputs
  util = util;
  icons = util.icons;
  hasModifyPermission = false;
  groupId = input.required<number>();
  closed = output();
  localRequiredData: RequiredData;
  detailForm: util.FormModel<Detail>;
  detailInitialValues: Detail;
  securityGroupPermissions: UntypedFormArray;
  mySelection: number[] = [];

  //observables, behaviorsubjects, etc.
  detail$: Observable<Detail>;
  refreshDetail$ = new BehaviorSubject<void>(null)
  detailLoading$ = new BehaviorSubject<boolean>(true)
  requiredData$: Observable<RequiredData>;
  refreshRequiredData$ = new BehaviorSubject<util.RefreshType>(null);
  securityActions$: Observable<util.IdName[]>;
  filterSecurityActions$ = new BehaviorSubject<string>(null);

  save$ = new Subject<[util.SaveType, util.AfterCompletedAction]>()
  saveResult$: Observable<[number, util.AfterCompletedAction]>;
  deleteResult$: Observable<object>;
  delete$ = new Subject()

  //injects
  service = inject(GroupDetailService);
  fb = inject(CustomFormBuilder);
  messageService = inject(MessageService);
  titleService = inject(Title);
  dialogService = inject(DialogService);
  notify = inject(NotifyService);
  activatedRoute = inject(ActivatedRoute);

  //constructor containing observable definition
  constructor() {
    this.detailForm = this.getDetailForm();
    this.detailInitialValues = this.detailForm.getRawValue() as Detail;

    this.requiredData$ = this.refreshRequiredData$.pipe(
      tap(() => this.detailLoading$.next(true)),
      switchMap(refreshType => {
        return combineLatest([this.service.getRequiredData(), of(refreshType)]);
      }),
      map(([requiredData, refreshType]) => {
        this.localRequiredData = requiredData;
        if (refreshType === util.RefreshType.SelfOnly)
          this.detailLoading$.next(false);
        return requiredData;
      }),
      tap(requiredData => {
        this.hasModifyPermission = requiredData.hasModifyPermission;
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.detail$ = this.refreshDetail$.pipe(
      switchMap(() => {
        if (this.groupId() === 0) {
          return of(this.detailInitialValues);
        }
        else
          return this.service.getDetail(this.groupId());
      }),
      map(detail => {
        this.createPermissions(detail);
        this.detailForm.setValue(detail);
        return detail;
      }),
      tap(() => {
        this.detailLoading$.next(false);
        util.focusInputTarget();
      }),
      shareReplay(1),
      catchError(err => {
        this.detailLoading$.next(false);
        return util.handleError(err, this.messageService);
      }), retry(10)
    )

    this.saveResult$ = this.save$.pipe(
      switchMap(([saveType, completedAction]) => {
        this.detailLoading$.next(true);
        const itemToSave: Detail = this.detailForm.value as Detail;
        return combineLatest([this.service.saveDetail(itemToSave, saveType), of(completedAction)]);
      }),
      tap(([saveResult, action]) => {
        this.notify.success('save successful');

        if (action === util.AfterCompletedAction.CloseDetail) {
          this.mySelection = [saveResult];
          this.closedClicked();
        }
        else
          this.refreshDetail$.next();
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.deleteResult$ = this.delete$.pipe(
      switchMap(() => {
        this.detailLoading$.next(true);
        const itemToDelete: Detail = this.detailForm.getRawValue();
        return this.service.deleteDetail(itemToDelete.id);
      }),
      tap(() => {
        this.notify.success('delete successful');
        this.closedClicked();
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )


    this.securityActions$ = this.filterSecurityActions$.pipe(util.filterSpecials(this.detailLoading$, this.requiredData$, 'securityActions', 'name'));
  }

  //methods

  getDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<Detail> = this.fb.group({
      id: fb.ctr(0, Validators.required),
      name: fb.ctr('', Validators.required),
      securityGroupPermissions: fb.arr([]),
    })

    return fg;
  }

  getPermissionDetailForm(sp: SecurityPermission) {
    const fb = this.fb;
    const fg: util.FormModel<SecurityPermission> = this.fb.group({
      securityActionId: fb.ctr(sp ? sp.securityActionId : 0, Validators.required),
      isAllowed: fb.ctr(sp ? sp.isAllowed : true, Validators.required),
      isViewAllowed: fb.ctr(sp ? sp.isViewAllowed : true, Validators.required),
      isModifyAllowed: fb.ctr(sp ? sp.isModifyAllowed : true, Validators.required),
    })

    return fg;
  }

  createPermissions(detail: Detail) {
    this.securityGroupPermissions = this.detailForm.get('securityGroupPermissions') as UntypedFormArray;
    this.securityGroupPermissions.clear();
    for (let i = 0; i < detail.securityGroupPermissions.length; i++) {
      this.securityGroupPermissions.push(this.getPermissionDetailForm(detail.securityGroupPermissions[i]));
    }
  }

  getPermissionName(permissionId: number) {
    const permissionName = this.localRequiredData.securityActions.find(x => x.id === permissionId).name;
    return permissionName;
  }

  addPermission(permissionId: number) {
    if (!permissionId)
      return;

    const newPermission = {
      securityActionId: permissionId,
      isAllowed: true,
      isViewAllowed: true,
      isModifyAllowed: true
    }
    if (!(this.securityGroupPermissions.value.find((newPermission: { securityActionId: number; }) => newPermission.securityActionId === permissionId)))
      this.securityGroupPermissions.push(this.getPermissionDetailForm(newPermission));

  }

  removePermission(index: number) {
    this.securityGroupPermissions.removeAt(index);
  }

  isPermissionViewable(permissionId: number) {
    const permission = this.localRequiredData.securityActions.find(x => x.id === permissionId)
    return permission.isView === true;
  }

  closedClicked() {
    this.closed.emit();
  }

  save = (saveType: util.SaveType, action: util.AfterCompletedAction) => {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid)
      this.save$.next([saveType, action]);
    else
      this.notify.error("validation failed");
  }

  delete(): void {
    const deleteConfirmSettings: DialogSettings = {
      title: "Please confirm",
      content: "Are you sure you want to delete this item?",
      actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
      cssClass: 'utilPrompt'
    }

    this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
      if (util.getDialogAction(result) === util.dialogAction.Yes)
        this.delete$.next(null);
    });
  }
}
