import { ChangeDetectionStrategy, ChangeDetectorRef, Component, computed, inject, input, model, output } from '@angular/core';
import { FAST_KENDO_COMMON } from '../../app.config';
import { ControlValueAccessor, NG_VALUE_ACCESSOR } from '@angular/forms';
import { NumberFormatOptions } from '@progress/kendo-angular-intl';

@Component({
  selector: 'fast-numerictextbox',
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [FAST_KENDO_COMMON],
  providers: [{
    provide: NG_VALUE_ACCESSOR,
    multi: true,
    useExisting: FastNumericTextBoxComponent
  }],
  template: `
  <kendo-numerictextbox
    class="flex border-2 border-solid rounded-sm h-7.5
    [&_button]:border-none
    [&_button]:w-5.5
    [&_input]:pr-0"
    [min]="min()"
    [max]="max()"
    [class]="conditionalClasses()"
    [autoCorrect]="autoCorrect()"
    [format]="format()"
    [spinners]="spinners()"
    [step]="step()"
    [tabindex]="tabindex()"
    [value]="value()"
    (valueChange)="onValueChange($event)"
    [disabled]="disabled()"
  />
  `
})
export class FastNumericTextBoxComponent implements ControlValueAccessor {
  cdr = inject(ChangeDetectorRef);

  min = input<number | null>(null);
  max = input<number | null>(null);
  tabindex = input<number>(0);
  spinners = input<boolean>(true);
  step = input<number>(1);
  autoCorrect = input<boolean>(false);
  valueChange = output<number | null>();
  format = input<string | NumberFormatOptions>();
  value = model<number | null>(null);
  disabled = model<boolean>(false);

  onChange: (value: number | null) => void = () => { };
  onTouched: () => void;

  writeValue(value: number | null) {
    this.value.set(value);
    this.cdr.markForCheck();
  }

  registerOnChange(fn: () => void) {
    this.onChange = fn;
  }

  registerOnTouched(fn: () => void) {
    this.onTouched = fn;
  }

  setDisabledState(isDisabled: boolean) {
    this.disabled.set(isDisabled);
  }

  onValueChange(newValue: number | null) {
    this.value.set(newValue);
    this.onChange(newValue);
  }

  conditionalClasses = computed(() => {
    const classes = [] as string[];

    classes.push(...this.getLightBaseClasses());
    classes.push(...this.getDarkBaseClasses());
    classes.push(...this.getLightActiveClasses());
    classes.push(...this.getDarkActiveClasses());

    const conditionalClasses = this.getConditionalClassesFromArrays(classes);
    return conditionalClasses;
  });

  getConditionalClassesFromArrays(classArray: string[]): { [key: string]: boolean } {
    const classes: { [key: string]: boolean } = {};
    classArray.forEach(className => {
      classes[className] = true;
    });
    return classes;
  }

  getLightBaseClasses() {
    return [
      "[&_button]:text-base-black-1000",
      "[&_button]:bg-base-white-250"];
  }

  getDarkBaseClasses() {
    return [
      "dark:[&_button]:text-base-white-500",
      "dark:[&_button]:bg-alt-gray-1000"];
  }

  getLightActiveClasses() {
    return [
      "[&_button]:active:bg-base-white-1000"];
  }

  getDarkActiveClasses() {
    return [
      "dark:[&_button]:active:bg-alt-blue-250"];
  }
}
