import { ChangeDetectionStrategy, ChangeDetectorRef, Component, computed, inject, input, model, output } from '@angular/core';
import { FAST_KENDO_COMMON } from '../../app.config';
import { ControlValueAccessor, NG_VALUE_ACCESSOR } from '@angular/forms';

@Component({
  selector: 'fast-multiselect',
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [FAST_KENDO_COMMON],
  providers: [{
    provide: NG_VALUE_ACCESSOR,
    multi: true,
    useExisting: FastMultiselectComponent
  }],
  template: `
  <kendo-multiselect
    [data]="data()"
    [textField]="textField()"
    [valueField]="valueField()"
    [valuePrimitive]="valuePrimitive()"
    [filterable]="filterable()"
    [value]="value()"
    [size]="'small'"
    [placeholder]="placeholder()"
    (filterChange)="onFilterChange($event)"
    (valueChange)="onValueChange($event)"
    [class]="conditionalClasses()"
    [disabled]="disabled()"
  />
  `
})
export class FastMultiselectComponent<T extends object> implements ControlValueAccessor {
  cdr = inject(ChangeDetectorRef);

  data = input.required<T[]>();
  textField = input<string>("");
  valueField = input<string>("");
  valuePrimitive = input<boolean>(true);
  filterable = input<boolean>(true);
  placeholder = input<string>("");
  filterChange = output<string>();
  valueChange = output<T[]>();
  value = model<T[] | null>(null);
  disabled = model<boolean>(false);

  onChange: (value: T[] | null) => void = () => { };
  onTouched = () => { };

  writeValue(value: T[] | null) {
    this.value.set(value);
    this.cdr.markForCheck();
  }

  registerOnChange(fn: () => void) {
    this.onChange = fn;
  }

  registerOnTouched(fn: () => void) {
    this.onTouched = fn;
  }

  setDisabledState(isDisabled: boolean) {
    this.disabled.set(isDisabled);
  }

  onValueChange(newValue: T[] | null) {
    this.value.set(newValue);
    this.onChange(newValue);
  }

  onFilterChange(event: string): void {
    this.filterChange.emit(event);
  }

  conditionalClasses = computed(() => {
    const classes = [] as string[];

    classes.push(...this.getCommonClasses());
    classes.push(...this.getLightBaseClasses());
    classes.push(...this.getDarkBaseClasses());

    const conditionalClasses = this.getConditionalClassesFromArrays(classes);
    return conditionalClasses;
  });


  getConditionalClassesFromArrays(classArray: string[]): { [key: string]: boolean } {
    const classes: { [key: string]: boolean } = {};
    classArray.forEach(className => {
      classes[className] = true;
    });
    return classes;
  }

  getCommonClasses() {
    return [
      "items-center",
      "h-full",
      "overflow-visible",
      "[&_.k-input-inner]:m-0",
      "[&_.k-input-inner]:pl-2",
      "[&_.k-input-inner]:p-px",
      "[&_.k-input-values]:gap-0.5",
      "[&_.k-chip-list_.k-chip]:leading-normal",
      "[&_.k-chip-list_.k-chip]:mt-0.5",
      "[&_.k-chip-list_.k-chip]:mb-0",
      "[&_.k-chip-list_.k-chip]:p-0",
      "[&_.k-chip-list_.k-chip]:bg-transparent",
      "[&_.k-chip-list_.k-chip-content]:leading-4",
      "[&_.k-chip-list_.k-chip-content]:m-0",
      "[&_.k-chip-list_.k-chip-content]:p-0",
      "[&_.k-chip-list_.k-chip.k-chip-solid-base]:bg-transparent",
      "[&_.k-chip-label]:px-0.5",
      "[&_.k-chip-label]:py-2"
    ];
  }

  getLightBaseClasses() {
    return [
      "[&_.k-chip-list_.k-chip.k-chip-solid-base]:border-base-gray-500",
      "[&_.k-chip-list_.k-chip]:text-base-black-1000",
      "[&_.k-chip-list_.k-chip]:hover:text-base-black-250",
      "[&_.k-chip-list_.k-chip.k-chip-solid-base]:hover:bg-base-gray-250",
      "[&_.k-chip-actions]:text-base-gray-1000",
      "[&_.k-chip-actions]:hover:text-base-red-1000"];
  }

  getDarkBaseClasses() {
    return [
      "dark:[&_.k-chip-list_.k-chip.k-chip-solid-base]:border-base-gray-1000",
      "dark:[&_.k-chip-list_.k-chip]:text-base-white-500",
      "dark:[&_.k-chip-list_.k-chip]:hover:text-base-white-1000",
      "dark:[&_.k-chip-list_.k-chip.k-chip-solid-base]:hover:bg-alt-blue-250",
      "dark:[&_.k-chip-actions]:text-base-gray-1000",
      "dark:[&_.k-chip-actions]:hover:text-alt-red-500"];
  }
}
