<#
.SYNOPSIS
Sets the last modified timestamp of files in a specified directory to a specified timestamp.
#>
function Set-LastModifiedTimestamp {
  param (
    [string]$directoryPath = (Get-Location).Path,
    [string]$fileFilter = "*.*",
    [datetime]$targetTimestamp
  )

  # Get all the files matching the filter under the directory
  $files = Get-ChildItem -Path $directoryPath -Filter $fileFilter -Recurse

  # Iterate through each file and set the last modified time to the target timestamp
  foreach ($file in $files) {
    # Set the last modified time to the target timestamp
    Set-ItemProperty -Path $file.FullName -Name LastWriteTime -Value $targetTimestamp
  }
}

<#
.SYNOPSIS
Removes files with the specified timestamp in a specified directory.
#>
function Remove-TimestampedFiles {
  param (
    [string]$directoryPath = (Get-Location).Path,
    [string]$fileFilter = "*.*",
    [datetime]$targetTimestamp
  )

  # Get all the files matching the filter under the directory
  $files = Get-ChildItem -Path $directoryPath -Filter $fileFilter -Recurse

  foreach ($file in $files) {
    # Check if the file's LastWriteTime matches the target timestamp
    if ($file.LastWriteTime -eq $targetTimestamp) {
      # Remove the file
      Remove-Item -Path $file.FullName -Force
      Write-Host "Removed file: $($file.FullName)"
    }
  }
}

<#
.SYNOPSIS
Changes the line endings of files in a specified directory from Windows format (CRLF) to Unix/Linux format (LF).
#>
function Convert-LineEndings {
  param (
    [string]$directoryPath = (Get-Location).Path,
    [string]$fileFilter = "*.*"
  )

  # Get all the files matching the filter under the directory
  $files = Get-ChildItem -Path $directoryPath -Filter $fileFilter -Recurse

  # Iterate through each file and replace CRLF with LF
  foreach ($file in $files) {
    # Read the content of the file
    $content = Get-Content -Path $file.FullName -Raw

    # Replace CRLF with LF
    $content = $content -replace "`r`n", "`n"

    # Write the updated content back to the file without an "extra" newline at the end
    # There will stil be a newline at the end of the file, but not an "extra" one
    # Ensure the encoding remains as UTF-8
    $content | Out-File -FilePath $file.FullName -Encoding UTF8 -NoNewline
  }

  Write-Host "Line endings have been converted in all $fileFilter files under the $directoryPath directory."
}
