import { Component, ChangeDetectionStrategy, ViewEncapsulation, Input, inject } from '@angular/core';
import { CommonData, CommonService, ScreenItem } from '../services/common.service';
import { MessageService } from '../services/message.service';
import { Router } from '@angular/router';
import { BehaviorSubject, Subject, Observable } from 'rxjs';
import { catchError, map, tap, switchMap } from 'rxjs/operators';
import { DragEndEvent, KENDO_SORTABLE } from '@progress/kendo-angular-sortable';
import * as util from '../utils/util';
import { AsyncPipe } from '@angular/common';
import { KENDO_SVGICON } from '@progress/kendo-angular-icons';
import { ActiveClassDirective } from '../directives/active-class-directive';
import { HoverClassDirective } from '../directives/hover-class-directive';

@Component({
    selector: 'global-favorites[isOverlay]',
    templateUrl: './global-favorites.component.html',
    styleUrls: ['./global-favorites.component.scss'],
    changeDetection: ChangeDetectionStrategy.OnPush,
    encapsulation: ViewEncapsulation.None,
    imports: [AsyncPipe, KENDO_SVGICON, KENDO_SORTABLE, ActiveClassDirective, HoverClassDirective]
})
export class GlobalFavoritesComponent {
  @Input() isOverlay: boolean = true;

  commonService = inject(CommonService);
  messageService = inject(MessageService);
  router = inject(Router);
  icons = util.icons;

  favorites$: Observable<CommonData>
  saveFavorites$ = new Subject()
  addFavorite$ = new Subject<string>()
  favoritesLoading$ = new BehaviorSubject<boolean>(true)
  saveFavoritesResult$: Observable<void>
  addFavoriteResult$: Observable<void>

  localFavorites: ScreenItem[] = [];
  isFavoriteDragging: boolean = false;

  constructor() {
    this.favorites$ = this.commonService.commonData$.pipe(
      tap((data) => {
        this.localFavorites = data.favorites;
        this.favoritesLoading$.next(false);
      })
    )

    this.saveFavoritesResult$ = this.saveFavorites$.pipe(
      switchMap(() => {
        this.favoritesLoading$.next(true);
        return this.commonService.saveFavorites(this.localFavorites);
      }),
      map(() => {
        this.favoritesLoading$.next(false);
      }),
      catchError(err => {
        this.favoritesLoading$.next(false);
        return util.handleError(err, this.messageService)
      })
    )

    this.addFavoriteResult$ = this.addFavorite$.pipe(
      switchMap(url => {
        this.favoritesLoading$.next(true);
        url = url || window.location.href;
        return this.commonService.addFavorite(url);
      }),
      map(result => {
        if (result) {
          this.localFavorites.push(result);
          this.localFavorites = [...this.localFavorites];
        }
        this.favoritesLoading$.next(false);
      }),
      catchError(err => {
        this.favoritesLoading$.next(false);
        return util.handleError(err, this.messageService)
      })
    )
  }

  addFavorite(url: string = null) {
    this.addFavorite$.next(url);
  }

  removeFavorite(favoriteId: number) {
    this.localFavorites = this.localFavorites.filter(x => x.id !== favoriteId);
    this.saveFavorites$.next(null);
  }

  favoriteClick(screenItem: ScreenItem) {
    if (!this.isFavoriteDragging)
      this.externalNavigate(screenItem.path);
  }

  favoriteDragStart() {
    this.isFavoriteDragging = true;
  }

  favoriteDragEnd(event: DragEndEvent) {
    if (event.index === event.oldIndex)
      //if dragged to same position then immediately set dragging to false so that tiny movements still trigger a navigation
      this.isFavoriteDragging = false;
    else {
      //if dragged to new position then delay setting dragging to false since we don't want actual drags to navigate to a new url
      setTimeout(() => this.isFavoriteDragging = false);
      this.saveFavorites$.next(null);
    }
  }

  externalNavigate(path: string) {
    this.router.navigate(['/externalRedirect', { externalUrl: path }], {
      skipLocationChange: true,
    });
  }
}
