import { Component, ChangeDetectionStrategy, Input, Output, EventEmitter, OnInit, OnDestroy } from '@angular/core';
import * as util from '../../utils/util';
import { WinHeightPipe } from '../../pipes/win-height.pipe';
import { WinWidthPipe } from '../../pipes/win-width.pipe';
import { KENDO_WINDOW } from '@progress/kendo-angular-dialog';
import { UrlParamsPipe } from '../../pipes/url-params.pipe';
import { KENDO_SVGICON } from '@progress/kendo-angular-icons';

@Component({
  selector: 'fast-frame',
  imports: [WinHeightPipe, WinWidthPipe, UrlParamsPipe, KENDO_WINDOW, KENDO_SVGICON],
  template: `
    <div class="overlay2"></div>
    <kendo-window class="window2" (close)="close()" [height]="height | winHeight"
      [width]="width | winWidth">
      <kendo-window-titlebar>
        <div class="row-g">
          <div class="rowItem-lc">{{title}}</div>
          <div class="rowItem-g"></div>
          <div class="k-window-titlebar-actions">
            <button kendoWindowMinimizeAction></button>
            <button kendoWindowMaximizeAction></button>
            <button kendoWindowRestoreAction></button>
            <button kendoWindowCloseAction></button>
          </div>
        </div>
      </kendo-window-titlebar>
      <div class="col-g">
        <!-- use an iframe even though you could use a component because kendo window popups inside the componenent don't position correctly otherwise -->
        <!-- this also helps with escape key handling and error encapsulation -->
        <iframe class="colItem-g"
          [src]="src | urlParams : externalParams"></iframe>
        @if (isCloseVisible) {
          <hr />
          <div class=" colItem bottomSection">
          <div class="row-g">
            <div class="rowItem-g"></div>
            <div class="rowItem-r">
              <button type="button" class="k-button button-md" (click)="close()">
              <kendo-svg-icon [icon]="icons.cancelIcon"></kendo-svg-icon>Close
              </button>
            </div>
          </div>
        </div>
        }
      </div>
    </kendo-window>
  `,
  styles: [`
    :host {
      position: absolute;
    }
  `],
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class FastFrameComponent implements OnInit, OnDestroy {
  @Input() title: string;
  @Input() src: string;
  @Input() externalParams: unknown;
  @Input() width: number = 1800;
  @Input() height: number = 0;
  @Input() isCloseVisible: boolean = true;
  @Output() closed = new EventEmitter();

  // this is needed so we can add and remove the same event listener correctly
  private boundHandleMessage: (event: MessageEvent) => void;

  icons = util.icons;

  ngOnInit(): void {
    // console.log('FastFrameComponent.ngOnInit()');
    this.boundHandleMessage = this.handleMessage.bind(this);
    this.addMessageEventListener();
  }

  ngOnDestroy(): void {
    // console.log('FastFrameComponent.ngOnDestroy()');
    this.removeMessageEventListener();
  }

  handleMessage(event: MessageEvent): void {
    // console.log('FastFrameComponent.handleMessage()');
    // console.log(event);
    if (event.data === 'close') {
      // console.log('FastFrameComponent:  window close message received');
      this.close();
    }
  }

  addMessageEventListener(): void {
    // console.log('FastFrameComponent.addMessageEventListener()');
    window.addEventListener('message', this.boundHandleMessage, { once: true });
  }

  removeMessageEventListener(): void {
    // console.log('FastFrameComponent.removeMessageEventListener()');
    window.removeEventListener('message', this.boundHandleMessage);
  }

  close(): void {
    // console.log('FastFrameComponent.close()');
    this.closed.emit();
  }
}
