import { CdkPortalOutlet, TemplatePortal } from '@angular/cdk/portal';
import { ChangeDetectionStrategy, Component, computed, inject, signal, TemplateRef, ViewContainerRef, output, } from '@angular/core';

@Component({
  selector: 'fast-tip-container',
  host: {
    '[class]': 'this.position()',
  },
  imports: [CdkPortalOutlet],
  template: `
<div role="tooltip" [class]="classes()" [id]="id()"
     (mouseenter)="containerMouseEnter.emit()" (mouseleave)="containerMouseLeave.emit()">
  @if (contentPortal()) {
  <ng-container [cdkPortalOutlet]="contentPortal()"></ng-container>
  } @else {
  {{ message() }}
  }
</div>
  `,
  styles: `
.tooltip {
  padding: 4px 6px;
  border-radius: 4px;
  font-size: 14px;
  transition: opacity 0.3s ease;
  position: relative;
  user-select: none;              /* reduce accidental selection */
  -webkit-touch-callout: none;    /* avoid iOS callout on long-press */
  touch-action: manipulation;     /* hint to reduce delays/gestures */
}

:host::after {
  content: '';
  position: absolute;
  width: 0;
  height: 0;
  border-style: solid;
}

:host.top::after {
  top: calc(100% - 1px);
  left: 50%;
  transform: translateX(-50%);
  border-width: 6px 6px 0 6px;
  border-color: var(--color-inverse) transparent transparent transparent;
}

:host.right::after {
  top: 50%;
  left: -5px;
  transform: translateY(-50%);
  border-width: 6px 6px 6px 0;
  border-color: transparent var(--color-inverse) transparent transparent;
}

:host.bottom::after {
  bottom: calc(100% - 1px);
  left: 50%;
  transform: translateX(-50%);
  border-width: 0 6px 6px 6px;
  border-color: transparent transparent var(--color-inverse) transparent;
}

:host.left::after {
  top: 50%;
  right: -5px;
  transform: translateY(-50%);
  border-width: 6px 0 6px 6px;
  border-color: transparent transparent transparent var(--color-inverse);
}

.dark :host.top::after {
  border-color: var(--color-dark-inverse) transparent transparent transparent;
}

.dark :host.right::after {
  border-color: transparent var(--color-dark-inverse) transparent transparent;
}

.dark :host.bottom::after {
  border-color: transparent transparent var(--color-dark-inverse) transparent;
}

.dark :host.left::after {
  border-color: transparent transparent transparent var(--color-dark-inverse);
}
  `,
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class FastTipContainerComponent {
  message = signal<string>('');

  position = signal<string>('top');

  bg = signal<string>('opaque');

  pushTooltip = signal<boolean>(true);

  content = signal<TemplateRef<unknown> | undefined>(undefined);

  context = signal<unknown>(undefined);

  vf = inject(ViewContainerRef);

  id = signal<string>('');

  containerMouseEnter = output<void>();
  containerMouseLeave = output<void>();

  contentPortal = computed(() => {
    const content = this.content();
    const context = this.context();
    if (content) {
      // Spread the context if it's an object to make properties directly accessible
      const templateContext = context && typeof context === 'object'
        ? { $implicit: context, ...context as Record<string, unknown> }
        : { $implicit: context };
      return new TemplatePortal(content, this.vf, templateContext);
    }
    return;
  });

  classes = computed(() => {
    const classes = [] as string[];

    classes.push(
      'tooltip',
      this.position());

    switch (this.bg()) {
      case ('opaque'):
        classes.push('bg-inverse', 'text-inverse-foreground', 'dark:bg-dark-inverse', 'dark:text-dark-inverse-foreground');
        break;
      case ('standard'):
        classes.push('bg-base-white-1000', 'dark:bg-alt-gray-1000');
        break;
    }

    if (this.pushTooltip())
      classes.push('max-w-125');
    return classes;
  });
}
