---
trigger: always_on
---

# Agent Instructions

## Project Overview

This is a full-stack application for crude oil and natural gas trading.

## Technology Stack

| Layer | Technology | Version |
|-------|------------|---------|
| Backend | ASP.NET Core | .NET 10.0 |
| Frontend | Angular | 21.x |
| Database | PostgreSQL | (via EF Core 10.0.1) |
| ORM | Entity Framework Core | 10.0.1 |
| UI Components | Kendo UI for Angular | 21.3.0 |
| CSS Framework | Tailwind CSS | 4.x |
| Testing (Backend) | TUnit | 1.11.x |
| Background Jobs | Hangfire | 1.8.x |

## Project Structure

```
porject repo root
├── Fast.sln                    # Solution file
├── Fast.Shared/                # Shared library (models, DB context, business logic)
│   ├── Database/
│   │   ├── MyDbContext.cs      # EF Core DbContext
│   │   └── Models/             # 242+ database entity models
│   ├── LocalMode/              # In-memory database for local development
│   │   └── SeedData/           # Test data seeders
│   ├── Logic/                  # Shared business logic (valuation, utilities, etc.)
│   └── ProgramShared.cs        # Shared startup configuration
├── Fast.Web/                   # Main web application
│   ├── ClientApp/              # Angular frontend
│   │   ├── src/app/
│   │   │   └── _shared/        # Shared components, services, pipes
│   │   ├── package.json
│   │   └── angular.json
│   ├── Controllers/            # API controllers
│   ├── Logic/                  # repo-specific business logic
│   └── Program.cs              # Application entry point
└── Test/                       # TUnit test project
    ├── GlobalSetup.cs          # Test initialization
    ├── Shared/                 # Shared tests
    └── Private/                # Private/integration tests
```

## Build Instructions

### Full Solution Build

```bash
dotnet build Fast.sln
```

### Angular Frontend Build

```bash
cd Fast.Web/ClientApp
npm run build
```

This command:
1. Installs dependencies (`npm i`)
2. Runs ESLint (`npm run lint`)
3. Builds production bundle to `Fast.Web/wwwroot/`

## Running Tests

### Backend Tests (TUnit)

**Run all tests:**
```bash
dotnet run --project Test/Test.csproj
```

**List available tests:**
```bash
dotnet run --project Test/Test.csproj -- --list-tests
```

**Run a specific test by name:**
```bash
dotnet run --project Test/Test.csproj -- --treenode-filter "/*/*/*/ValTest"
```

The filter path is `/<assembly>/<namespace>/<class>/<test>`. Use `*` for wildcards:
```bash
# Run all tests starting with "Auth"
dotnet run --project Test/Test.csproj -- --treenode-filter "/*/*/*/Auth**"
```

> **Note:** Do NOT use `dotnet test` - TUnit on .NET 10 requires running tests via `dotnet run`.

### Frontend Tests

Angular tests are currently disabled (`skipTests: true` in angular.json). No `.spec.ts` files exist in the project.

## Browser Testing

**Important:** When performing browser testing, assume the Angular application is already being served. Simply open:

```
https://127.0.0.1:{port}
```

Replace `{port}` above with the configured port suffix in `.vscode/settings.json`.
Do not try running on the default port 5000. It will not work.

## Development Mode (LocalMode)

LocalMode enables an in-memory database with seeded test data and disables authentication. This is useful for local development without a PostgreSQL connection.

**To enable LocalMode:**

Edit [Fast.Shared/ProgramShared.cs](Fast.Shared/ProgramShared.cs) and change:
```csharp
public static bool IsLocalModeEnabled { get; private set; } = false;
```
to:
```csharp
public static bool IsLocalModeEnabled { get; private set; } = true;
```

When LocalMode is enabled:
- Uses in-memory database instead of PostgreSQL
- Seeds fake data from `Fast.Shared/LocalMode/SeedData/`
- Authentication is automatically disabled
- Hangfire background jobs are skipped

## Key Configuration Files

| File | Purpose |
|------|---------|
| [Fast.Shared/appsettings.json](Fast.Shared/appsettings.json) | Database connection, Serilog config, Gotenberg URL |
| [.vscode/settings.json](.vscode/settings.json) | Port suffix (`fast.port.suffix`), editor settings |
| [.editorconfig](.editorconfig) | Code style (4 spaces C#, 2 spaces TS/JSON, naming conventions) |
| [Fast.Web/ClientApp/angular.json](Fast.Web/ClientApp/angular.json) | Angular build configuration |
| [Fast.Web/ClientApp/eslint.config.mjs](Fast.Web/ClientApp/eslint.config.mjs) | ESLint rules for TypeScript |

## API Architecture

- **OData Support:** Custom EDM model for queryable endpoints
- **Authentication:** Azure AD / OpenID Connect with cookie-based BFF pattern (when enabled)
- **Authorization:** Role-based with `AspRole` and `AppUser` entities
- **Logging:** Serilog to PostgreSQL `app_log` table (production only)

## Database

- **ORM:** Entity Framework Core 10.0.1 with PostgreSQL provider (Npgsql)
- **Models:** Located in [Fast.Shared/Database/Models/](Fast.Shared/Database/Models/) (242+ entities)
- **Context:** [Fast.Shared/Database/MyDbContext.cs](Fast.Shared/Database/MyDbContext.cs)
- **Connection String:** Configured in `appsettings.json` under `ConnectionStrings:Database`

## Coding Conventions

Follow existing patterns in the codebase. Key conventions from `.editorconfig`:

**C#:**
- 4 spaces indentation
- File-scoped namespaces (`namespace Foo;` not `namespace Foo { }`)
- PascalCase for public members, _camelCase for private fields
- UTF-8 with BOM

**TypeScript/Angular:**
- 2 spaces indentation
- ESLint enforced (`npm run lint` with zero warnings tolerance)
- Kendo UI components for grids, dialogs, inputs
- Services in `_shared/services/`, pipes in `_shared/pipes/`

**General:**
- Line endings: CRLF (`\r\n`)
- Max line length: 160 characters
- Trim trailing whitespace
- Insert final newline

## Key Business Logic Locations

| Domain | Location |
|--------|----------|
| Valuation/Pricing | [Fast.Shared/Logic/Valuation/](Fast.Shared/Logic/Valuation/) |

## External Integrations

- **Business Central (ERP):** API integration via [Fast.Web/Logic/BcApiHelper.cs](Fast.Web/Logic/BcApiHelper.cs)
- **SharePoint:** File storage for documents
- **Gotenberg:** PDF generation service (URL in appsettings.json)
- **Azure AD:** Authentication provider (when auth enabled)

## Useful Commands

```bash
# Start Angular dev server (reads port from .vscode/settings.json)
cd Fast.Web/ClientApp && npm start

# Run .NET backend
dotnet run --project Fast.Web/Fast.Web.csproj

# Check linting
cd Fast.Web/ClientApp && npm run lint
```