
import { Component, computed, inject, input, ViewChild } from "@angular/core";
import { ColumnComponent, FieldDataType, FilterService, KENDO_GRID } from "@progress/kendo-angular-grid";
import { FastGridDateFilterCell } from "./fast-grid-date-filter-cell.component";
import { MessageService } from "../services/message.service";
import { DateType } from "./fast-datepicker.component";
import { handleError } from "../utils/util";

export type GridColType = 'text' | 'num-c#' | 'num-c0' | 'num-c2' | 'num-c5' | 'num-d#' | 'num-d0' | 'num-d2' | 'num-d5' | 'num-p#' | DateType;

export interface TypeFormat {
  type: GridColType;
  format?: string;
}

@Component({
  selector: 'fast-grid-col',
  imports: [KENDO_GRID, FastGridDateFilterCell],
  template: `
    <kendo-grid-column #kendoCol [field]="field()" [title]="computedTitle()" [width]="width()" [filter]="computedFilterType()" [format]="computedFormat()">
      @if (type().includes('date-')) {
        <ng-template kendoGridFilterCellTemplate let-filter="filter" let-column="column">
          <fast-grid-date-filter-cell #dateFilterCell [field]="field()" [type]="computedDateType()" />
        </ng-template>
      }
    </kendo-grid-column>
  `,
})
export class FastGridColComponent {
  @ViewChild('kendoCol') kendoCol: ColumnComponent;
  @ViewChild(FastGridDateFilterCell) dateFilterCell: FastGridDateFilterCell;
  messageService = inject(MessageService);

  field = input.required<string>();
  title = input<string>();
  width = input<number>();
  type = input.required<GridColType>();

  formats: Record<GridColType, TypeFormat> = {
    'text': { type: 'text' },
    'num-c#': { type: 'num-c#', format: '{0:$#,##.#####;($#,##.#####)}' },
    'num-c0': { type: 'num-c0', format: '{0:$#,##;($#,##)}' },
    'num-c2': { type: 'num-c2', format: '{0:$#,##.00;($#,##.00)}' },
    'num-c5': { type: 'num-c5', format: '{0:$#,##.00000;($#,##.00000)}' },
    'num-d#': { type: 'num-d#', format: '{0:#,##.#####;-#,##.#####}' },
    'num-d0': { type: 'num-d0', format: '{0:#,##;-#,##}' },
    'num-d2': { type: 'num-d2', format: '{0:#,##.00;-#,##.00}' },
    'num-d5': { type: 'num-d5', format: '{0:#,##.00000;-#,##.00000}' },
    'num-p#': { type: 'num-p#', format: '#,.#####\\%;(#,.#####)\\%}' },
    'date-short': { type: 'date-short', format: '{0:M/d/yyyy}' },
    'date-month': { type: 'date-month', format: '{0:MMM yyyy}' }
  }

  computedDateType = computed(() => {
    const dateType = this.type() as DateType;
    if (dateType)
      return dateType
    else
      this.throwTypeError();
  });

  computedTitle = computed(() => {
    return this.title() ?? this.field();
  });

  computedFilterType = computed((): FieldDataType => {
    if (this.type() === 'text')
      return 'text';
    else if (this.type().includes('date-'))
      return 'date';
    else if (this.type().includes('num-'))
      return 'numeric'
    else
      this.throwTypeError();
  });

  computedFormat = computed(() => {
    if (this.type() in this.formats) {
      return this.formats[this.type()].format;
    }
    else {
      this.throwTypeError();
    }
  });

  throwTypeError() {
    handleError(`Type {${this.type()}} not implemented in fast-grid-col`, this.messageService);
  }

  public getKendoCol() {
    return this.kendoCol;
  }

  public setFilterService(filterService: FilterService) {
    this.dateFilterCell.setFilterService(filterService);
  }
}
