import { CommonModule } from "@angular/common";
import { ChangeDetectionStrategy, ChangeDetectorRef, Component, computed, HostBinding, inject, input, OnChanges, signal } from "@angular/core";
import * as util from '../utils/util';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from "../../app.config";
import { twMerge } from 'tailwind-merge'

type ButtonType = 'basic' | 'delete' | 'export' | 'primary';

@Component({
  selector: 'fast-button',
  changeDetection: ChangeDetectionStrategy.OnPush,
  standalone: true,
  imports: [CommonModule, FAST_KENDO_COMMON, FAST_PAGE_COMMON],
  template: `
  @if(type() !== 'export'){
  <button class="
    " [ngClass]="conditionalClasses()" type="button" [disabled]="isDisabled()"
      [value]="value()">
      <ng-content></ng-content>
    </button>
    }
  @if(type() === 'export'){
  <button [ngClass]="conditionalClasses()" type="button" [disabled]="isDisabled()"
      [value]="value()" [hidden]="exporting()">
      <kendo-svg-icon class="mb-0.5 min-w-auto" [icon]="icons.fileExcelIcon"></kendo-svg-icon>Export
    </button>
    <button [ngClass]="conditionalClasses()" type="button" [disabled]="isDisabled()"
      [value]="value()" [disabled]="true" [hidden]="!exporting()">
      Exporting...
    </button>
    }
  `
})
export class FastButtonComponent implements OnChanges {
  cdr = inject(ChangeDetectorRef);
  util = util;
  icons = util.icons;

  type = input<ButtonType>('basic');
  disabled = input<boolean>(false);
  value = input<string>('');
  exporting = input<boolean>(false);
  class = input<string>('');
  defaultClasses = 'flex w-28 h-7.5';

  // Internal signal to track form control disabled state
  private formControlDisabled = signal<boolean>(false);

  // Computed signal that combines input disabled state and form control disabled state
  isDisabled = computed(() => {
    return this.disabled() || this.formControlDisabled()
  });

  @HostBinding('class') hostClass: string = this.defaultClasses;

  ngOnChanges(): void {
    this.hostClass = twMerge(this.defaultClasses, this.class());
  }

  setDisabledState(isDisabled: boolean) {
    this.formControlDisabled.set(isDisabled);
    this.cdr.markForCheck();
  }

  conditionalClasses = computed(() => {
    const classes = [] as string[];

    classes.push(...this.getCommonClasses());
    classes.push(...this.getLightBaseClasses(this.type()));
    classes.push(...this.getDarkBaseClasses(this.type()));
    if (!this.isDisabled()) {
      classes.push(...this.getLightHoverClasses());
      classes.push(...this.getDarkHoverClasses());
      classes.push(...this.getLightActiveClasses(this.type()));
      classes.push(...this.getDarkActiveClasses(this.type()));
    }

    const conditionalClasses = this.getConditionalClassesFromArrays(classes);
    return conditionalClasses;
  });

  getConditionalClassesFromArrays(classArray: string[]): { [key: string]: boolean } {
    const classes: { [key: string]: boolean } = {};
    classArray.forEach(className => {
      classes[className] = true;
    });
    return classes;
  }

  getCommonClasses() {
    return [
      "flex",
      "flex-grow",
      "text-sm",
      "leading-0",
      "border-solid",
      "border-1",
      "rounded-md",
      "pl-2",
      "pr-2",
      "pt-1",
      "pb-1",
      "gap-1",
      "items-center",
      "justify-center",
      "cursor-pointer",
    ];
  }

  getLightBaseClasses(type: ButtonType) {
    const lightBase = ["text-base-black-1000", "border-base-gray-1000"]
    if (type == 'basic' || type == 'export')
      lightBase.push("bg-gradient-to-b", "from-base-white-250", "via-base-blue-250", "to-base-blue-250");

    if (type == 'primary')
      lightBase.push("bg-gradient-to-b", "from-base-white-250", "via-base-blue-250", "to-base-white-250");

    if (type == 'delete')
      lightBase.push("bg-gradient-to-b", "from-base-white-250", "via-base-red-250", "to-base-red-500");

    return lightBase;
  }

  getDarkBaseClasses(type: ButtonType) {
    const darkBase = ["dark:text-base-white-500", "dark:border-alt-white-1000"]
    if (type == 'basic' || type == 'export')
      darkBase.push("dark:bg-gradient-to-b", "dark:from-alt-blue-250", "dark:via-alt-gray-1000", "dark:to-alt-blue-750");

    if (type == 'primary')
      darkBase.push("dark:bg-gradient-to-b", "dark:from-base-blue-250", "dark:via-alt-gray-1000", "dark:to-alt-blue-750");

    if (type == 'delete')
      darkBase.push("dark:bg-gradient-to-b", "dark:from-alt-red-500", "dark:via-base-black-1000", "dark:to-base-black-1000");

    return darkBase;
  }

  getLightHoverClasses() {
    return ["hover:text-alt-blue-250", "hover:border-alt-blue-250", "hover:brightness-90",
    ];
  }

  getDarkHoverClasses() {
    return ["dark:hover:brightness-150", "dark:hover:text-base-blue-250", "dark:hover:border-base-blue-250",
    ];
  }

  getLightActiveClasses(type: ButtonType) {
    if (type == 'basic' || type == 'export')
      return ["active:bg-gradient-to-b", "active:from-base-blue-250", "active:via-base-blue-250", "active:to-base-white-250", "active:hover:sepia"];

    if (type == 'primary')
      return ["active:bg-gradient-to-b", "active:from-base-blue-250", "active:via-base-white-250", "active:to-base-blue-250", "active:hover:sepia"];

    if (type == 'delete')
      return ["active:bg-gradient-to-b", "active:from-base-red-500", "active:via-base-red-250", "active:to-base-white-250", "active:hover:sepia"];
  }

  getDarkActiveClasses(type: ButtonType) {
    if (type == 'basic' || type == 'export')
      return ["dark:active:bg-gradient-to-b", "dark:active:from-alt-blue-750", "dark:active:via-alt-gray-1000", "dark:active:to-alt-blue-250", "active:hover:sepia"];

    if (type == 'primary')
      return ["dark:active:bg-gradient-to-b", "dark:active:from-alt-blue-750", "dark:active:via-alt-gray-1000", "dark:active:to-base-blue-250", "active:hover:sepia"];

    if (type == 'delete')
      return ["dark:active:bg-gradient-to-b", "dark:active:from-base-black-1000", "dark:active:via-base-black-1000", "dark:active:to-alt-red-500", "active:hover:sepia"];
  }
}

