import { inject, Injectable } from '@angular/core';
import { HttpClient, HttpHeaders } from '@angular/common/http';
import { OdataGridHelper } from '../_shared/utils/odata-grid-helper';
import { IdName, SaveType } from '../_shared/utils/util';
import { Observable } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { GridDataResult } from '@progress/kendo-angular-grid';

export interface PriceIndexItem {
  id: number,
  name: string,
  indexType: string,
  commodity: string,
  hybrid: string,
  classification: string,
  publication: string,
  forwardCurve: string
}

export interface IndexAlias {
  id: number
  aliasName: string,
  aliasTypeId: number,
  label: string
}

export interface HybridIndexFunction {
  definition: string,
  textFiller: string
}

export interface IndexSuffix {
  definition: string,
  textFiller: string
}

export interface HybridVariable {
  definition: string,
  textFiller: string
}

export interface PriceIndexDetailItem {
  id: number,
  name: string,
  aliases: IndexAlias[],
  indexType: number,
  commodityId: number,
  classificationTypeId: number,
  publicationId: number,
  description: string,
  notes: string,
  hybridFormula: string,
  unitPriceId: number,
  forwardCurveId: number,
  formulaFunction: HybridIndexFunction,
  indexSuffix: IndexSuffix,
  hybridVariable: HybridVariable,
  formulaIndex: IdName
}

export interface RequiredData {
  commodities: IdName[],
  classifications: IdName[],
  publications: IdName[],
  priceUnits: IdName[],
  indexIdNames: IdName[],
  aliasTypes: IdName[],
  indexTypes: IdName[],
  functions: HybridIndexFunction[],
  variables: HybridVariable[],
  suffixes: IndexSuffix[],
  hasModifyPermission: boolean;
}

export interface HybridFormulaResult {
  hybridIndexId: number,
  isValid: boolean,
  message: string
}

@Injectable({
  providedIn: 'root'
})
export class PriceIndexService extends OdataGridHelper {
  private baseUrl = `${window.location.origin}/api/PriceIndex`;
  http = inject(HttpClient);

  requiredData$ = this.http.get<RequiredData>(`${this.baseUrl}/getRequiredData`)

  getPriceIndexes(state: State): Observable<GridDataResult> {
    return this.fetch(`GetPriceIndexItems`, state, null, false)
  }

  getPriceIndexDetail(id: number): Observable<PriceIndexDetailItem> {
    const url = `${this.baseUrl}/GetPriceIndexDetail/${id}`;
    return this.http.get<PriceIndexDetailItem>(url);
  }

  savePriceIndex(item: PriceIndexDetailItem, saveType: SaveType): Observable<HybridFormulaResult> {
    const url = `${this.baseUrl}/SavePriceIndex?saveType=${saveType}`;
    return this.http.put<HybridFormulaResult>(url, item);
  }

  deletePriceIndex(id: number): Observable<unknown> {
    const url = `${this.baseUrl}/DeletePriceIndex/${id}`;
    return this.http.delete<unknown>(url);
  }

  exportPriceIndexItems(state: State, fileNameOriginal: string) {
    return this.fetchExport(`GetPriceIndexItems`, state, null, fileNameOriginal);
  }

  testHybridFormula(formula: string) {
    const headers = new HttpHeaders({ 'Content-Type': 'application/json' });
    const json = JSON.stringify(formula);

    const url = `${this.baseUrl}/TestHybridFormula`;
    return this.http.put<HybridFormulaResult>(url, json, { headers });
  }
}
