namespace Fast.Logic.FileService;

public class FileServiceConfig(string contentRootPath = "")
{
    /// The base content root path for local file operations.
    public string ContentRootPath { get; set; } = contentRootPath;

    /// Mapping of logical folder names to their actual local filesystem paths.
    /// Only used when UseLocalFiles is true.
    public Dictionary<string, string> LocalFolderMappings { get; set; } = new();
    public SharePointConfiguration SharePoint { get; set; } = new();
    public PerformanceConfiguration Performance { get; set; } = new();
}

public class SharePointConfiguration
{
    public string SiteName { get; set; } = "superiornatgas.sharepoint.com:/sites/T-Rex";
    public string DriveName { get; set; } = "documents";
    public string? TenantId { get; set; }
    public string? ClientId { get; set; }
    public string? ClientSecret { get; set; }

    /// Maximum file size (in bytes) for simple upload. Files larger than this use upload sessions.
    /// Default is 4MB (Graph API limit for simple upload).
    public long MaxSimpleUploadSize { get; set; } = 4 * 1024 * 1024;

    /// Chunk size for large file uploads using upload sessions.
    /// Default is 5MB (must be a multiple of 320KB).
    public int UploadChunkSize { get; set; } = 5 * 1024 * 1024;
}

public class PerformanceConfiguration
{
    public int MaxDegreeOfParallelism { get; set; } = 4;

    /// Maximum number of requests in a Graph API batch (max 20 per Graph API spec).
    public int MaxBatchSize { get; set; } = 20;

    /// Maximum number of retry attempts for transient failures.
    public int MaxRetryAttempts { get; set; } = 3;

    /// Initial delay for exponential backoff retry (in milliseconds).
    public int InitialRetryDelayMs { get; set; } = 1000;

    /// Maximum delay for exponential backoff retry (in milliseconds).
    public int MaxRetryDelayMs { get; set; } = 30000;

    /// Time-to-live for cached folder existence checks and DriveItem IDs (in seconds).
    public int CacheTtlSeconds { get; set; } = 300;

    /// Circuit breaker failure threshold - number of failures before circuit opens.
    public int CircuitBreakerFailureThreshold { get; set; } = 5;

    /// Circuit breaker reset timeout (in seconds).
    public int CircuitBreakerResetTimeoutSeconds { get; set; } = 60;
}