using Newtonsoft.Json;

namespace Fast.Web.Models;

public class DataMineListResponse
{
    [JsonProperty("data")]
    public List<DataMineListDay> Data { get; set; } = new();

    [JsonProperty("paging")]
    public DataMinePagingInfo Paging { get; set; } = new();

    [JsonIgnore]
    public List<DataMineFileDetail> Files
    {
        get
        {
            // Flatten DataMineListDay/DataMineListFile API response into DataMineFileDetail
            var list = new List<DataMineFileDetail>();
            foreach (var day in Data)
            {
                DateTime periodDt = DateTime.MinValue;
                if (!string.IsNullOrWhiteSpace(day.PeriodDate) && day.PeriodDate.Length == 8)
                {
                    // yyyyMMdd -> DateTime (UTC midnight)
                    var yyyy = int.Parse(day.PeriodDate[..4]);
                    var mm = int.Parse(day.PeriodDate[4..6]);
                    var dd = int.Parse(day.PeriodDate[6..8]);
                    periodDt = new DateTime(yyyy, mm, dd, 0, 0, 0, DateTimeKind.Utc);
                }

                foreach (var f in day.Files)
                {
                    list.Add(new DataMineFileDetail
                    {
                        Dataset = day.DatasetCode ?? "",
                        Yyyymmdd = day.PeriodDate ?? "",
                        Url = f.ApiDownloadLink ?? "",
                        Fid = f.FileId ?? "",
                        Size = (int)f.Size,
                        Period = periodDt,
                        FileName = f.FileName ?? ""
                    });
                }
            }
            return list;
        }
    }
}

public class DataMineListDay
{
    [JsonProperty("category_code")]
    public string CategoryCode { get; set; } = "";

    [JsonProperty("dataset_code")]
    public string DatasetCode { get; set; } = "";

    [JsonProperty("dataset_name")]
    public string DatasetName { get; set; } = "";

    // exchange_code is often null
    [JsonProperty("exchange_code")]
    public string? ExchangeCode { get; set; }

    [JsonProperty("files")]
    public List<DataMineListFile> Files { get; set; } = new();

    [JsonProperty("num_of_files")]
    public int NumOfFiles { get; set; }

    // e.g. "20250610"
    [JsonProperty("period_date")]
    public string PeriodDate { get; set; } = "";

    [JsonProperty("product_code")]
    public string? ProductCode { get; set; }

    [JsonProperty("total_size")]
    public long TotalSize { get; set; }
}

public class DataMineListFile
{
    [JsonProperty("api_download_link")]
    public string ApiDownloadLink { get; set; } = "";

    [JsonProperty("file_id")]
    public string FileId { get; set; } = "";

    [JsonProperty("file_name")]
    public string FileName { get; set; } = "";

    [JsonProperty("sftp_path")]
    public string SftpPath { get; set; } = "";

    [JsonProperty("size")]
    public long Size { get; set; }
}

public class DataMinePagingInfo
{
    [JsonProperty("previous")]
    public string Previous { get; set; } = "";

    [JsonProperty("next")]
    public string Next { get; set; } = "";
}

// Flattened convenience model used by the processing code
public class DataMineFileDetail
{
    public string Dataset { get; set; } = "";
    public string Yyyymmdd { get; set; } = "";
    public string Url { get; set; } = "";
    public string Fid { get; set; } = "";
    public int Size { get; set; }
    public DateTime Period { get; set; }
    public string FileName { get; set; } = "";
}
