import { ChangeDetectionStrategy, Component, computed, input } from '@angular/core';
import { FAST_KENDO_COMMON } from '../../app.config';
import { ControlValueAccessor, NG_VALUE_ACCESSOR } from '@angular/forms';

type ToggleType = 'basic' | 'allow-deny' | 'show-hide' | 'combine-separate';

@Component({
  selector: 'fast-toggle',
  imports: [FAST_KENDO_COMMON],
  providers: [{
    provide: NG_VALUE_ACCESSOR,
    multi: true,
    useExisting: FastToggleComponent
  }],
  template: `
  <kendo-switch
  class = "text-xs flex flex-grow whitespace-nowrap transition-none
  text-base-black-1000
  dark:text-base-white-500
  [&_.k-switch-track]:border-2
  [&_.k-switch-track]:border-base-gray-500
  dark:[&_.k-switch-track]:border-alt-blue-250
  focus:ring-4
  focus:ring-offset-2
  focus:outline-0
  [&_.k-switch_span.k-switch-track]:content-start
  [&_.k-switch_span.k-switch-track]:items-center
  [&_.k-switch-label-on]:left-2
  [&_.k-switch-label-on]:inline
  [&_.k-switch-label-off]:right-2
  [&_.k-switch-label-off]:inline
  [&_.k-switch-label-off]:items-end
  [&_.k-switch-thumb]:bg-white"
  [onLabel]="onLabel()"
  [offLabel]="offLabel()"
  [checked]="isTrue"
  [disabled]="disabled()"
  [class]="conditionalClasses()"
  (valueChange)="onValueChange($event)">
  </kendo-switch>
  `,
  styles: `:host { @apply flex; }`,
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class FastToggleComponent implements ControlValueAccessor {
  type = input<ToggleType>('basic');
  onLabel = input<string>();
  offLabel = input<string>();
  isTrue: boolean = false;

  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  onChange = (isTrue: boolean) => { };
  onTouched = () => { };
  disabled = input<boolean>(false);

  writeValue(isTrue: boolean): void {
    this.isTrue = isTrue;
  }

  registerOnChange(fn: (value: boolean) => void) {
    this.onChange = fn;
  }

  registerOnTouched(fn: () => void) {
    this.onTouched = fn;
  }

  onValueChange(isTrue: boolean) {
    this.isTrue = isTrue;
    this.onChange(isTrue);
    this.onTouched(); // Mark as touched when value changes
  }

  conditionalClasses = computed(() => {
    const classes = [] as string[];
    classes.push(...this.getLightBaseClasses(this.type()));
    classes.push(...this.getDarkBaseClasses(this.type()));

    const conditionalClasses = this.getConditionalClassesFromArrays(classes);
    return conditionalClasses;
  });


  getConditionalClassesFromArrays(classArray: string[]): { [key: string]: boolean } {
    const classes: { [key: string]: boolean } = {};
    classArray.forEach(className => {
      classes[className] = true;
    });
    return classes;
  }

  getLightBaseClasses(type: ToggleType) {
    if (type === 'allow-deny') {
      return ["[&.k-switch-off_.k-switch-track]:bg-gradient-to-b",
        "[&.k-switch-off_.k-switch-track]:from-base-white-250",
        "[&.k-switch-off_.k-switch-track]:via-red-400",
        "[&.k-switch-off_.k-switch-track]:to-red-300",
        "[&.k-switch-on_.k-switch-track]:bg-gradient-to-b",
        "[&.k-switch-on_.k-switch-track]:from-base-white-250",
        "[&.k-switch-on_.k-switch-track]:via-green-400",
        "[&.k-switch-on_.k-switch-track]:to-green-300",
        "ring-base-gray-750/50"];
    }

    return ["[&.k-switch-off_.k-switch-track]:bg-gradient-to-b",
      "[&.k-switch-off_.k-switch-track]:from-base-white-1000",
      "[&.k-switch-off_.k-switch-track]:via-base-white-1000",
      "[&.k-switch-off_.k-switch-track]:to-to-base-white-250",
      "[&.k-switch-on_.k-switch-track]:bg-gradient-to-b",
      "[&.k-switch-on_.k-switch-track]:from-base-blue-1000",
      "[&.k-switch-on_.k-switch-track]:via-base-blue-1000",
      "[&.k-switch-on_.k-switch-track]:to-base-blue-250",
      "ring-base-gray-750/50"
    ];
  }

  getDarkBaseClasses(type: ToggleType) {
    if (type === 'allow-deny') {
      return ["dark:[&.k-switch-off_.k-switch-track]:bg-gradient-to-b",
        "dark:[&.k-switch-off_.k-switch-track]:from-alt-gray-250",
        "dark:[&.k-switch-off_.k-switch-track]:via-red-700",
        "dark:[&.k-switch-off_.k-switch-track]:to-red-600",
        "dark:[&.k-switch-on_.k-switch-track]:bg-gradient-to-b",
        "dark:[&.k-switch-on_.k-switch-track]:from-alt-gray-250",
        "dark:[&.k-switch-on_.k-switch-track]:via-green-700",
        "dark:[&.k-switch-on_.k-switch-track]:to-green-600",
        "dark:ring-base-gray-750/50"];
    }

    return ["dark:[&.k-switch-off_.k-switch-track]:bg-gradient-to-b",
      "dark:[&.k-switch-off_.k-switch-track]:from-alt-gray-1000",
      "dark:[&.k-switch-off_.k-switch-track]:via-alt-gray-1000",
      "dark:[&.k-switch-off_.k-switch-track]:to-alt-gray-500",
      "dark:[&.k-switch-on_.k-switch-track]:bg-gradient-to-b",
      "dark:[&.k-switch-on_.k-switch-track]:from-base-blurple-500",
      "dark:[&.k-switch-on_.k-switch-track]:via-base-blurple-500",
      "dark:[&.k-switch-on_.k-switch-track]:to-alt-blue-250",
      "dark:ring-base-gray-750/50"];
  }
}
