import { ChangeDetectionStrategy, ChangeDetectorRef, Component, computed, inject, input, model, output } from '@angular/core';
import { FAST_KENDO_COMMON } from '../../app.config';
import { ControlValueAccessor, NG_VALUE_ACCESSOR } from '@angular/forms';
import { CalendarView, FormatSettings } from '@progress/kendo-angular-dateinputs';

@Component({
  selector: 'fast-datepicker',
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [FAST_KENDO_COMMON],
  providers: [{
    provide: NG_VALUE_ACCESSOR,
    multi: true,
    useExisting: FastDatePickerComponent
  }],
  template: `
  <kendo-datepicker
    class="flex h-7 border-2 border-solid rounded-sm text-sm
    [&_.k-input-button]:border-none [&_.k-input-button]:w-6
    [&_.k-input-button]:bg-transparent
    [&_input]:pl-1"
    [min]="min()"
    [max]="max()"
    [class]="conditionalClasses()"
    [format]="format()"
    [tabindex]="tabindex()"
    [topView]="topView()"
    [bottomView]="bottomView()"
    [value]="this.value()"
    (valueChange)="onValueChange($event)"
    [disabled]="disabled()"
  />
  `
})
export class FastDatePickerComponent implements ControlValueAccessor {
  cdr = inject(ChangeDetectorRef);

  min = input<Date | null>(null);
  max = input<Date | null>(null);
  tabindex = input<number>(0);
  topView = input<CalendarView>();
  bottomView = input<CalendarView>();
  valueChange = output<Date | null>();
  format = input<string | FormatSettings>('M/d/yyyy');
  value = model<Date | null>(null);
  disabled = model<boolean>(false);

  onChange: (value: Date | null) => void;
  onTouched = () => { };

  writeValue(value: Date | null) {
    this.value.set(value);
    this.cdr.markForCheck();
  }

  registerOnChange(fn: () => void) {
    this.onChange = fn;
  }

  registerOnTouched(fn: () => void) {
    this.onTouched = fn;
  }

  setDisabledState(isDisabled: boolean) {
    this.disabled.set(isDisabled);
  }

  onValueChange(newValue: Date | null) {
    this.value.set(newValue);
    this.onChange(newValue);
  }

  conditionalClasses = computed(() => {
    const classes = [] as string[];

    classes.push(...this.getLightBaseClasses());
    classes.push(...this.getDarkBaseClasses());
    classes.push(...this.getLightActiveClasses());
    classes.push(...this.getDarkActiveClasses());

    const conditionalClasses = this.getConditionalClassesFromArrays(classes);
    return conditionalClasses;
  });

  getConditionalClassesFromArrays(classArray: string[]): { [key: string]: boolean } {
    const classes: { [key: string]: boolean } = {};
    classArray.forEach(className => {
      classes[className] = true;
    });
    return classes;
  }

  getLightBaseClasses() {
    return [
      "[&_.k-input-button]:hover:bg-base-white-1000",
      "[&_.k-svg-i-calendar]:text-base-black-1000"];
  }

  getDarkBaseClasses() {
    return [
      "dark:[&_.k-input-button]:hover:bg-alt-blue-750",
      "dark:[&_.k-svg-i-calendar]:text-base-white-500"];
  }

  getLightActiveClasses() {
    return [
      "[&_.k-input-button]:active:bg-base-gray-1000",];
  }

  getDarkActiveClasses() {
    return [
      "dark:[&_.k-input-button]:active:bg-base-blue-1000",];
  }
}
