import { ChangeDetectionStrategy, ChangeDetectorRef, Component, computed, inject, input, model, output } from '@angular/core';
import { FAST_KENDO_COMMON } from '../../app.config';
import { ControlValueAccessor, NG_VALUE_ACCESSOR } from '@angular/forms';
import { DropDownFilterSettings } from '@progress/kendo-angular-dropdowns';

@Component({
  selector: 'fast-combobox',
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [FAST_KENDO_COMMON],
  providers: [{
    provide: NG_VALUE_ACCESSOR,
    multi: true,
    useExisting: FastComboboxComponent
  }],
  template: `
  <kendo-combobox
    class=" h-7 items-center [&_button]:border-none [&_button]:w-6.5 [&_button]:items-center"
    [data]="data()"
    [textField]="textField()"
    [valueField]="valueField()"
    [valuePrimitive]="valuePrimitive()"
    [kendoDropDownFilter]="filterSettings"
    [value]="value()"
    [placeholder]="placeholder()"
    [clearButton]="clearButton()"
    [virtual]="{ itemHeight: 28 }"
    (valueChange)="onValueChange($event)"
    [class]="conditionalClasses()"
    [disabled]="disabled()"
  />
  `
})
export class FastComboboxComponent<T extends object> implements ControlValueAccessor {
  cdr = inject(ChangeDetectorRef);

  data = input.required<T[]>();
  textField = input<string>("");
  valueField = input<string>("");
  valuePrimitive = input<boolean>(true);
  clearButton = input<boolean>(false);
  placeholder = input<string>("");
  valueChange = output<T | null>();
  value = model<T | null>(null);
  disabled = model<boolean>(false);
  clearOnValueChange = input<boolean>(false);

  onChange: (value: T | null) => void;
  onTouched = () => { };

  writeValue(value: T | null) {
    this.value.set(value);
    this.cdr.markForCheck();
  }

  registerOnChange(fn: () => void) {
    this.onChange = fn;
  }

  registerOnTouched(fn: () => void) {
    this.onTouched = fn;
  }

  setDisabledState(isDisabled: boolean) {
    this.disabled.set(isDisabled);
  }

  onValueChange(newValue: T | null) {
    newValue = this.clearOnValueChange() ? null : newValue;
    this.value.set(newValue);
    this.onChange(newValue);
  }

  conditionalClasses = computed(() => {
    const classes = [] as string[];

    classes.push(...this.getLightBaseClasses());
    classes.push(...this.getDarkBaseClasses());
    classes.push(...this.getLightActiveClasses());
    classes.push(...this.getDarkActiveClasses());

    const conditionalClasses = this.getConditionalClassesFromArrays(classes);
    return conditionalClasses;
  });


  getConditionalClassesFromArrays(classArray: string[]): { [key: string]: boolean } {
    const classes: { [key: string]: boolean } = {};
    classArray.forEach(className => {
      classes[className] = true;
    });
    return classes;
  }

  getLightBaseClasses() {
    return [
      "text-base-black-1000",
      "[&_button]:text-base-black-1000",
      "[&_button]:bg-base-white-250",
      "[&_button]:hover:bg-base-white-750"];
  }

  getDarkBaseClasses() {
    return [
      "dark:text-base-white-500",
      "dark:[&_button]:text-base-white-500",
      "dark:[&_button]:bg-alt-gray-500",
      "dark:[&_button]:hover:bg-alt-blue-750"];
  }

  getLightActiveClasses() {
    return [
      "[&_button]:active:bg-base-white-1000"];
  }

  getDarkActiveClasses() {
    return [
      "dark:[&_button]:active:bg-alt-blue-250"];
  }

  public filterSettings: DropDownFilterSettings = {
    caseSensitive: false,
    operator: "contains",
  };
}
