using Fast.Logic.NaturalGas;
using Fast.Shared;
using Fast.Shared.Logic.FileService;
using Fast.Web.Logic.NaturalGas;
using Fast.Web.Models;

namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class InvoiceNatGasDetailController : ODataController
{
    private readonly MyDbContext db;
    private readonly AuthorizationHelper authHelper;
    private readonly IFileService fileService;
    private const string securityActionApprove = "Invoice Natural Gas Approval";

    public InvoiceNatGasDetailController(MyDbContext db, IWebHostEnvironment env)
    {
        this.db = db;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
        var config = new FileServiceConfig(env.ContentRootPath);
        fileService = new FileService(config);
    }

    [Permission("Invoice Natural Gas", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData(int counterpartyId, int internalEntityId, DateOnly month)
    {
        var firstOfMonth = Util.Date.FirstDayOfMonth(month);
        var lastOfMonth = Util.Date.LastDayOfMonth(month);
        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Invoice Natural Gas", PermissionType.Modify);
        var hasApprovalPermission = await authHelper.IsAuthorizedAsync(User, securityActionApprove, PermissionType.Standard);
        var pipelines = (await DataHelper.GetPipelinesAsync(false)).Where(x => x.IsGasPipe == true).ToList();
        var meters = (await DataHelper.GetMetersByProductAsync(Enums.ProductCategory.NaturalGasAndLng)).Select(x => x).ToList();

        var dealNums = await (
            from d in db.Deals
            where d.TransactionTypeId == (int)Enums.TransactionType.PhysicalGas
                && d.StartDate <= lastOfMonth
                && d.EndDate >= firstOfMonth
                && d.CounterpartyId == counterpartyId
                && d.InternalEntityId == internalEntityId
                && d.NettingContractNumber != null
            select new
            {
                d.Id,
                Name = d.TicketNum,
                Trader = d.Trader != null ? d.Trader.Initials : null,
            }
        ).AsNoTracking().ToListAsync();

        var states = await DataHelper.GetTerritoriesAsync();
        var countries = db.Countries
            .OrderBy(x => x.Name)
            .Select(x => new IdName(x.Id, x.Name))
            .ToList();


        var result = new { hasModifyPermission, hasApprovalPermission, states, countries, pipelines, dealNums, meters };

        return Ok(result);
    }

    /// <summary>
    /// not currently being used
    /// </summary>
    public class SaveInvoiceResult
    {
        public GasInvoice? Invoice;
        public string? Error;
    }

    [Permission("Invoice Natural Gas", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetInvoice(int? invoiceId, int counterpartyId, int internalEntityId, DateOnly month)
    {
        month = Util.Date.FirstDayOfMonth(month);

        GasInvoice invoice = await GetInvoiceHelper.GetInvoiceAsync(invoiceId, counterpartyId, internalEntityId, month);
        FillTradersForLines(invoice.Lines);

        return Ok(invoice);
    }

    [Permission("Invoice Natural Gas", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetMainInvoice(int counterpartyId, int internalEntityId, DateOnly month)
    {
        month = Util.Date.FirstDayOfMonth(month);

        var mainInvoice = await GetInvoiceHelper.GetNewMainInvoiceAsync(counterpartyId, internalEntityId, month);

        return Ok(mainInvoice);
    }

    public enum InvoiceSaveType
    {
        New = 1,
        Draft = 2,
    }

    [Permission("Invoice Natural Gas", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> SaveInvoice(InvoiceSaveType saveType, GasInvoice invoice)
    {
        var saveHelper = new SaveInvoiceHelper(authHelper, User);
        var invoiceNum = await saveHelper.SaveInvoice(saveType, invoice);

        try
        {
            var invoiceDoc = new InvoiceDoc(fileService, "Invoices", "InvoiceTemplates", "Signatures", db);
            var (fileNameOriginal, fileNameOnDisk) = await invoiceDoc.GenerateAsync(invoiceNum);
            var dbInvoice = await db.InvoiceGas.Where(x => x.InvoiceNum == invoiceNum).FirstAsync();
            dbInvoice.FileNameOriginal = fileNameOriginal;
            dbInvoice.FileNameOnDisk = fileNameOnDisk;
            await db.SaveChangesAsync();
        }
        catch (Exception ex)
        {
            var errorDetails = ex.Message;
            var newMessage = $"""
                Error generating document for invoice {invoiceNum}.
                The draft info was saved, but the document was not generated.
                This may indicate a problem with the invoice template.

                {errorDetails}
            """;
            throw new Exception(newMessage);
        }

        return Ok(null);
    }

    private void FillTradersForLines(List<InvoiceGasLineItem> invoiceLines)
    {
        var linesWithDeals = invoiceLines
        .Where(x => x.DealId != null)
        .Select(x => x.DealId)
        .ToList();

        var matchedTraders = db.Deals
        .Where(x => linesWithDeals.Contains(x.Id))
        .Select(x => new { x.Id, Trader = x.Trader != null ? x.Trader.Initials : null })
        .ToList();

        foreach (var line in invoiceLines)
        {
            if (line.DealId.HasValue)
            {
                line.Trader = matchedTraders
                .Where(x => x.Id == line.DealId)
                .Select(x => x.Trader)
                .FirstOrDefault();
            }
            else
            {
                line.Trader = null;
            }
        }
    }

    [Permission("Invoice Natural Gas", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetTaxRate(int meterId, int counterpartyId, DateOnly month)
    {
        var prodMonthEnd = Util.Date.LastDayOfMonth(month);
        var meterState = await db.Meters
            .Where(x => x.Id == meterId)
            .Select(x => x.StateId)
            .FirstOrDefaultAsync();

        var hasValidExemption = await (
            from q in db.SalesTaxExemptions
            where q.CompanyId == counterpartyId
                && q.EffectiveDate <= prodMonthEnd
                && prodMonthEnd <= q.ExpirationDate
                && q.StateId == meterState
            select q.CompanyId
        ).AnyAsync();

        if (hasValidExemption)
            return Ok(0);

        decimal taxRate = 0;
        var countyForMeter = await db.Meters
            .Where(x => x.Id == meterId)
            .Select(x => x.CountyId)
            .FirstOrDefaultAsync();

        if (countyForMeter != null)
        {
            taxRate = await db.SalesTaxRates
                .Where(x => x.CountyId == countyForMeter)
                .Select(x => x.Rate)
                .FirstOrDefaultAsync();
        }

        return Ok(taxRate);
    }

    [Permission("Invoice Natural Gas", PermissionType.Modify)]
    [Route("[action]/{id}")]
    public IActionResult DeleteInvoice(int id)
    {
        InvoiceGa? dbItem = db.InvoiceGas.Where(x => x.Id == id).FirstOrDefault();
        if (dbItem != null)
            db.Entry(dbItem).State = EntityState.Deleted;

        db.SaveChanges();

        return Ok();
    }

    class InvoiceSummaryLine
    {
        public string InvoiceNum = "";
        public decimal? TotalQuantity;
        public decimal? TotalAmount;
    }

    class InvoiceSummary
    {
        public decimal? TotalQuantity;
        public decimal? TotalAmount;
        public List<InvoiceSummaryLine> Lines = new();
    }

    [Permission("Invoice Natural Gas", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetInvoiceVolAndQuantity(DateOnly month, int internalEntityId, int counterpartyId)
    {
        var summary = new InvoiceSummary();

        var initialLinesBeforeMods = await InvoiceLineHelper.GetInitialLinesAsync(counterpartyId, internalEntityId, month);
        summary.TotalQuantity = initialLinesBeforeMods.Sum(x => x.Quantity);
        summary.TotalAmount = Math.Round(initialLinesBeforeMods.Sum(x => x.Amount), 2, MidpointRounding.AwayFromZero);

        summary.Lines = await db.InvoiceGas
        .Where(x => x.Month == month && x.InternalEntityId == internalEntityId && x.CounterpartyId == counterpartyId)
        .Select(q => new InvoiceSummaryLine
        { InvoiceNum = q.InvoiceNum, TotalQuantity = q.TotalQuantity, TotalAmount = q.Subtotal })
        .ToListAsync();

        return Ok(summary);
    }

    [Permission("Invoice Natural Gas", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> RegenerateLines(int counterpartyId, int internalEntityId, DateOnly month, List<InvoiceGasLineItem> existingLines)
    {
        var regeneratedLines = await InvoiceLineHelper.GetRegeneratedLinesAsync(counterpartyId, internalEntityId, month, existingLines);
        return Ok(regeneratedLines);
    }

    [Permission("Invoice Natural Gas", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> ReviseInvoice(RevisionParams p)
    {
        InvoiceLineHelper.ReverseSelectedLines(p.LineSelection, p.ExistingLines);
        var reversedLines = p.ExistingLines.Where(l => p.LineSelection.Contains(l.LineNum)).ToList();
        var regeneratedLines = await InvoiceLineHelper.GetRegeneratedLinesAsync(p.CounterpartyId, p.InternalEntityId, p.Month, p.ExistingLines);
        var revisedInvoiceLines = reversedLines.Concat(regeneratedLines).ToList();

        for (int i = 0; i < revisedInvoiceLines.Count; i++)
        {
            var line = revisedInvoiceLines[i];
            line.LineNum = i + 1;
            line.IsApproved = false;
        }

        return Ok(revisedInvoiceLines);
    }

    public class RevisionParams
    {
        public int CounterpartyId { get; set; }
        public int InternalEntityId { get; set; }
        public DateOnly Month { get; set; }
        public List<InvoiceGasLineItem> ExistingLines { get; set; } = new();
        public List<int> LineSelection { get; set; } = new();
    }
}
