import { Pipe, PipeTransform } from '@angular/core';
import { CurrencyPipe } from '@angular/common';

/**
 * Formats currency using accounting style - negative values are wrapped in parentheses.
 * Usage: {{ value | accountingCurrency }}
 * Example: -1234.56 becomes "($1,234.56)" instead of "-$1,234.56"
 */
@Pipe({
    name: 'accountingCurrency',
    standalone: true
})
export class AccountingCurrencyPipe implements PipeTransform {
    private currencyPipe = new CurrencyPipe('en-US');

    transform(value: number | null | undefined, currencyCode: string = 'USD'): string {
        if (value == null) {
            return this.currencyPipe.transform(0, currencyCode) ?? '$0.00';
        }

        const isNegative = value < 0;
        const absValue = Math.abs(value);
        const formatted = this.currencyPipe.transform(absValue, currencyCode) ?? '$0.00';

        return isNegative ? `(${formatted})` : formatted;
    }
}
