import { ChangeDetectionStrategy, ChangeDetectorRef, Component, computed, DoCheck, inject, Injector, input, OnInit, output, signal } from '@angular/core';
import { FAST_KENDO_COMMON } from '../../app.config';
import { ControlValueAccessor, FormControl, FormControlDirective, FormControlName, FormGroupDirective, NG_VALUE_ACCESSOR, NgControl } from '@angular/forms';
import { MessageService } from '../services/message.service';
import { dayPlaceholder, handleError, maxDate, minDate, monthPlaceholder } from '../utils/util';
import { isMobileDevice } from '../utils/util';

export type DateType = 'date-short' | 'date-month' | 'date-time';

@Component({
  selector: 'fast-datepicker',
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [FAST_KENDO_COMMON],
  providers: [{
    provide: NG_VALUE_ACCESSOR,
    multi: true,
    useExisting: FastDatePickerComponent
  },
  {
    provide: NgControl,
    multi: true,
    useExisting: FastDatePickerComponent
  }],
  template: `
  <kendo-datepicker
    [min]="min()"
    [max]="max()"
    [class]="conditionalClasses()"
    [format]="computedFormat()"
    [tabindex]="tabindex()"
    [value]="internalValue()"
    (valueChange)="onValueChange($event)"
    [disabled]="isDisabled()"
    [topView]="'decade'" [bottomView]="type() === 'date-month' ? 'year' : 'month'"
    [formatPlaceholder]="computedFormatPlaceholder()"
    [placeholder]="placeholder()"
    [adaptiveMode]="'auto'"
    [readOnlyInput]="isMobileDevice()"
    [inputAttributes]="{ inputmode: 'none' }"
    [popupSettings]="{ appendTo: 'component' }"
  />
  `
})
export class FastDatePickerComponent implements ControlValueAccessor, OnInit, DoCheck {
  messageService = inject(MessageService);
  injector = inject(Injector);
  isMobileDevice = isMobileDevice;

  cdr = inject(ChangeDetectorRef);
  min = input<Date | null>(minDate);
  max = input<Date | null>(maxDate);
  tabindex = input<number>(0);
  disabled = input<boolean>(false);
  valueChange = output<Date | null>();
  value = input<Date | null>(null);
  internalValue = signal<Date | null>(null);
  type = input<DateType>('date-short');
  placeholder = input<string>(null);
  invalidTrigger = signal(0);
  formControl: FormControl;

  // Internal signal to track form control disabled state
  private formControlDisabled = signal<boolean>(false);

  // Computed signal that combines input disabled state and form control disabled state
  isDisabled = computed(() => this.disabled() || this.formControlDisabled());

  ngDoCheck() {
    //needed so that isInvalid is recomputed when the form control is touched with changing the value
    // and when `markAllAsTouched` is called on the parent form
    if (this.formControl?.touched)
      this.invalidTrigger.update((v) => v + 1);
  }

  ngOnInit() {
    const ngControl = this.injector.get(NgControl);

    if (ngControl instanceof FormControlName) {
      this.formControl = this.injector
        .get(FormGroupDirective)
        .getControl(ngControl)
    } else {
      this.formControl = (ngControl as FormControlDirective).form as FormControl;
    }
  }

  isInvalid = computed(() => {
    this.invalidTrigger();
    const control = this.formControl;
    if (control && !this.isDisabled())
      return control.invalid && (control.touched || control.dirty)
    else
      return false;
  });

  computedFormat = computed(() => {
    if (this.type() === 'date-short')
      return 'M/d/yyyy';
    else if (this.type() === 'date-month')
      return 'MMM yyyy';
    else if (this.type() === 'date-time')
      return 'M/d/yyyy';
    else
      this.throwTypeError();
  });

  computedFormatPlaceholder = computed(() => {
    if (this.type() === 'date-short')
      return dayPlaceholder;
    else if (this.type() === 'date-month')
      return monthPlaceholder;
    else if (this.type() === 'date-time')
      return dayPlaceholder;
    else
      this.throwTypeError();
  });

  throwTypeError() {
    handleError(`Type {${this.type()}} not implemented in fast-datepicker`, this.messageService);
  }

  onChange: (value: Date | null) => void = () => { };
  onTouched = () => { };

  writeValue(value: Date | null) {
    this.internalValue.set(value);
    this.cdr.markForCheck();
  }

  registerOnChange(fn: () => void) {
    this.onChange = fn;
  }

  registerOnTouched(fn: () => void) {
    this.onTouched = fn;
  }
  setDisabledState(isDisabled: boolean) {
    this.formControlDisabled.set(isDisabled);
    this.cdr.markForCheck();
  }

  onValueChange(newValue: Date | null) {
    this.internalValue.set(newValue);
    this.onChange(newValue);
    this.invalidTrigger.update((v) => v + 1);
    this.valueChange.emit(newValue);
  }

  conditionalClasses = computed(() => {
    const classes = [] as string[];

    classes.push(...this.getCommonClasses());
    classes.push(...this.getLightBaseClasses());
    classes.push(...this.getDarkBaseClasses());
    if (!this.isDisabled()) {
      classes.push(...this.getLightHoverClasses());
      classes.push(...this.getDarkHoverClasses());
      classes.push(...this.getLightActiveClasses());
      classes.push(...this.getDarkActiveClasses());
    }

    const conditionalClasses = this.getConditionalClassesFromArrays(classes);
    return conditionalClasses;
  });

  getConditionalClassesFromArrays(classArray: string[]): { [key: string]: boolean } {
    const classes: { [key: string]: boolean } = {};
    classArray.forEach(className => {
      classes[className] = true;
    });
    return classes;
  }

  getCommonClasses() {
    return [
      "flex",
      "h-7",
      "border-solid",
      "rounded-sm",
      "text-sm",
      "[&_.k-input-button]:border-none",
      "[&_.k-input-button]:w-6",
      "[&_.k-input-button]:bg-transparent",
      "[&_input]:pl-3",
      "[&_input]:pr-0"];
  }

  getLightBaseClasses() {
    const classes = [
      "[&_.k-svg-i-calendar]:text-base-black-1000"];

    if (this.isInvalid()) {
      classes.push(
        "border-red-500",
        "ring-red-500/50",
        "ring-2"
      );
    } else {
      classes.push(
        "border-base-gray-500",
        "ring-base-blue-250/50"
      );
    }

    return classes;
  }

  getDarkBaseClasses() {
    const classes = [
      "dark:[&_.k-svg-i-calendar]:text-base-white-500"];

    if (this.isInvalid()) {
      classes.push(
        "dark:border-red-500",
        "dark:ring-red-500/50"
      );
    } else {
      classes.push(
        "dark:border-alt-blue-250",
        "dark:ring-base-blurple-250/50"
      );
    }

    return classes;
  }

  getLightHoverClasses() {
    return [
      "[&_.k-input-button]:hover:bg-base-white-1000"];
  }

  getDarkHoverClasses() {
    return [
      "dark:[&_.k-input-button]:hover:bg-alt-blue-750"];
  }

  getLightActiveClasses() {
    return [
      "[&_.k-input-button]:active:bg-base-gray-1000",];
  }

  getDarkActiveClasses() {
    return [
      "dark:[&_.k-input-button]:active:bg-base-blue-1000",];
  }
}
