﻿using System.Text.RegularExpressions;
using Fast.Shared.Logic.FileService;
using Fast.Shared.Logic.Package;
using Fast.Shared.Logic.Valuation;
using static Fast.Shared.Models.Enums;

namespace Fast.Logic;

public class TicketDocPhysicalGas(IFileService fileService, string filesFolderName, string templatesFolderName, string signaturesFolderName, string logosFolderName, MyDbContext db)
    : DocBaseOpenXml(fileService, filesFolderName, templatesFolderName, signaturesFolderName, logosFolderName, false)
{
    protected override string TemplateFileName => "Physical Gas Ticket.docx";

    public async Task<DealConfirmation> Generate(string dealNum, int userId)
    {
        DateOnly noStartDate = new(1900, 1, 1);
        DateOnly noEndDate = new(9999, 12, 31);

        //get the deal from the database without tracking so that we don't modify data in the database
        var deal = (
            from q in db.Deals
            where q.TicketNum == dealNum
            select new
            {
                q.Created,
                q.Modified,
                Counterparty = q.Counterparty == null ? "" : q.Counterparty.Name,
                InternalEntity = q.InternalEntity == null ? "" : q.InternalEntity.Name,
                Contact = q.Contact == null ? "" : q.Contact.DisplayName,
                BuySell = q.BuyButtonNavigation == null ? "" : q.BuyButtonNavigation.Name,
                q.NettingContractNumber,
                q.TradingDate,
                q.StartDate,
                q.EndDate,
                DealType = q.PhysicalDealType == null ? "" : q.PhysicalDealType.Name,
                PhysicalDealTypeId = q.PhysicalDealTypeId.GetValueOrDefault(),
                q.PremiumOrDiscount,
                IsFixedPrice = q.FixedPriceButton == 1,
                q.FixedPrice,
                q.DemandCharge,
                IndexName = q.PriceIndex != null ? q.PriceIndex.Name : "",
                NormalIndexDetermination = q.PriceIndex != null ? q.PriceIndex.PriceDetermination : "",
                IsHybrid = q.PriceIndex != null && q.PriceIndex.IndexTypeId == (int)Enums.MarketIndexType.Hybrid,
                HybridIndexDefinition = q.PriceIndex != null ? q.PriceIndex.HybridIndexDefinition : "",
                q.Volume,
                q.VolumeTypeId,
                PipeName = q.Pipeline == null ? "" : q.Pipeline.Name,
                PointName = q.Point == null ? "" : q.Point.Name,
                q.Comments,
                q.InternalMemo,
                TraderName = q.Trader == null ? "" : q.Trader.DisplayName,
                Region = q.Region == null ? "" : q.Region.Name,
                Broker = q.Broker == null ? "" : q.Broker.Name,
                LogoFileName = q.InternalEntity == null ? null : q.InternalEntity.LogoFileNameOnDisk,
            }
        ).AsNoTracking().First();

        var val = new Valuater();
        var valParams = new ValParams { TicketNum = dealNum };
        valParams.PositionDateRanges.Add(new DateRange(DateStyle.DateRange, deal.StartDate ?? noStartDate, deal.EndDate ?? noEndDate));
        var valResults = await val.GetValuationValuesGrouped(valParams);

        var counterpartyContract = (
            from q in db.ContractContacts
            where q.Contract != null && q.Contract.ContractNum == deal.NettingContractNumber
            select new
            {
                q.ConfirmationsAttn,
                q.ConfirmationsTelephoneNum,
                q.ConfirmationsFaxNum
            }
        ).AsNoTracking().FirstOrDefault();

        var docResult = (from q in db.DealConfirmations where q.TicketNum == dealNum select q).FirstOrDefault();
        if (docResult == null)
        {
            docResult = new DealConfirmation
            {
                TicketNum = dealNum
            };
            db.DealConfirmations.Add(docResult);
        }

        docResult.TicketFile = GetNewFileName(deal.Counterparty, dealNum);
        docResult.TicketedBy = userId;
        docResult.TicketDate = DateTime.Now.Date.ToUniversalTime();
        docResult.ModificationDate = DateTime.UtcNow;

        bool isRevised = IsRevised(deal.Created, deal.Modified, docResult.FaxDate, docResult.EmailDate);

        // Use SetLogoImage helper - sharding is handled internally by fileService
        var logoFileName = deal.LogoFileName ?? deal.InternalEntity + ".jpg";  // TODO: remove "??" fallback after transition to GUID filenames.
        SetLogoImage(logoFileName, "EntityLogo", dealNum);

        SetText("RevisedOrDeleted", isRevised ? "Revised Transaction" : "");
        SetText("TicketNum", dealNum);
        SetText("BuySell", deal.BuySell);
        SetText("DealType", deal.DealType);
        SetText("TraderName", deal.TraderName);
        SetText("StartDate", deal.StartDate?.ToString("M/d/yyyy") ?? "");
        SetText("EndDate", deal.EndDate?.ToString("M/d/yyyy") ?? "");
        SetText("TradeDate", deal.TradingDate?.ToString("M/d/yyyy") ?? "");
        SetText("Region", deal.Region);
        SetText("CounterpartyName", deal.Counterparty);
        SetText("RemgCredit", "???");
        SetText("BrokerExecuting", deal.Broker);
        SetText("CounterpartyAttn", counterpartyContract?.ConfirmationsAttn ?? "");
        SetText("CounterpartyPhone", counterpartyContract?.ConfirmationsTelephoneNum ?? "");
        SetText("CounterpartyFax", counterpartyContract?.ConfirmationsFaxNum ?? "");
        SetText("Contact", deal.Contact);
        SetText("Pipeline", deal.PipeName);
        SetText("Point", deal.PointName);
        SetText("AvgVolume", FormatNum((decimal)Math.Abs(deal.Volume.GetValueOrDefault()), NumFormat.n0));
        SetText("VolUnit", DealHelper.GetVolUnitText(deal.VolumeTypeId));
        SetText("FixedIndex", GetFixedIndexText(deal.PhysicalDealTypeId, deal.IsFixedPrice));
        SetText("FixedPrice", GetFixedPriceText(deal.PhysicalDealTypeId, deal.IsFixedPrice, deal.FixedPrice, deal.DemandCharge));
        SetText("IndexPrice", GetIndexPriceText(valResults, deal.PhysicalDealTypeId, deal.IsFixedPrice, deal.IndexName));
        SetText("PremiumOrDiscount", FormatNum((decimal?)deal.PremiumOrDiscount, NumFormat.n5));
        SetText("InvoicePrice", GetInvoicePriceText(valResults));
        SetText("Comments", deal.Comments);
        SetText("InternalMemo", deal.InternalMemo);
        SetText("PriceDetermination", GetPriceDetermination(deal.PhysicalDealTypeId, deal.IsFixedPrice, deal.NormalIndexDetermination, deal.IsHybrid, deal.HybridIndexDefinition));


        wordprocessingDocument.PackageProperties.Title = docResult.TicketFile;

        await using var docStream = SaveToMemoryStream();
        var pdfStream = await Shared.Logic.Package.PdfConverter.ConvertToStream(docStream, "document.docx");
        byte[] pdfBytes = pdfStream.ToArray();

        // Use fileService to write the PDF bytes directly - sharding is handled internally
        var uploadedFileName = await fileService.WriteAllBytesAsync(filesFolderName, docResult.TicketFile, pdfBytes);
        docResult.TicketFile = uploadedFileName;
        await db.SaveChangesAsync();

        return docResult;
    }

    private static string GetFixedIndexText(int physicalDealTypeId, bool isFixedPrice)
    {
        string fixedIndexText;
        if (physicalDealTypeId == (int)DealType.Demand)
            fixedIndexText = "Demand";
        else if (isFixedPrice)
            fixedIndexText = "Fixed";
        else
            fixedIndexText = "Index";

        return fixedIndexText;
    }

    private string GetPriceDetermination(int physicalDealTypeId, bool isFixedPrice, string normalIndexDetermination, bool isHybridIndex, string hybridIndexDefinition)
    {
        string priceDetermination = "";
        if (isHybridIndex)
        {
            HybridIndexHelper hybridHelper = new(db);
            string formula = hybridHelper.GetFormulaWithNames(hybridIndexDefinition);
            string hybridDetermination = hybridHelper.GetHybridDetermination(hybridIndexDefinition);
            priceDetermination = formula;
            if (!string.IsNullOrEmpty(hybridDetermination))
                priceDetermination += " where " + hybridDetermination;
        }
        else if (!isFixedPrice && physicalDealTypeId != (int)DealType.Demand)
        {
            priceDetermination = normalIndexDetermination;
        }

        return priceDetermination;
    }

    private static string GetFixedPriceText(int physicalDealTypeId, bool isFixedPrice, double? fixedPrice, string demandCharge)
    {
        string fixedPriceText = "";

        if (physicalDealTypeId == (int)DealType.Demand)
            fixedPriceText = demandCharge;
        else if (isFixedPrice)
            fixedPriceText = FormatNum((decimal)fixedPrice.GetValueOrDefault(), NumFormat.n5);

        return fixedPriceText;
    }

    private static string GetIndexPriceText(List<ValuationResultGrouped> valResults, int physicalDealTypeId, bool isFixedPrice, string indexName)
    {
        string indexPriceText = "";

        if (valResults.Any() && physicalDealTypeId != (int)DealType.Demand && !isFixedPrice)
        {
            var result = valResults.First();
            if (result.MinIsPosted)
                indexPriceText = FormatNum((decimal)result.AvgContractPrice, NumFormat.n5);
            else
                indexPriceText = indexName;
        }

        return indexPriceText;
    }

    private static string GetInvoicePriceText(List<ValuationResultGrouped> valResults)
    {
        string invoicePriceText = "";

        if (valResults.Any())
        {
            var result = valResults.First();
            if (result.MinIsPosted)
                invoicePriceText = FormatNum((decimal)result.AvgInvoicePrice, NumFormat.n5);
        }

        return invoicePriceText;
    }

    private static bool IsRevised(DateTime? created, DateTime? modified, DateTime? faxed, DateTime? emailed)
    {
        bool isRevised = false;
        if (created.HasValue && modified.HasValue && created.Value != modified.Value)
        {
            if (faxed.HasValue && modified.Value > faxed.Value)
                isRevised = true;

            if (emailed.HasValue && modified.Value > emailed.Value)
                isRevised = true;
        }

        return isRevised;
    }

    protected override HashSet<string> GetDocKeys()
    {
        HashSet<string> keys = new(StringComparer.OrdinalIgnoreCase)
            {
                "RevisedOrDeleted",
                "TicketNum",
                "BuySell",
                "DealType",
                "TraderName",
                "StartDate",
                "EndDate",
                "TradeDate",
                "Region",
                "CounterpartyName",
                "RemgCredit",
                "BrokerExecuting",
                "CounterpartyAttn",
                "CounterpartyPhone",
                "CounterpartyFax",
                "Contact",
                "Pipeline",
                "Point",
                "AvgVolume",
                "VolUnit",
                "FixedIndex",
                "FixedPrice",
                "IndexPrice",
                "PremiumOrDiscount",
                "InvoicePrice",
                "Comments",
                "InternalMemo",
                "PriceDetermination"
            };

        return keys;
    }

    private static string GetNewFileName(string counterparty, string dealNum)
    {
        var monthStr = DateTime.Now.ToString("MM-dd-yyyy");
        counterparty = Util.String.GetLegalFileName(counterparty);
        string newFileName = $"T-{counterparty}-{monthStr}-{dealNum}.pdf";
        return newFileName;
    }
}
