﻿using System.Text.RegularExpressions;
using Fast.Shared.Logic.FileService;
using Fast.Shared.Logic.Package;
using static Fast.Shared.Models.Enums;

namespace Fast.Logic;


public class ConfirmDocPhysicalGas(IFileService fileService, string filesFolderName, string templatesFolderName, string signaturesFolderName, string logosFolderName, MyDbContext db)
    : DocBaseOpenXml(fileService, filesFolderName, templatesFolderName, signaturesFolderName, logosFolderName, false)
{
    protected override string TemplateFileName => "Physical Gas Confirm.docx";

    public async Task<DealConfirmation> Generate(string dealNum, int userId)
    {
        //get the deal from the database without tracking so that we don't modify data in the database
        var deal = (
            from q in db.Deals
            where q.TicketNum == dealNum
            select new
            {
                q.Created,
                q.Modified,
                Counterparty = q.Counterparty == null ? "" : q.Counterparty.Name,
                InternalEntity = q.InternalEntity == null ? "" : q.InternalEntity.Name,
                Contact = q.Contact == null ? "" : q.Contact.DisplayName,
                q.BuyButton,
                q.NettingContractNumber,
                q.InternalEntityId,
                q.TradingDate,
                q.StartDate,
                q.EndDate,
                PhysicalDealTypeId = q.PhysicalDealTypeId.GetValueOrDefault(),
                q.PremiumOrDiscount,
                IsFixedPrice = q.FixedPriceButton == 1,
                q.FixedPrice,
                q.DemandCharge,
                NormalIndexDetermination = q.PriceIndex != null ? q.PriceIndex.PriceDetermination : "",
                IsHybrid = q.PriceIndex != null && q.PriceIndex.IndexTypeId == (int)Enums.MarketIndexType.Hybrid,
                HybridIndexDefinition = q.PriceIndex != null ? q.PriceIndex.HybridIndexDefinition : "",
                q.Volume,
                q.VolumeTypeId,
                PipeName = q.Pipeline == null ? "" : q.Pipeline.Name,
                PointName = q.Point == null ? "" : q.Point.Name,
                q.Comments,
                q.FmLanguage,
                FullLegalName = q.Trader == null ? "" : q.Trader.LegalName,
                InternalAnalystTitle = q.Trader == null ? "" : q.Trader.Title,
                AttachLclanguage = q.AttachLcLanguage.GetValueOrDefault(),
                LogoFileName = q.InternalEntity == null ? null : q.InternalEntity.LogoFileNameOnDisk,
                SignatureFileName = q.Trader == null ? null : q.Trader.SignatureFileNameOnDisk,
            }
        ).AsNoTracking().First();

        var counterpartyContract = (
            from q in db.ContractContacts
            where q.Contract != null && q.Contract.ContractNum == deal.NettingContractNumber
            select new
            {
                q.ConfirmationsAddressLine1,
                q.ConfirmationsAddressLine2,
                q.ConfirmationsCity,
                ConfirmationsState = q.ConfirmationsState == null ? "" : q.ConfirmationsState.Name,
                q.ConfirmationsZip,
                q.ConfirmationsAttn,
                q.ConfirmationsTelephoneNum,
                q.ConfirmationsFaxNum
            }
        ).AsNoTracking().FirstOrDefault();

        var internalEntityContract = (
            from q in db.ContractOthers
            where q.Contract.InternalEntityId == deal.InternalEntityId
            select new
            {
                q.AgentClause,
                q.LcLanguage
            }
        ).AsNoTracking().FirstOrDefault();

        var docResult = (from q in db.DealConfirmations where q.TicketNum == dealNum select q).FirstOrDefault();
        if (docResult == null)
        {
            docResult = new DealConfirmation
            {
                TicketNum = dealNum
            };
            db.DealConfirmations.Add(docResult);
        }

        docResult.ConfirmFile = GetNewFileName(deal.Counterparty, dealNum);
        docResult.ConfirmedBy = userId;
        docResult.ConfirmDate = DateTime.Now.Date.ToUniversalTime();
        docResult.ModificationDate = DateTime.UtcNow;

        bool isRevised = IsRevised(deal.Created, deal.Modified, docResult.FaxDate, docResult.EmailDate);
        DateTime currentTime = DateTime.Now;
        var city = counterpartyContract?.ConfirmationsCity;
        var comma = string.IsNullOrWhiteSpace(city) ? "" : ",";
        var state = counterpartyContract?.ConfirmationsState;
        var zip = counterpartyContract?.ConfirmationsZip;
        var cityStateZip = $"{city}{comma} {state} {zip}";
        var attention = counterpartyContract?.ConfirmationsAttn;

        // Use SetLogoImage and SetSignatureImage helpers - sharding is handled internally by fileService
        var logoFileName = deal.LogoFileName ?? deal.InternalEntity + ".jpg";  // TODO: remove "??" fallback after transition to GUID filenames.
        var signatureFileName = deal.SignatureFileName ?? deal.FullLegalName + ".jpg";  // TODO: remove "??" fallback after transition to GUID filenames.
        SetLogoImage(logoFileName, "EntityLogo", dealNum);
        SetSignatureImage(signatureFileName, "TraderSignature", dealNum);

        SetText("RevisedOrDeleted", isRevised ? "Revised Transaction" : "");
        SetText("TodaysDate", currentTime.Date.ToString("MM/dd/yyyy"));
        SetText("TimeGenerated", currentTime.ToString("t"));
        SetText("TicketNum", dealNum);
        SetText("CounterpartyName", deal.Counterparty);
        SetText("CounterpartyAddress1", counterpartyContract?.ConfirmationsAddressLine1 ?? "");
        SetText("CounterpartyAddress2", counterpartyContract?.ConfirmationsAddressLine2 ?? "");
        SetText("CounterpartyCityStateZip", cityStateZip);
        SetText("CounterpartyAttn", !string.IsNullOrWhiteSpace(attention) ? "Attn: " + attention : "");
        SetText("Contact", deal.Contact);
        SetText("CounterpartyBuyerSeller", deal.BuyButton == -1 ? "Buyer" : deal.BuyButton == 1 ? "Seller" : "N/A");
        SetText("CounterpartyPhone", counterpartyContract?.ConfirmationsTelephoneNum ?? "");
        SetText("CounterpartyFax", counterpartyContract?.ConfirmationsFaxNum ?? "");
        SetText("InternalEntity", deal.InternalEntity);
        SetText("InternalBuyerSeller", deal.BuyButton == -1 ? "Seller" : deal.BuyButton == 1 ? "Buyer" : "N/A");
        SetText("TradeDate", deal.TradingDate?.ToString("M/d/yyyy") ?? "");
        SetText("StartDate", deal.StartDate?.ToString("M/d/yyyy") ?? "");
        SetText("EndDate", deal.EndDate?.ToString("M/d/yyyy") ?? "");
        SetText("DealType", GetDealTypeText(deal.PhysicalDealTypeId));
        SetText("Price", GetPriceText(deal.PhysicalDealTypeId, deal.IsHybrid, deal.IsFixedPrice, deal.FixedPrice, deal.DemandCharge, deal.PremiumOrDiscount));
        SetText("PriceDetermination", GetPriceDetermination(deal.PhysicalDealTypeId, deal.IsFixedPrice, deal.NormalIndexDetermination, deal.IsHybrid, deal.HybridIndexDefinition));
        SetText("PipelinePoint", deal.PipeName + " ; " + deal.PointName);
        SetText("Comments", deal.Comments);
        SetText("AgentClause", deal.BuyButton == -1 ? internalEntityContract?.AgentClause ?? "" : "");
        SetText("NoticeForceMajeure", !string.IsNullOrWhiteSpace(deal.FmLanguage) ? "Notice of Force Majeure" : "");
        SetText("ForceMajeure", deal.FmLanguage);
        SetText("TraderName", deal.FullLegalName);
        SetText("TraderTitle", deal.InternalAnalystTitle);
        SetText("LCLanguage", deal.AttachLclanguage ? internalEntityContract?.LcLanguage ?? "" : "");
        SetText("AvgVolume", FormatNum((decimal)Math.Abs(deal.Volume.GetValueOrDefault()), NumFormat.n0));
        SetText("VolUnit", DealHelper.GetVolUnitText(deal.VolumeTypeId));

        wordprocessingDocument.PackageProperties.Title = docResult.ConfirmFile;

        await using var docStream = SaveToMemoryStream();
        var pdfStream = await Shared.Logic.Package.PdfConverter.ConvertToStream(docStream, "document.docx");
        byte[] pdfBytes = pdfStream.ToArray();

        // Use fileService to write the PDF bytes directly - sharding is handled internally
        var uploadedFileName = await fileService.WriteAllBytesAsync(filesFolderName, docResult.ConfirmFile, pdfBytes);
        docResult.ConfirmFile = uploadedFileName;
        await db.SaveChangesAsync();

        return docResult;
    }

    private string GetPriceDetermination(int physicalDealTypeId, bool isFixedPrice, string normalIndexDetermination, bool isHybridIndex, string hybridIndexDefinition)
    {
        string priceDetermination = "";
        if (isHybridIndex)
        {
            HybridIndexHelper hybridHelper = new(db);
            string formula = hybridHelper.GetFormulaWithNames(hybridIndexDefinition);
            string hybridDetermination = hybridHelper.GetHybridDetermination(hybridIndexDefinition);
            priceDetermination = formula;
            if (!string.IsNullOrEmpty(hybridDetermination))
                priceDetermination += " where " + hybridDetermination;
        }
        else if (!isFixedPrice && physicalDealTypeId != (int)DealType.Demand)
        {
            priceDetermination = normalIndexDetermination;
        }

        return priceDetermination;
    }

    private static string GetPriceText(int physicalDealTypeId, bool isHybrid, bool isFixedPrice, double? fixedPrice, string demandCharge, double? premDisc)
    {
        string priceText;
        if (physicalDealTypeId == (int)DealType.Demand)
            priceText = demandCharge;
        else if (isFixedPrice)
            priceText = FormatNum((decimal)fixedPrice.GetValueOrDefault(), NumFormat.n5) + " ($/MMBTU)";
        else if (isHybrid)
            priceText = "See Price Determination below";
        else
        {
            string suffix;
            if (premDisc.HasValue && premDisc.Value > 0)
                suffix = "plus $" + FormatNum((decimal)Math.Abs(premDisc.Value), NumFormat.n5);
            else if (premDisc.HasValue)
                suffix = "minus $" + FormatNum((decimal)Math.Abs(premDisc.Value), NumFormat.n5);
            else
                suffix = "$Even";

            priceText = $"\"Index Price\" {suffix} ($/MMBTU)";
        }

        return priceText;
    }

    private static string GetDealTypeText(int physicalDealTypeId)
    {
        string dealType;
        var firmDealTypeIds = new List<int> { 2, 4, 5, 7 };
        var firmVariableDealTypeId = 11;
        if (firmDealTypeIds.Contains(physicalDealTypeId))
            dealType = "Firm";
        else if (physicalDealTypeId == firmVariableDealTypeId)
            dealType = "Firm - Variable";
        else
            dealType = "Interruptible";

        return dealType;
    }

    private static bool IsRevised(DateTime? created, DateTime? modified, DateTime? faxed, DateTime? emailed)
    {
        bool isRevised = false;
        if (created.HasValue && modified.HasValue && created.Value != modified.Value)
        {
            if (faxed.HasValue && modified.Value > faxed.Value)
                isRevised = true;

            if (emailed.HasValue && modified.Value > emailed.Value)
                isRevised = true;
        }

        return isRevised;
    }

    protected override HashSet<string> GetDocKeys()
    {
        HashSet<string> keys = new(StringComparer.OrdinalIgnoreCase)
            {
                "RevisedOrDeleted",
                "TodaysDate",
                "TicketNum",
                "CounterpartyBuyerSeller",
                "CounterpartyName",
                "CounterpartyAddress1",
                "CounterpartyAddress2",
                "CounterpartyCityStateZip",
                "CounterpartyAttn",
                "CounterpartyPhone",
                "CounterpartyFax",
                "Contact",
                "InternalBuyerSeller",
                "InternalEntity",
                "TradeDate",
                "StartDate",
                "EndDate",
                "AvgVolume",
                "VolUnit",
                "DealType",
                "Price",
                "PriceDetermination",
                "PipelinePoint",
                "Comments",
                "AgentClause",
                "NoticeForceMajeure",
                "ForceMajeure",
                "TraderName",
                "TraderTitle",
                "LCLanguage",
                "Timegenerated"
            };

        return keys;
    }

    private static string GetNewFileName(string counterparty, string dealNum)
    {
        var monthStr = DateTime.Now.ToString("MM-dd-yyyy");
        counterparty = Util.String.GetLegalFileName(counterparty);
        string newFileName = $"C-{counterparty}-{monthStr}-{dealNum}.pdf";
        return newFileName;
    }
}
