﻿using System.Text.RegularExpressions;
using Fast.Shared.Logic.FileService;
using Fast.Shared.Logic.Package;
using static Fast.Shared.Models.Enums;

namespace Fast.Logic;

public class TicketDocFutures(IFileService fileService, string filesFolderName, string templatesFolderName, string signaturesFolderName, string logosFolderName, MyDbContext db)
    : DocBaseOpenXml(fileService, filesFolderName, templatesFolderName, signaturesFolderName, logosFolderName, false)
{
    protected override string TemplateFileName => "Futures Ticket.docx";

    public async Task<DealConfirmation> Generate(string dealNum, int userId)
    {
        DateOnly noStartDate = new(1900, 1, 1);
        DateOnly noEndDate = new(9999, 12, 31);

        //get the deal from the database without tracking so that we don't modify data in the database
        var futuresDeal = await (
            from q in db.Deals
            where q.TicketNum == dealNum
            select new
            {
                q.Created,
                q.Modified,
                InternalEntity = q.InternalEntity == null ? "" : q.InternalEntity.Name,
                q.WaspNum,
                BuySell = q.BuyButtonNavigation == null ? "" : q.BuyButtonNavigation.Name,
                StartDate = q.StartDate ?? noStartDate,
                TraderName = q.Trader == null ? "" : q.Trader.DisplayName ?? "",
                q.NumOfContracts,
                Product = q.Product.Name,
                Region = q.Region == null ? "" : q.Region.Name ?? "",
                q.FixedPrice,
                q.HedgeFee,
                q.TradingDate,
                Broker = q.Broker == null ? "" : q.Broker.Name ?? "",
                DealPurpose = q.DealPurpose == null ? "" : q.DealPurpose.Name ?? "",
                q.Comments,
                q.InternalMemo,
                LogoFileName = q.InternalEntity == null ? null : q.InternalEntity.LogoFileNameOnDisk,
            }
        ).AsNoTracking().FirstAsync();

        int numDaysInMonth = DateTime.DaysInMonth(futuresDeal.StartDate.Year, futuresDeal.StartDate.Month);
        bool hasWaspNum = !string.IsNullOrWhiteSpace(futuresDeal.WaspNum);

        var hedgeDeal = (
            from q in db.Deals
            where hasWaspNum && q.WaspNum == futuresDeal.WaspNum
            orderby q.Id descending
            select new
            {
                DealNum = q.TicketNum,
                BuySell = q.BuyButtonNavigation == null ? "" : q.BuyButtonNavigation.Name ?? "",
                Counterparty = q.Counterparty == null ? "" : q.Counterparty.Name,
                DealPurpose = q.DealPurpose == null ? "" : q.DealPurpose.Name ?? "",
                StartDate = q.StartDate ?? noStartDate,
                EndDate = q.EndDate ?? noEndDate,
                q.FixedPrice,
                q.Volume,
                PipeName = q.Pipeline == null ? "" : q.Pipeline.Name ?? "",
                PointName = q.Point == null ? "" : q.Point.Name ?? ""
            }
        ).AsNoTracking().FirstOrDefault();

        var docResult = (from q in db.DealConfirmations where q.TicketNum == dealNum select q).FirstOrDefault();
        if (docResult == null)
        {
            docResult = new DealConfirmation
            {
                TicketNum = dealNum
            };
            db.DealConfirmations.Add(docResult);
        }

        docResult.TicketFile = GetNewFileName(hedgeDeal?.Counterparty ?? "", dealNum);
        docResult.TicketedBy = userId;
        docResult.TicketDate = DateTime.Now.Date.ToUniversalTime();
        docResult.ModificationDate = DateTime.UtcNow;

        // Use SetLogoImage helper - sharding is handled internally by fileService
        var logoFileName = futuresDeal.LogoFileName ?? futuresDeal.InternalEntity + ".jpg";  // TODO: remove "??" fallback after transition to GUID filenames.
        SetLogoImage(logoFileName, "EntityLogo", dealNum);

        bool isRevised = IsRevised(futuresDeal.Created, futuresDeal.Modified);

        SetText("RevisedOrDeleted", isRevised ? "Revised Transaction" : "");
        SetText("TicketNum", dealNum);
        SetText("BuySell", futuresDeal.BuySell == "Buy" ? "Buyer" : "Seller");
        SetText("ContractMonth", Util.Date.FirstDayOfMonth(futuresDeal.StartDate).ToString("MMM-yyyy"));
        SetText("TraderName", futuresDeal.TraderName);
        SetText("NumOfContracts", FormatNum(futuresDeal.NumOfContracts, NumFormat.n4));
        SetText("Product", futuresDeal.Product);
        SetText("Region", futuresDeal.Region);
        SetText("Price", FormatNum((decimal)(futuresDeal?.FixedPrice ?? 0), NumFormat.n5));
        SetText("TradeDate", futuresDeal?.TradingDate?.ToString("M/d/yyyy") ?? "");
        SetText("HedgeFee", FormatNum((decimal?)(futuresDeal?.HedgeFee ?? 0), NumFormat.n5));
        SetText("BrokerExecuting", futuresDeal?.Broker ?? "");
        SetText("BrokerClearing", "");
        SetText("CPBrokerExecuting", "");
        SetText("CPBrokerClearing", "");
        SetText("DealPurpose", futuresDeal?.DealPurpose ?? "");
        SetText("LastDays", "");
        SetText("Comments", futuresDeal?.Comments ?? "");
        SetText("InternalMemo", futuresDeal?.InternalMemo ?? "");

        if (hedgeDeal != null)
        {
            SetText("TTicketNum", hedgeDeal.DealNum);
            SetText("TBuySell", hedgeDeal.BuySell == "Buy" ? "Buyer" : "Seller");
            SetText("TCounterpartyName", hedgeDeal.Counterparty);
            SetText("TDealType", hedgeDeal.DealPurpose);
            SetText("TStartDate", hedgeDeal.StartDate.ToString("M/d/yyyy"));
            SetText("TEndDate", hedgeDeal.EndDate.ToString("M/d/yyyy"));
            SetText("TPrice", FormatNum((decimal?)hedgeDeal.FixedPrice, NumFormat.n5));
            SetText("TTotalVolume", FormatNum((decimal?)hedgeDeal.Volume * numDaysInMonth, NumFormat.n5));
            SetText("TAvgVolume", FormatNum((decimal?)hedgeDeal.Volume, NumFormat.n5));
            SetText("TPipelinePoint", hedgeDeal.PipeName + " + " + hedgeDeal.PointName);
        }
        else
        {
            SetText("TTicketNum", "");
            SetText("TBuySell", "");
            SetText("TCounterpartyName", "");
            SetText("TDealType", "");
            SetText("TStartDate", "");
            SetText("TEndDate", "");
            SetText("TPrice", "");
            SetText("TTotalVolume", "");
            SetText("TAvgVolume", "");
            SetText("TPipelinePoint", "");
        }

        wordprocessingDocument.PackageProperties.Title = docResult.TicketFile;

        await using var docStream = SaveToMemoryStream();
        var pdfStream = await Shared.Logic.Package.PdfConverter.ConvertToStream(docStream, "document.docx");
        byte[] pdfBytes = pdfStream.ToArray();

        // Use fileService to write the PDF bytes directly - sharding is handled internally
        var uploadedFileName = await fileService.WriteAllBytesAsync(filesFolderName, docResult.TicketFile, pdfBytes);
        docResult.TicketFile = uploadedFileName;
        await db.SaveChangesAsync();

        return docResult;
    }


    private static bool IsRevised(DateTime? created, DateTime? modified)
    {
        bool isRevised = false;
        if (created.HasValue && modified.HasValue && created.Value != modified.Value)
            isRevised = true;

        return isRevised;
    }

    protected override HashSet<string> GetDocKeys()
    {
        HashSet<string> keys = new(StringComparer.OrdinalIgnoreCase) {
                "RevisedOrDeleted",
                "TicketNum",
                "BuySell",
                "ContractMonth",
                "TraderName",
                "NumOfContracts",
                "Product",
                "Region",
                "Price",
                "TradeDate",
                "HedgeFee",
                "BrokerExecuting",
                "BrokerClearing",
                "CPBrokerExecuting",
                "CPBrokerClearing",
                "DealPurpose",
                "LastDays",
                "Comments",
                "InternalMemo",
                "TTicketNum",
                "TBuySell",
                "TCounterpartyName",
                "TDealType",
                "TStartDate",
                "TEndDate",
                "TPrice",
                "TTotalVolume",
                "TAvgVolume",
                "TPipelinePoint"
            };

        return keys;
    }

    private static string GetNewFileName(string counterparty, string dealNum)
    {
        var monthStr = DateTime.Today.ToString("MM-dd-yyyy");
        string newFileName;
        if (string.IsNullOrWhiteSpace(counterparty))
            newFileName = $"T-{monthStr}-{dealNum}.pdf";
        else
        {
            counterparty = Util.String.GetLegalFileName(counterparty);
            newFileName = $"T-{counterparty}-{monthStr}-{dealNum}.pdf";
        }

        return newFileName;
    }
}
