﻿namespace Fast.Logic;

public static class DealHelper
{
    public class BookInfo
    {
        public int BookId;
        public string BookName = "";
        public bool IsActive;
        public int? TraderId;
    }

    public static List<BookInfo> GetBookInfos()
    {
        var db = Main.CreateContext();
        var items = (
            from b in db.Books
            join mbt in db.BookToTraders on b.Id equals mbt.BookId into j1
            from mbt in j1.DefaultIfEmpty()
            orderby mbt.Book.Name
            select new BookInfo
            {
                BookId = b.Id,
                BookName = b.IsActive ? b.Name ?? "" : "{Inactive} " + (b.Name ?? ""),
                IsActive = b.IsActive,
                TraderId = mbt.TraderId
            }
        ).ToList();

        if (items == null || !items.Any())
            items = new List<BookInfo> { new BookInfo { BookId = 1, BookName = "Unassigned", IsActive = true, TraderId = null } };

        items = items.OrderByDescending(x => x.BookId == 1).ThenBy(x => x.BookName).ToList();

        return items;
    }

    public static List<IdNameActive> GetPortfolios()
    {
        var db = Main.CreateContext();
        var items = (
            from q in db.Portfolios.AsEnumerable()
            orderby q.Id == 1 descending, q.Name
            select new IdNameActive(q.Id, q.Name ?? "", q.IsActive)
        ).ToList();

        return items;
    }

    public static List<IdNameActive> GetRegions()
    {
        var db = Main.CreateContext();
        var items = (
            from q in db.Regions.AsEnumerable()
            orderby q.Id == 1 descending, q.Name
            select new IdNameActive(q.Id, q.Name ?? "", q.IsActive)
        ).ToList();

        return items;
    }

    public static List<IdNameActive> GetStrategies()
    {
        var db = Main.CreateContext();
        var items = (
            from q in db.Strategies.AsEnumerable()
            orderby q.Id == 1 descending, q.Name
            select new IdNameActive(q.Id, q.Name ?? "", q.IsActive)
        ).ToList();

        return items;
    }

    public class ContactInfoItem
    {
        public int Id;
        public string Name = "";
        public int? CounterpartyId;
        public string Info = "";
    }
    public static List<ContactInfoItem> GetTradingContacts()
    {
        var db = Main.CreateContext();
        var items = (
            from q in db.ContactCounterparties
            where q.Contact.ContactToContactTypes.Any(x => x.ContactTypeId == 1)
            orderby q.Contact.DisplayName
            select new
            {
                q.Contact.Id,
                q.Contact.DisplayName,
                CounterpartyId = q.Counterparty.Id,
                q.Contact.OfficePhone,
                q.Contact.CellPhone,
                q.Contact.Fax,
                q.Contact.Email,
            }
        ).ToList();

        var contacts = new List<ContactInfoItem>();

        foreach (var item in items)
        {
            var strs = new List<string>();
            if (!string.IsNullOrWhiteSpace(item.OfficePhone))
                strs.Add("Office Phone: " + item.OfficePhone);
            if (!string.IsNullOrWhiteSpace(item.CellPhone))
                strs.Add("Cell Phone: " + item.CellPhone);
            if (!string.IsNullOrWhiteSpace(item.Fax))
                strs.Add("Fax: " + item.Fax);
            if (!string.IsNullOrWhiteSpace(item.Email))
                strs.Add("Email: " + item.Email);

            var contact = new ContactInfoItem
            {
                Id = item.Id,
                Name = item.DisplayName,
                CounterpartyId = item.CounterpartyId,
                Info = string.Join("•", strs)
            };
            contacts.Add(contact);
        }

        return contacts;
    }

    public static List<DealFilter> GetFilters(int userId)
    {
        var db = Main.CreateContext();
        DealFilterHelper.AddDefaultFiltersIfNeeded(db, userId);

        var result = (
            from q in db.DealFilters
            where q.UserId == userId
            orderby q.Name
            select new DealFilter
            {
                Id = q.Id,
                Name = q.Name,
                IsSelected = q.IsSelected,
                Columns = q.Columns,
                State = q.State
            }
        ).ToList();

        return result;
    }

    public class TraderInfo
    {
        public int TraderId;
        public string TraderName = "";
        public string UserName = "";
        public bool IsActive;
    }

    public static List<TraderInfo> GetTraders()
    {
        var currentTime = DateTime.UtcNow;
        var db = Main.CreateContext();
        var items = (
            from q in db.AppUsers.AsEnumerable()
            orderby q.DisplayName
            select new TraderInfo
            {
                TraderId = q.Id,
                TraderName = q.DisplayName ?? "",
                UserName = q.UserName ?? "",
                IsActive = q.LockoutEnd == null || currentTime < q.LockoutEnd
            }
        ).ToList();

        return items;
    }

    public static List<CounterpartyInfo> GetCounterparties()
    {
        var db = Main.CreateContext();
        var entitiesInitial = (
            from q in db.Counterparties
            orderby q.Name
            select new
            {
                q.Id,
                q.Name,
                q.CounterpartyRelationships,
                q.CounterpartyProducts,
                IsActive = q.InactiveDate == null
            }
        ).ToList();

        var entitiesFinal = (
            from q in entitiesInitial
            orderby q.IsActive descending, q.Name
            select new CounterpartyInfo
            {
                CounterpartyId = q.Id,
                CounterpartyName = q.Name,
                IsCrudeEntity = q.CounterpartyProducts.Any(x => x.ProductId == (int)Enums.Product.CrudeOil),
                IsFinancialEntity = q.CounterpartyRelationships.Any(x => x.BusinessRelationshipId == (int)Enums.BusinessRelationship.FinancialOnlyTradingCounterparty),
                IsTradingEntity = q.CounterpartyRelationships.Any(x => x.BusinessRelationshipId == (int)Enums.BusinessRelationship.TradingCounterparty),
                IsTransportEntity = q.CounterpartyRelationships.Any(x => x.BusinessRelationshipId == (int)Enums.BusinessRelationship.Transporter),
                IsGasEntity = q.CounterpartyProducts.Any(x => x.ProductId == (int)Enums.Product.NaturalGas),
                IsActive = q.IsActive
            }
        ).ToList();

        return entitiesFinal;
    }

    public static List<IdNameActive> GetNewCounterparties()
    {
        var db = Main.CreateContext();

        var items = (
            from q in db.Counterparties
            orderby q.Name
            select new IdNameActive(q.Id, q.Name, !q.InactiveDate.HasValue)
        ).ToList();

        return items;
    }

    public static List<IdNameActive> GetInternalEntities()
    {
        var db = Main.CreateContext();

        var items = (
            from q in db.Counterparties
            where q.CounterpartyRelationships.Any(x => x.BusinessRelationshipId == 1)
            orderby q.Name
            select new IdNameActive(q.Id, q.Name, q.InactiveDate == null)
        ).ToList();

        return items;
    }

    public static List<IdName> GetDealStatuses()
    {
        var db = Main.CreateContext();

        var items = (
            from q in db.DealStatuses
            where q.ForPhysicalGas == true
            orderby q.Name
            select new IdName(q.Id, q.Name ?? "")
        ).ToList();

        return items;
    }

    public static List<IdName> GetDealPurposes()
    {
        var db = Main.CreateContext();

        var items = (
            from q in db.DealPurposeTypes
            where q.ForPhysicalGas == true
            orderby q.Name
            select new IdName(q.Id, q.Name ?? "")
        ).ToList();

        return items;
    }

    public static List<IdName> GetDealTypes()
    {
        var db = Main.CreateContext();

        var items = (
            from q in db.PhysicalDealTypes
            where q.IsActive == true
            orderby q.Name
            select new IdName(q.Id, q.Name ?? "")
        ).ToList();

        return items;
    }

    [Serializable]
    public class CounterpartyInfo
    {
        public int CounterpartyId;
        public string CounterpartyName = "";
        public bool IsGasEntity;
        public bool IsCrudeEntity;
        public bool IsFinancialEntity;
        public bool IsTradingEntity;
        public bool IsTransportEntity;
        public bool IsActive;
    }

    [Serializable]
    public class PointInfo
    {
        public int PointId;
        public string PointName = "";
        public bool IsActive;
        public int PipeId;
        public bool IsGasPoint;
        public bool IsCrudePoint;
        public bool IsNglPoint;
    }

    public static List<PointInfo> GetPoints()
    {
        var gasProductId = (int)Enums.Product.NaturalGas;
        var crudeProductId = (int)Enums.Product.CrudeOil;
        var nglproductIds = new List<int> {
            (int)Enums.Product.Ethane,
            (int)Enums.Product.Propane,
            (int)Enums.Product.NormalButane,
            (int)Enums.Product.IsoButane,
            (int)Enums.Product.NaturalGasoline,
            (int)Enums.Product.YGrade,
            (int)Enums.Product.UnspecifiedNgl
        };
        var db = Main.CreateContext();
        var itemsTemp = (
            from p in db.Points
            orderby p.Name
            select new
            {
                p.Id,
                PointName = p.IsActive ? p.Name ?? "" : "{Inactive} " + (p.Name ?? ""),
                p.IsActive,
                PipeId = p.PipelineId ?? 0,
                p.ProductIds
            }
        ).AsNoTracking().ToList();

        var items = (
            from q in itemsTemp
            select new PointInfo
            {
                PointId = q.Id,
                PointName = q.PointName,
                IsActive = q.IsActive,
                PipeId = q.PipeId,
                IsGasPoint = q.ProductIds.Split(",", StringSplitOptions.None).Select(x => int.Parse(x)).Any(x => x == gasProductId),
                IsCrudePoint = q.ProductIds.Split(",", StringSplitOptions.None).Select(x => int.Parse(x)).Any(x => x == crudeProductId),
                IsNglPoint = q.ProductIds.Split(",", StringSplitOptions.None).Select(x => int.Parse(x)).Any(x => nglproductIds.Contains(x))
            }
        ).ToList();

        return items;
    }

    public static List<IdName> GetDeliveryMethods()
    {
        var db = Main.CreateContext();

        var items = (
            from q in db.DeliveryModes
            orderby q.Name
            select new IdName(q.Id, q.Name ?? "")
        ).ToList();

        return items;
    }

    public static List<IdName> GetBrokers()
    {
        var db = Main.CreateContext();

        var items = (
            from q in db.Brokers
            orderby q.Name
            select new IdName(q.Id, q.Name ?? "")
        ).ToList();

        return items;
    }

    [Serializable]
    public class BrokerAccountInfo
    {
        public int BrokerAccountId;
        public string BrokerAccountName = "";
        public int BrokerId;
    }

    public static List<BrokerAccountInfo> GetBrokerAccounts()
    {
        var db = Main.CreateContext();
        var items = (
            from q in db.BrokerAccounts
            orderby q.Name
            select new BrokerAccountInfo
            {
                BrokerAccountId = q.Id,
                BrokerAccountName = q.Name ?? "",
                BrokerId = q.BrokerId == null ? 0 : int.Parse(q.BrokerId)
            }
        ).ToList();

        return items;
    }

    public class IndexInfo
    {
        public int Id;
        public string Name = "";
        public int? ProductId;
    }


    public static List<IndexInfo> GetPriceIndexes()
    {
        var db = Main.CreateContext();

        var items = (
            from q in db.MarketIndices
            orderby q.Name
            select new IndexInfo
            {
                Id = q.Id,
                Name = q.Name ?? "",
                ProductId = q.ProductId
            }
        ).ToList();

        return items;
    }

    public class ContractInfo
    {
        public string ContractText = "";
        public string ContractNumber = "";
        public int CounterpartyId;
        public int InternalEntityId;
        public bool IsActive;
    }

    public static List<ContractInfo> GetContracts()
    {
        var db = Main.CreateContext();
        var today = DateTime.Today.ToUniversalTime();

        var items = (
            from q in db.Contracts
            let isActive = !q.TerminationEffectiveDate.HasValue || q.TerminationEffectiveDate.Value > today
            where q.Product.CategoryId == (int)Enums.ProductCategory.CrudeOil || q.Product.CategoryId == (int)Enums.ProductCategory.NaturalGasAndLng
            orderby q.EffectiveDate descending, q.ContractNum
            select new ContractInfo
            {
                ContractText = isActive ? q.ContractNum : "{Inactive} " + q.ContractNum,
                ContractNumber = q.ContractNum,
                CounterpartyId = q.Counterparty == null ? 0 : q.Counterparty.Id,
                InternalEntityId = q.InternalEntity == null ? 0 : q.InternalEntity.Id,
                IsActive = isActive
            }
        ).ToList();

        return items;
    }

    public static List<IdName> GetVolumeTypes()
    {
        var db = Main.CreateContext();

        var items = (
            from q in db.VolumeTypes
            where q.Id != (int)Enums.VolumeType.Other
            orderby q.Name
            select new IdName(q.Id, q.Name)
        ).ToList();

        return items;
    }

    public static List<IdName> GetFuelCalculationTypes()
    {
        var db = Main.CreateContext();

        var items = (
            from q in db.FuelCalculationTypes
            orderby q.Name
            select new IdName(q.Id, q.Name)
        ).ToList();

        return items;
    }

    public class ForceMajeureInfo
    {
        public string Label = "";
        public string Language = "";
    }

    public static List<ForceMajeureInfo> GetForceMajeures()
    {
        var db = Main.CreateContext();

        var items = (
            from q in db.ForceMajeures
            orderby q.Label
            select new ForceMajeureInfo
            {
                Label = q.Label,
                Language = q.Language ?? ""
            }
        ).ToList();

        return items;
    }

    public static int GetDealModifyTimeLag()
    {
        var db = Main.CreateContext();
        var dealModifyTimeLagStr = (from q in db.AppSettings where q.Name == "DealModifyTimeLag" select q.Value).FirstOrDefault();
        var dealModifyTimeLag = string.IsNullOrWhiteSpace(dealModifyTimeLagStr) ? 0 : int.Parse(dealModifyTimeLagStr);
        return dealModifyTimeLag;
    }

    public static string GetUpdateNotes(DateTime created, string createdName, DateTime modified, string modifiedName)
    {
        var createdStr = created.ToString(@"MMM d yyyy \a\t h:mm tt");
        var modifiedStr = modified.ToString(@"MMM d yyyy \a\t h:mm tt");
        string updateNotes = $"Created {createdStr}\r\nBy {createdName}\r\n\r\nModified {modifiedStr}\r\nBy {modifiedName}";
        return updateNotes;
    }

    public static List<IdName> GetTransactionTypes()
    {
        var db = Main.CreateContext();
        var transactionTypeIds = new List<int>() { 1, 3, 9, 11 };

        var items = (
            from q in db.TransactionTypes
            where transactionTypeIds.Contains(q.Id)
            orderby q.Id
            select new IdName(q.Id, q.Name ?? "")
        ).ToList();

        return items;
    }

    public static async Task<Deal> GetCopiedDealAsync(Deal dealToCopy, MyDbContext db)
    {
        //the db context must be passed in since this is often called in a transaction

        var time = DateTime.UtcNow;
        var newTicketNum = await Util.GetNewDealNumAsync((int)Enums.TransactionType.PhysicalGas, db);

        var n = new Deal();
        var o = dealToCopy;
        n.TicketNum = newTicketNum;
        n.CreatedBy = o.CreatedBy;
        n.Created = time;
        n.TraderId = o.TraderId;
        n.TradingDate = o.TradingDate;
        n.ProductId = o.ProductId;
        n.AccountingMonth = o.AccountingMonth;
        n.Basis = o.Basis;
        n.BookId = o.BookId;
        n.BrokerAccountId = o.BrokerAccountId;
        n.BrokerId = o.BrokerId;
        n.BuyButton = o.BuyButton;
        n.Comments = null;
        n.ContactId = o.ContactId;
        n.NumOfContracts = o.NumOfContracts;
        n.CostBasisInc = o.CostBasisInc;
        n.CostFuelCalculationTypeId = o.CostFuelCalculationTypeId;
        n.CostPremInc = o.CostPremInc;
        n.CounterpartyId = o.CounterpartyId;
        n.DealPurposeId = o.DealPurposeId;
        n.DealStatusId = o.DealStatusId;
        n.EndDate = o.EndDate;
        n.FixedPrice = o.FixedPrice;
        n.FixedPriceButton = o.FixedPriceButton;
        n.FmLanguage = o.FmLanguage;
        n.HedgeFee = o.HedgeFee;
        n.HypotheticalId = o.HypotheticalId;
        n.InternalEntityId = o.InternalEntityId;
        n.InternalMemo = null;
        n.IsVariableVolume = o.IsVariableVolume;
        n.NettingContractNumber = o.NettingContractNumber;
        n.PhysicalDealTypeId = o.PhysicalDealTypeId;
        n.PipelineId = o.PipelineId;
        n.PipelineSourceDeliveryId = o.PipelineSourceDeliveryId;
        n.PointId = o.PointId;
        n.PortfolioId = o.PortfolioId;
        n.PremiumOrDiscount = o.PremiumOrDiscount;
        n.PriceIndexId = o.PriceIndexId;
        n.RegionId = o.RegionId;
        n.StartDate = o.StartDate;
        n.Strategy = o.Strategy;
        n.TransactionTypeId = o.TransactionTypeId;
        n.Volume = o.Volume;
        n.VolumeTypeId = o.VolumeTypeId;
        n.WaspNum = o.WaspNum;
        n.VolumeC2 = o.VolumeC2;
        n.VolumeC3 = o.VolumeC3;
        n.VolumeIc4 = o.VolumeIc4;
        n.VolumeNc4 = o.VolumeNc4;
        n.VolumeC5P = o.VolumeC5P;
        n.FixedPriceC2 = o.FixedPriceC2;
        n.FixedPriceC3 = o.FixedPriceC3;
        n.FixedPriceIc4 = o.FixedPriceIc4;
        n.FixedPriceNc4 = o.FixedPriceNc4;
        n.FixedPriceC5P = o.FixedPriceC5P;
        n.PriceIndexIdC2 = o.PriceIndexIdC2;
        n.PriceIndexIdC3 = o.PriceIndexIdC3;
        n.PriceIndexIdIc4 = o.PriceIndexIdIc4;
        n.PriceIndexIdNc4 = o.PriceIndexIdNc4;
        n.PriceIndexIdC5P = o.PriceIndexIdC5P;
        n.BasisC2 = o.BasisC2;
        n.BasisC3 = o.BasisC3;
        n.BasisIc4 = o.BasisIc4;
        n.BasisNc4 = o.BasisNc4;
        n.BasisC5P = o.BasisC5P;
        n.PremDiscC2 = o.PremDiscC2;
        n.PremDiscC3 = o.PremDiscC3;
        n.PremDiscIc4 = o.PremDiscIc4;
        n.PremDiscNc4 = o.PremDiscNc4;
        n.PremDiscC5P = o.PremDiscC5P;
        n.InSpecMarketingFee = o.InSpecMarketingFee;
        n.InSpecMarketingFeeTypeId = o.InSpecMarketingFeeTypeId;
        n.OutSpecMarketingFee = o.OutSpecMarketingFee;
        n.OutSpecMarketingFeeTypeId = o.OutSpecMarketingFeeTypeId;
        n.SuperiorFee = o.SuperiorFee;
        n.SuperiorFeeTypeId = o.SuperiorFeeTypeId;
        n.ModifiedBy = o.ModifiedBy;
        n.Modified = time;

        var createdName = (from q in db.AppUsers where q.Id == n.CreatedBy select q.DisplayName).First();
        var modifiedName = (from q in db.AppUsers where q.Id == n.ModifiedBy select q.DisplayName).First();
        n.UpdateNotes = GetUpdateNotes(n.Created.Value, createdName, n.Modified.Value, modifiedName);

        if (o.PointSourceDeliveries != null)
        {
            foreach (var pointSourceDelivery in o.PointSourceDeliveries)
                n.PointSourceDeliveries.Add(new PointSourceDelivery { PointId = pointSourceDelivery.PointId });
        }

        if (o.IsVariableVolume && o.DealVolumes != null)
        {
            foreach (var dealVolume in o.DealVolumes)
            {
                var dv = new DealVolume
                {
                    StartDate = dealVolume.StartDate,
                    EndDate = dealVolume.EndDate,
                    PhysicalVolume = dealVolume.PhysicalVolume
                };
                n.DealVolumes.Add(dv);
            }
        }

        return n;
    }

    public static string GetVolUnitText(int volumeTypeId)
    {
        if (volumeTypeId == (int)Enums.VolumeType.Daily)
            return "MMBTU/Day";
        else if (volumeTypeId == (int)Enums.VolumeType.Monthly)
            return "MMBTU/Month";
        else if (volumeTypeId == (int)Enums.VolumeType.Total)
            return "MMBTU Total";
        else
            return "MMBTU";
    }
}
