﻿using System.Data;
using System.Linq.Expressions;
using static Fast.Shared.Models.Enums;
using Fast.Models;

namespace Fast.Logic;

public class DealFilterHelper
{
    public readonly MyDbContext db;
    public readonly int filterId;
    public Dictionary<string, DealFilterParameter> filterParamsDic;

    public DealFilterHelper(MyDbContext context, int filterId)
    {
        db = context;
        this.filterId = filterId;
        filterParamsDic = GetFilterParamsDic();
    }

    private Dictionary<string, DealFilterParameter> GetFilterParamsDic()
    {
        var filterParamsDic = db.DealFilterParameters.Where(x => x.FilterId == filterId).ToDictionary(x => x.Name, StringComparer.OrdinalIgnoreCase);
        return filterParamsDic;
    }

    public static void AddDefaultFiltersIfNeeded(MyDbContext db, int userId)
    {
        var filters = (from q in db.DealFilters where q.UserId == userId select q).ToList();
        var hasFilters = filters.Count != 0;
        var hasDefaultView = hasFilters && filters.Where(x => x.Name == "Default View").Any();
        var hasDistributionView = hasFilters && filters.Where(x => x.Name == "Distribution View").Any();

        if (!hasDefaultView)
            AddDefaultView(db, userId);

        if (!hasDistributionView)
            AddDistributionView(db, userId);

        //re-retrieve filters since collection may have been added to after adding defaults
        filters = (from q in db.DealFilters where q.UserId == userId select q).ToList();
        if (!filters.Where(x => x.IsSelected).Any())
        {
            filters.Where(x => x.Name == "Default View").First().IsSelected = true;
            db.SaveChanges();
        }
    }

    private static void AddDefaultView(MyDbContext db, int userId)
    {
        DealFilter defaultView = new()
        {
            Name = "Default View",
            IsSelected = true,
            UserId = userId,
            Columns = "DealNum,BuySell,Counterparty,Contact,Pipeline,Point,Volume,DealType,FixedPrice,Basis,PriceIndex,PremDisc,StartDate,EndDate",
            State = GetDefaultStateJson()
        };
        db.DealFilters.Add(defaultView);
        db.SaveChanges();
    }

    private static void AddDistributionView(MyDbContext db, int userId)
    {
        DealFilter defaultView = new()
        {
            Name = "Distribution View",
            IsSelected = false,
            UserId = userId,
            Columns = "DealNum,BuySell,Counterparty,SendEmail,EmailAddress,EmailDate,EmailedBy,SendFax,FaxNumber,FaxDate,FaxedBy,Volume,DealType,FixedPrice,Basis,PriceIndex,PremDisc,StartDate,EndDate",
            State = GetDefaultStateJson()
        };
        db.DealFilters.Add(defaultView);
        db.SaveChanges();
    }

    public static string GetDefaultStateJson()
    {
        return @"{""filter"":null,""group"":null,""skip"":0,""sort"":[{""field"":""DealNum"",""dir"":""desc""}],""take"":100}";
    }

    public List<int> GetIdList(string paramName)
    {
        List<int> idList = new();
        if (filterParamsDic.TryGetValue(paramName, out var dealFilterParam))
        {
            string value1 = dealFilterParam.Value1;
            idList = Util.String.ConvertCommaSeparatedIdsToList(value1);
        }
        return idList;
    }

    public int GetBuySellValue(string paramName)
    {
        string paramValue = filterParamsDic[paramName].Value1;
        int result = paramValue == "Buy Deals" ? 1 : -1;
        return result;
    }

    public int? GetHypotheticalValue(string paramName)
    {
        int? result;
        if (!HasParam(paramName) || filterParamsDic[paramName].Value1 == "Real Deals")
            result = 0; //real deals
        else if (filterParamsDic[paramName].Value1 == "Hypothetical Deals")
            result = 1; //hypothetical deals
        else if (filterParamsDic[paramName].Value1 == "All Deals")
            result = null; //all deals
        else
            throw new Exception("Unknown filter selection for 'Hypothetical'");
        return result;
    }

    public bool HasParam(string paramName)
    {
        return filterParamsDic.ContainsKey(paramName);
    }

    public DateOnly GetDate(string paramName)
    {
        DateOnly result = DateOnly.ParseExact(filterParamsDic[paramName].Value1, "yyyy/MM/dd", null);
        return result;
    }

    public List<DateRange> GetDateRanges(DateStyle dateStyle, string paramName)
    {
        List<DateRange> dateRanges = new();
        var dateRangeStrings = filterParamsDic[paramName].Value1.Split(",").ToList();
        foreach (string dateRangeString in dateRangeStrings)
        {
            DateRange newDateRange = new(dateStyle);
            DateOnly date1;
            DateOnly date2;
            var dateRangeSplit = dateRangeString.Split("-");
            date1 = DateOnly.ParseExact(dateRangeSplit[0], "yyyy/MM/dd", null);
            if (dateRangeSplit.Length > 1)
                date2 = DateOnly.ParseExact(dateRangeSplit[1], "yyyy/MM/dd", null);
            else
                date2 = date1;
            if (date1 <= date2)
            {
                newDateRange.FromDate = date1;
                newDateRange.ToDate = date2;
            }
            else
            {
                newDateRange.FromDate = date2;
                newDateRange.ToDate = date1;
            }
            dateRanges.Add(newDateRange);
        }
        return dateRanges;
    }

    public Expression<Func<T, bool>> GetDateRangeExpression<T>(string filterParamName, string dbDatePropertyName, bool allowNulls)
    {
        var dateRanges = GetDateRanges(DateStyle.Unknown, filterParamName);
        var exp = Util.Date.GetDateRangeExpression<T>(dateRanges, dbDatePropertyName, allowNulls);
        return exp;
    }

    public Expression<Func<T, bool>> GetDateRangeExpression<T>(string filterParamName, string dbFromDatePropertyName, string dbToDatePropertyName, bool allowNulls)
    {
        var dateRanges = GetDateRanges(DateStyle.Unknown, filterParamName);
        var exp = Util.Date.GetDateRangeExpression<T>(dateRanges, dbFromDatePropertyName, dbToDatePropertyName, allowNulls);
        return exp;
    }

    [System.Diagnostics.CodeAnalysis.SuppressMessage("Style", "IDE0060:Remove unused parameter", Justification = "<Pending>")]
    public async Task<IQueryable<DealListItem>> ApplyDealFilters(IQueryable<DealListItem> originalQuery, bool hasDealNumFilter)
    {
        //, bool hasDealNumFilter
        //IQueryable<DealListItem> query;
        //if (hasDealNumFilter)
        //    query = originalQuery.DeepClone();
        //else
        //    query = originalQuery;

        IQueryable<DealListItem> query;
        query = originalQuery;

        string paramName;

        paramName = "Book";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.BookId.HasValue && idList.Contains(x.BookId.Value));
        }

        paramName = "Buy/Sell";
        if (HasParam(paramName))
        {
            var buySellValue = GetBuySellValue(paramName);
            query = query.Where(x => x.BuyButton.HasValue && x.BuyButton.Value == buySellValue);
        }

        paramName = "Broker";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.BrokerId.HasValue && idList.Contains(x.BrokerId.Value));
        }

        paramName = "Broker Account";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.BrokerAccountId.HasValue && idList.Contains(x.BrokerAccountId.Value));
        }

        paramName = "Counterparty";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.CounterpartyId.HasValue && idList.Contains(x.CounterpartyId.Value));
        }

        paramName = "Deal Purpose";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.DealPurposeId.HasValue && idList.Contains(x.DealPurposeId.Value));
        }

        paramName = "Deal Status";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => idList.Contains(x.DealStatusId));
        }

        paramName = "Deal Type";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.DealTypeId.HasValue && idList.Contains(x.DealTypeId.Value));
        }

        paramName = "Hypothetical";
        int? hypotheticalId = GetHypotheticalValue(paramName);
        if (hypotheticalId.HasValue)
        {
            query = query.Where(x => x.HypotheticalId == hypotheticalId);
        }

        paramName = "Internal Entity";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.InternalEntityId.HasValue && idList.Contains(x.InternalEntityId.Value));
        }

        paramName = "Pipeline";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.PipelineId.HasValue && (idList.Contains(x.PipelineId.Value) || idList.Contains(x.PipelineSourceDeliveryId ?? 0)));
        }

        paramName = "Point/Hub";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.PointId.HasValue && idList.Contains(x.PointId.Value));
        }

        paramName = "Portfolio";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.PortfolioId.HasValue && idList.Contains(x.PortfolioId.Value));
        }

        paramName = "Price Index";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.PriceIndexId.HasValue && idList.Contains(x.PriceIndexId.Value));
        }

        paramName = "Product";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => idList.Contains(x.ProductId));
        }

        paramName = "Region";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.RegionId.HasValue && idList.Contains(x.RegionId.Value));
        }

        paramName = "Strategy";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.StrategyId.HasValue && idList.Contains(x.StrategyId.Value));
        }

        paramName = "Trader";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var idList = GetIdList(paramName);
            query = query.Where(x => x.TraderId.HasValue && idList.Contains(x.TraderId.Value));
        }

        paramName = "Transaction Type";
        if (HasParam(paramName))
        {
            await FixInvalidFilterParams(paramName);
            var savedIds = GetIdList(paramName);
            query = query.Where(x => x.TransactionTypeId.HasValue && savedIds.Contains(x.TransactionTypeId.Value));
        }

        paramName = "Postion Date";
        if (HasParam(paramName))
        {
            var positionDateExp = GetDateRangeExpression<DealListItem>(paramName, "StartDate", "EndDate", false);
            query = query.Where(positionDateExp);
        }

        paramName = "Trade Date";
        if (HasParam(paramName))
        {
            var exp = GetDateRangeExpression<DealListItem>(paramName, "TradeDate", true);
            query = query.Where(exp);
        }

        return query;
    }

    private async Task FixInvalidFilterParams(string paramName)
    {
        var hasChanges = false;
        HashSet<int>? allIds = paramName switch
        {
            "Book" => await db.Books.Select(x => x.Id).ToHashSetAsync(),
            "Broker" => await db.Brokers.Select(x => x.Id).ToHashSetAsync(),
            "Broker Account" => await db.BrokerAccounts.Select(x => x.Id).ToHashSetAsync(),
            "Counterparty" => await db.Counterparties.Select(x => x.Id).ToHashSetAsync(),
            "Deal Purpose" => await db.DealPurposeTypes.Select(x => x.Id).ToHashSetAsync(),
            "Deal Status" => await db.DealStatuses.Select(x => x.Id).ToHashSetAsync(),
            "Deal Type" => await db.PhysicalDealTypes.Select(x => x.Id).ToHashSetAsync(),
            "Internal Entity" => await db.Counterparties.Select(x => x.Id).ToHashSetAsync(),
            "Pipeline" => await db.Pipelines.Select(x => x.Id).ToHashSetAsync(),
            "Point/Hub" => await db.Points.Select(x => x.Id).ToHashSetAsync(),
            "Portfolio" => await db.Portfolios.Select(x => x.Id).ToHashSetAsync(),
            "Price Index" => await db.MarketIndices.Select(x => x.Id).ToHashSetAsync(),
            "Product" => await db.Products.Select(x => x.Id).ToHashSetAsync(),
            "Region" => await db.Regions.Select(x => x.Id).ToHashSetAsync(),
            "Strategy" => await db.Strategies.Select(x => x.Id).ToHashSetAsync(),
            "Trader" => await db.AppUsers.Select(x => x.Id).ToHashSetAsync(),
            "Transaction Type" => await db.TransactionTypes.Select(x => x.Id).ToHashSetAsync(),
            _ => throw new ArgumentOutOfRangeException(paramName, $"`{paramName}` filter is not supported."),
        };

        var savedIdStrings = await db.DealFilterParameters.Where(x => x.Name == paramName).Select(x => x.Value1).Distinct().ToListAsync();
        var savedIds = savedIdStrings.SelectMany(x => x.Split(',', StringSplitOptions.TrimEntries)).Select(x => int.Parse(x)).Distinct().ToList();
        if (allIds == null || allIds.Count == 0 || savedIds.Count == 0)
            return;

        foreach (var savedId in savedIds)
        {
            if (allIds.Contains(savedId))
                continue;

            //if savedId is not in allIds, then remove it by updating the comma separated list in Value1 of DealFilterParameter, to a new Value1 without the invalid savedId
            var possibleInvalidFilterParams = await db.DealFilterParameters.Where(x => x.Name == paramName && x.Value1.Contains(savedId.ToString())).ToListAsync();
            var invalidDealFilterParams = possibleInvalidFilterParams.Where(x => x.Value1.Split(',', StringSplitOptions.TrimEntries).Select(x => int.Parse(x)).Contains(savedId));
            foreach (var invalidDealFilterParameter in invalidDealFilterParams)
            {
                var newValue1 = string.Join(",", invalidDealFilterParameter.Value1.Split(',', StringSplitOptions.TrimEntries).Select(x => int.Parse(x)).Where(x => x != savedId).ToList());
                if (string.IsNullOrEmpty(newValue1))
                    db.DealFilterParameters.Remove(invalidDealFilterParameter);
                else
                {
                    invalidDealFilterParameter.Value1 = newValue1;
                    invalidDealFilterParameter.Preview = "Filter needs re-save";
                }
                hasChanges = true;
            }
        }
        await db.SaveChangesAsync();
        if (hasChanges)
            filterParamsDic = GetFilterParamsDic();
    }
}
