﻿using System.Text;
using DocumentFormat.OpenXml.Wordprocessing;
using Fast.Shared.Logic.FileService;
using Fast.Shared.Logic.Package;

namespace Fast.Logic;

public class ContractDocExB(IFileService fileService, string filesFolderName, string templatesFolderName, string signaturesFolderName, MyDbContext db)
    : DocBaseOpenXml(fileService, filesFolderName, templatesFolderName, signaturesFolderName, string.Empty, true)
{
    protected override string TemplateFileName => "NAESBExhibitB.docx";

    public async Task Generate(int mainContractId)
    {
        //get the deal from the database without tracking so that we don't modify data in the database
        var contract = await (
            from q in db.Contracts
                .Include(x => x.ContractExhibitBSelections)
                .ThenInclude(x => x.ContractExhibitB)
                .ThenInclude(x => x.Parent)
            where q.Id == mainContractId
            select new
            {
                q.ContractName,
                q.ContractNum,
                q.EffectiveDate,
                q.ExecutionDate,
                AmendmentDate = q.ExhibitBAmendmentDate,
                ContractSignerName = q.Signer == null ? "" : q.Signer.LegalName,
                ContractSignerTitle = q.Signer == null ? "" : q.Signer.Title,
                InternalName = q.InternalEntity.Name,
                CounterpartyName = q.Counterparty.Name,
                ExhibitBSelections = q.ContractExhibitBSelections
            }
        ).AsNoTracking().FirstAsync();

        if (!contract.ExhibitBSelections.Any())
            return; //don't generate the document

        var c = contract;
        SetText("A-Name", c.InternalName);
        SetText("B-Name", c.CounterpartyName);
        SetText("EffectiveDateShort", c.EffectiveDate?.ToShortDateString() ?? "");
        SetText("EffectiveDateLong", c.EffectiveDate?.ToString(ContractHelper.dateLongFormat) ?? "");
        SetText("ExecutionDateShort", c.EffectiveDate?.ToShortDateString() ?? "");
        SetText("ExecutionDateLong", c.EffectiveDate?.ToString(ContractHelper.dateLongFormat) ?? "");
        SetText("LastAmendmentDateShort", c.AmendmentDate?.ToShortDateString() ?? "");
        SetText("LastAmendmentDateLong", c.AmendmentDate?.ToString(ContractHelper.dateLongFormat) ?? "");
        SetText("ContractSignerName", c.ContractSignerName);
        SetText("ContractSignerTitle", c.ContractSignerTitle);
        SetText("A-ContractNumberA", c.ContractNum);

        HashSet<int> uniqueParentIds = new();

        //order by the parent ordinal first if we have one or we are the parent, then order by the child ordinal
        var selectedExhibitBItems = (
            from q in c.ExhibitBSelections
            orderby
                q.ContractExhibitB.Parent != null ? q.ContractExhibitB.Parent.Ordinal : q.ContractExhibitB.Ordinal,
                q.ContractExhibitB.Ordinal
            select q.ContractExhibitB
        ).ToList();

        if (docRunsByKey.ContainsKey("InsertedText"))
        {
            SetText("InsertedText", "");
            var firstRun = docRunsByKey["InsertedText"].First();
            var paraPrev = firstRun.Ancestors<Paragraph>().FirstOrDefault();
            var docBody = paraPrev?.Parent;
            foreach (var exhibitBItem in selectedExhibitBItems)
            {
                var parent = exhibitBItem.ParentId == null ? exhibitBItem : exhibitBItem.Parent;
                var child = exhibitBItem.ParentId == null ? null : exhibitBItem;
                if (paraPrev != null && parent != null && !uniqueParentIds.Contains(parent.Id))
                {
                    uniqueParentIds.Add(parent.Id);
                    List<string> bodyByLine = parent.Body?.Split("\n").ToList() ?? new();
                    foreach (var line in bodyByLine)
                    {
                        var paraNew = new Paragraph();

                        var text = line.TrimEnd();
                        var textElement = new Text(text);
                        var run = new Run();
                        run.Append(textElement);
                        var runProperties = new RunProperties();

                        var firstRunProps = firstRun.RunProperties;
                        if (firstRunProps != null)
                        {
                            if (firstRunProps.RunFonts != null)
                            {
                                runProperties.Append(new RunFonts() { Ascii = firstRunProps.RunFonts.Ascii });
                            }

                            if (firstRunProps.FontSize != null)
                            {
                                runProperties.Append(new FontSize() { Val = firstRunProps.FontSize.Val });
                            }
                        }

                        run.PrependChild(runProperties);
                        paraNew.Append(run);
                        paraPrev.InsertAfterSelf(paraNew);
                        paraPrev = paraNew;
                    }
                }

                if (paraPrev != null && child != null)
                {
                    child.Body = child.Body + "\n";
                    List<string> bodyByLine = child.Body?.Split("\n").ToList() ?? new();
                    foreach (var line in bodyByLine)
                    {
                        var paraNew = new Paragraph();
                        paraNew.ParagraphProperties = new ParagraphProperties();
                        paraNew.ParagraphProperties.Indentation = new Indentation() { Left = "20" };

                        var text = line.TrimEnd();
                        var textElement = new Text(text);
                        var run = new Run();
                        run.Append(textElement);
                        var runProperties = new RunProperties();

                        var firstRunProps = firstRun.RunProperties;
                        if (firstRunProps != null)
                        {
                            if (firstRunProps.RunFonts != null)
                            {
                                runProperties.Append(new RunFonts() { Ascii = firstRunProps.RunFonts.Ascii });
                            }

                            if (firstRunProps.FontSize != null)
                            {
                                runProperties.Append(new FontSize() { Val = firstRunProps.FontSize.Val });
                            }
                        }

                        run.PrependChild(runProperties);
                        paraNew.Append(run);
                        paraPrev.InsertAfterSelf(paraNew);
                        paraPrev = paraNew;
                    }
                }
            }
        }

        string fileNameOriginal = $"{c.ContractName}-{c.ContractNum}-ExhibitB.docx";
        fileNameOriginal = Util.String.GetLegalFileName(fileNameOriginal);
        string fileNameOnDisk = Util.String.GetNewGuid() + ".docx";

        wordprocessingDocument.PackageProperties.Title = fileNameOriginal;

        // Use SaveToMemoryStream and write via fileService - sharding is handled internally
        var docStream = SaveToMemoryStream();
        var docBytes = docStream.ToArray();
        var uploadedFileName = await fileService.WriteAllBytesAsync(filesFolderName, fileNameOnDisk, docBytes);

        if (!string.IsNullOrWhiteSpace(uploadedFileName))
        {
            var newDoc = new ContractDoc
            {
                ContractId = mainContractId,
                FileNameOriginal = fileNameOriginal,
                FileNameOnDisk = uploadedFileName,
                InactiveDate = null
            };
            db.ContractDocs.Add(newDoc);
            await db.SaveChangesAsync();
        }
    }

    protected override HashSet<string> GetDocKeys()
    {
        return ContractHelper.GetDocKeys();
    }
}
