using System.Runtime.Serialization.Json;
using System.Text.RegularExpressions;
using Fast.Shared.Logic.FileService;
using HtmlAgilityPack;
using Microsoft.AspNetCore.OData.Formatter;
using Fast.Models;
using Microsoft.AspNetCore.OData.Routing.Controllers;

namespace Fast.Web.Controllers
{
    [Authorize]
    [ApiController]
    [Route("api/[controller]/[action]")]
    public class PaymentInstructionsController : ODataController
    {
        public enum SaveType
        {
            New = 1,
            Normal = 2
        }

        private readonly AuthorizationHelper authHelper;
        private readonly MyDbContext db;
        private readonly IFileService fileService;
        private readonly string uploadFolderName = "PaymentInstructions";

        public PaymentInstructionsController(MyDbContext context, IWebHostEnvironment env)
        {
            db = context;
            authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);

            var config = new FileServiceConfig(env.ContentRootPath);
            fileService = new FileService(config);
        }

        [Permission("Payment Instructions", PermissionType.View)]
        [Route("/odata/GetPaymentInstructionsItems")]
        public IActionResult GetItems(ODataQueryOptions<PaymentInstructionsListItem> queryOptions, bool isExport, bool showInactive)
        {
            queryOptions = Util.GetQueryOptionsWithConvertedDates(queryOptions);
            var itemsQueryable = GetItemsInternal(showInactive);
            var items = (queryOptions.ApplyTo(itemsQueryable) as IEnumerable<PaymentInstructionsListItem>)?.ToList();

            if (isExport)
            {
                return File(Util.Excel.GetExportFileStream(items), "application/octet-stream");
            }

            return Ok(items);
        }

        private IQueryable<PaymentInstructionsListItem> GetItemsInternal(bool showInactive)
        {
            //get max effective dates for each counterparty
            //this is so that we have at least one of the most recent index prices for each index
            var maxDates = from q in db.PaymentInstructions
                           group q by new { q.CounterpartyId } into g
                           select new { g.Key.CounterpartyId, MaxEffectiveDate = g.Max(x => x.EffectiveDate) };

            IQueryable<PaymentInstruction> itemsQueryable;

            if (!showInactive)
            {
                itemsQueryable = from q in db.PaymentInstructions
                                 join md in maxDates on new { q.CounterpartyId, q.EffectiveDate } equals new { md.CounterpartyId, EffectiveDate = md.MaxEffectiveDate }
                                 select q;
            }
            else
            {
                itemsQueryable = from q in db.PaymentInstructions
                                 select q;
            }

            var finalQueryable = from q in itemsQueryable
                                 join oi in db.VwPaymentInstructionOverviewInfos on q.Id equals oi.PaymentInstructionId into j1
                                 from oi in j1.DefaultIfEmpty()
                                 select new PaymentInstructionsListItem
                                 {
                                     PaymentInstructionsId = q.Id,
                                     Counterparty = q.Counterparty.Name,
                                     AccountHolder = q.AccountHolder,
                                     EffectiveDate = q.EffectiveDate,
                                     BankName = q.BankName,
                                     Relationships = oi.Relationships
                                 };

            return finalQueryable;
        }

        [Permission("Payment Instructions", PermissionType.View)]
        public async Task<IActionResult> GetRequiredData()
        {
            var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Payment Instructions", PermissionType.Modify);

            var counterparties = await GetCounterpartiesAsync(false);

            var states = await DataHelper.GetTerritoriesAsync();

            var result = new { hasModifyPermission, counterparties, states };
            return Ok(result);
        }

        public static async Task<List<CounterpartyItem>> GetCounterpartiesAsync(bool useShortNames)
        {
            var db = Main.CreateContext();
            var currentDate = DateOnly.FromDateTime(DateTime.Today);

            var activeCounterparties = await (
                from q in db.Counterparties
                where q.InactiveDate == null || q.InactiveDate.Value > currentDate
                orderby q.Name
                select new CounterpartyItem
                {
                    Id = q.Id,
                    Name = useShortNames && !string.IsNullOrWhiteSpace(q.ShortName) ? q.ShortName : q.Name,
                    CustomerNum = q.InternalCustomerNum ?? "None",
                    VendorNum = q.InternalVendorNum ?? "None"
                }
            ).ToListAsync();

            var inactiveCounterparties = await (
                from q in db.Counterparties
                where q.InactiveDate != null && q.InactiveDate <= currentDate
                orderby q.Name
                select new CounterpartyItem
                {
                    Id = q.Id,
                    Name = "{Inactive} " +
                           (useShortNames && !string.IsNullOrWhiteSpace(q.ShortName) ? q.ShortName : q.Name),
                    CustomerNum = q.InternalCustomerNum ?? "None",
                    VendorNum = q.InternalVendorNum ?? "None"
                }
            ).ToListAsync();

            var counterparties = activeCounterparties.Concat(inactiveCounterparties).ToList();
            return counterparties;
        }

        [Permission("Payment Instructions", PermissionType.View)]
        public IActionResult GetDetail(int id)
        {
            var detail = (
                from q in db.PaymentInstructions
                where q.Id == id
                select new PaymentInstructionsDetail
                {
                    PaymentInstructionsId = q.Id,
                    CounterpartyId = q.CounterpartyId,
                    EffectiveDate = q.EffectiveDate,
                    BankName = q.BankName,
                    BankCity = q.BankCity,
                    StateId = q.BankStateId,
                    AccountHolder = q.AccountHolder,
                    BankWireNum = q.BankWireNum,
                    BankAchNum = q.BankAchNum,
                    AccountNum = q.AccountNum,
                    SwiftCode = q.SwiftCode,
                    FcTo = q.FurtherCreditTo,
                    FcAccountNum = q.FurtherCreditAccountNum,
                    Notes = q.Notes,
                    Documents = q.PaymentInstructionDocs.Select(x => new DocItem
                    { FileNameOriginal = x.FileNameOriginal, FileNameOnDisk = x.FileNameOnDisk }).ToList(),
                    AbaWireCity = q.BankWireCity,
                    AbaAchCity = q.BankAchCity,
                    AbaAchStateId = q.BankAchStateId,
                    AbaWireStateId = q.BankWireStateId,
                    Ccd = q.HasPaymentTypeCcd,
                    Ctx = q.HasPaymentTypeCtx,
                    Ppd = q.HasPaymentTypePpd,
                    Tax = q.HasPaymentTypeTax,
                    Wire = q.HasPaymentTypeWire
                }
            ).First();

            return Ok(detail);
        }

        [Permission("Payment Instructions", PermissionType.Modify)]
        public async Task<IActionResult> SaveDetail(PaymentInstructionsDetail detail, SaveType saveType)
        {
            var resultId = 0;

            await db.Database.CreateExecutionStrategy().Execute(async () =>
            {
                await using var dbContextTransaction = await db.Database.BeginTransactionAsync();
                PaymentInstruction? dbItem = null;
                if (saveType != SaveType.New)
                {
                    dbItem = await (
                        from q in db.PaymentInstructions
                            .Include(x => x.PaymentInstructionDocs)
                        where q.Id == detail.PaymentInstructionsId
                        select q
                    ).FirstOrDefaultAsync();
                }

                if (dbItem == null) //if the item does not exist then add it
                {
                    dbItem = new PaymentInstruction();
                    db.PaymentInstructions.Add(dbItem);
                }
                else
                {
                    //remove existing items so that they get completely re-inserted
                    db.PaymentInstructionDocs.RemoveRange(dbItem.PaymentInstructionDocs);
                }

                var d = detail;

                dbItem.CounterpartyId = d.CounterpartyId;
                dbItem.EffectiveDate = d.EffectiveDate;
                dbItem.BankName = d.BankName;
                dbItem.BankCity = d.BankCity;
                dbItem.BankStateId = d.StateId;
                dbItem.BankWireNum = string.IsNullOrWhiteSpace(d.BankWireNum) ? null : d.BankWireNum.Trim();
                dbItem.BankAchNum = string.IsNullOrWhiteSpace(d.BankAchNum) ? null : d.BankAchNum.Trim();
                dbItem.AccountHolder = d.AccountHolder;
                dbItem.AccountNum = string.IsNullOrWhiteSpace(d.AccountNum) ? null : d.AccountNum.Trim();
                dbItem.SwiftCode = d.SwiftCode;
                dbItem.FurtherCreditTo = d.FcTo;
                dbItem.FurtherCreditAccountNum = string.IsNullOrWhiteSpace(d.FcAccountNum) ? null : d.FcAccountNum.Trim();
                dbItem.Notes = d.Notes;
                dbItem.BankAchStateId = d.AbaAchStateId;
                dbItem.BankWireStateId = d.AbaWireStateId;
                dbItem.BankAchCity = d.AbaAchCity;
                dbItem.BankWireCity = d.AbaWireCity;
                dbItem.HasPaymentTypeWire = d.Wire;
                dbItem.HasPaymentTypeCcd = d.Ccd;
                dbItem.HasPaymentTypeCtx = d.Ctx;
                dbItem.HasPaymentTypePpd = d.Ppd;
                dbItem.HasPaymentTypeTax = d.Tax;

                var paymentDocs = d.Documents.Select(x => new PaymentInstructionDoc
                {
                    PaymentInstructionId = d.PaymentInstructionsId,
                    FileNameOriginal = x.FileNameOriginal,
                    FileNameOnDisk = x.FileNameOnDisk
                }).ToList();
                dbItem.PaymentInstructionDocs!.Clear();
                paymentDocs.ForEach(x => dbItem.PaymentInstructionDocs!.Add(x));

                await db.SaveChangesAsync();
                resultId = dbItem.Id;

                await dbContextTransaction.CommitAsync();
            });

            return Ok(resultId);
        }

        [Permission("Payment Instructions", PermissionType.Modify)]
        [Route("{id:int}")]
        public IActionResult DeleteDetail(int id)
        {
            var dbItem = db.PaymentInstructions.First(x => x.Id == id);
            db.PaymentInstructions.Remove(dbItem);
            db.SaveChanges();

            return Ok();
        }

        [Permission("Payment Instructions", PermissionType.Modify)]
        [Route("{id:int}")]
        public async Task<IActionResult> SendNotification(int id)
        {
            var dbItem = await (
                from q in db.PaymentInstructions
                where q.Id == id
                select new
                {
                    CpName = q.Counterparty.Name,
                    q.EffectiveDate
                }
            ).AsNoTracking().FirstAsync();

            var notification = db.Notifications.Where(x => x.Id == (int)Enums.NotificationType.PaymentInstructions).First();
            var cpName = dbItem.CpName;
            var effectiveDate = dbItem.EffectiveDate.ToString("MM/dd/yyyy");
            var currentUrl = HttpContext.Request.Host.Value;
            var detailUrl = $"https://{currentUrl}/#/PaymentInstructions?detail={id}";

            var subject = $"New payment instructions for {cpName}";
            var body = $"""
                New instructions for {cpName}<br>
                Effective date: {effectiveDate}<br>
                <a href='{detailUrl}'>{detailUrl}</a>
            """;

            var emailer = new Util.Email(db, Enums.NotificationType.PaymentInstructions);
            emailer.SendEmail(null, subject, body, null);

            return Ok();
        }

        [Permission("Payment Instructions", PermissionType.Modify)]
        public async Task<IActionResult> UploadDoc(IEnumerable<IFormFile> files, [FromODataUri] string metaData)
        {
            var docItems = new List<DocItem>();

            MemoryStream ms = new(Encoding.UTF8.GetBytes(metaData));
            var serializer = new DataContractJsonSerializer(typeof(ChunkMetaData));
            var chunkData = serializer.ReadObject(ms) as ChunkMetaData;
            string? fileNameOnDisk = null;

            if (files != null && chunkData != null)
            {
                foreach (var file in files)
                {
                    if (file.Length > 0)
                    {
                        // Generate a unique filename with GUID
                        var newFileName = Util.String.GetNewGuid() + Path.GetExtension(chunkData.FileName);
                        fileNameOnDisk = await fileService.UploadFileAsync(uploadFolderName, newFileName, file);
                    }
                }
            }

            if (fileNameOnDisk != null && chunkData != null && chunkData.ChunkIndex == chunkData.TotalChunks - 1)
            {
                docItems.Add(new DocItem
                {
                    FileNameOriginal = chunkData.FileName,
                    FileNameOnDisk = fileNameOnDisk
                });
                return Ok(docItems);
            }

            return Ok();
        }

        [Permission("Payment Instructions", PermissionType.View)]
        public async Task<IActionResult> DownloadDoc(string fileNameOnDisk)
        {
            try
            {
                var fileResponse = await fileService.DownloadFileAsync(uploadFolderName, fileNameOnDisk);

                return File(fileResponse.Stream, fileResponse.ContentType, fileResponse.FileName);
            }
            catch (Exception ex)
            {
                const string messagePrefix = "Upload Doc";
                var bytes = Util.GetExceptionFilesBytes(ex, messagePrefix);
                return File(bytes, "text/plain");
            }
        }


        [Permission("Payment Instructions", PermissionType.View)]
        public async Task<IActionResult> GetRoutingData(string routingNum)
        {
            var routingNumStr = routingNum.ToString();

            var routingData = await (
                from q in db.BankRoutings
                where q.RoutingNum == routingNumStr
                select q
            ).FirstOrDefaultAsync();

            var oneDayAgo = DateTime.Now.AddDays(-1);

            //Only get routing data from external source if we don't
            //have it yet or the data that we have is old (more than one day old)
            if (routingData == null || routingData.LastUpdated <= oneDayAgo)
            {
                try
                {
                    var urlStr = $"https://www.usbanklocations.com/crn.php?q={routingNum}";
                    HtmlWeb web = new();
                    var htmlDoc = web.Load(urlStr);

                    //We only update the BankRouting table if the user typed in a valid routing number that
                    //actually exists according to the external source
                    if (!htmlDoc.DocumentNode.OuterHtml.Contains("is not a valid bank routing number"))
                    {
                        //If a record for the routing # does not exist then create it
                        //Otherwise we update the fields of the existing record
                        if (routingData == null)
                        {
                            routingData = new();
                            db.BankRoutings.Add(routingData);
                        }

                        //Save the raw response from the external source to the BankRouting table
                        //This saves the record before we've done any parsing just in case the
                        //parsing of name, city, or state throws an error
                        routingData.RoutingNum = routingNumStr;
                        routingData.RawExternalResponse = htmlDoc.DocumentNode.OuterHtml;
                        //Set the time that we got the raw response from the external source
                        routingData.RawExternalUpdated = DateTime.UtcNow;
                        await db.SaveChangesAsync();

                        //This tracks whether or not parsing of the external bank name, city, or state
                        //was at least partially successful (we filled at least one of the fields)
                        bool isParsingSuccessful = false;

                        //Loop through all divs in the document that have class ublcrnsubheader
                        int tableNum = 1;
                        foreach (var div in htmlDoc.DocumentNode.SelectNodes("//div[@class='ublcrnsubheader']"))
                        {
                            //If there are more than 2 FedACH or Fedwire tables then
                            //ignore all tables after the first 2
                            if (tableNum > 2)
                                break;

                            if (div.InnerText == "FedACH Routing")
                            {
                                //We assume that the HTML table for ACH will come after
                                //the div with text FedACH Routing (we use tableNum to find it)
                                var achTableNode = htmlDoc.DocumentNode.SelectSingleNode($"//table[{tableNum}]");
                                if (achTableNode != null)
                                {
                                    //Get the data for the Name field from the HTML table (Bank Name)
                                    var name = GetTableNodeData(achTableNode, "Name");

                                    //Get the data for the City and State from the HTML table
                                    //they are parsed from either the Address or Location field
                                    var cityState = GetTableCityState(achTableNode);
                                    var city = cityState[0];
                                    var state = cityState[1];

                                    //Update the Bank Name (AchName) and City in the BankRouting table
                                    routingData.AchName = name?.ToUpper();
                                    routingData.AchCity = city;
                                    //Only update the State in the BankRouting table if
                                    //the parsed state abbreviation exists in the StateName table
                                    if (StatesDic.ContainsKey(state))
                                        routingData.AchStateId = StatesDic[state];
                                    tableNum++;
                                    isParsingSuccessful = true;
                                }
                            }

                            if (div.InnerText == "Fedwire Routing")
                            {
                                //We assume that the HTML table for Wire will come after
                                //the div with text Fedwire Routing (we use tableNum to find it)
                                var wireTableNode = htmlDoc.DocumentNode.SelectSingleNode($"//table[{tableNum}]");
                                if (wireTableNode != null)
                                {
                                    //Get the data for the Name field from the HTML table (Bank Name)
                                    var name = GetTableNodeData(wireTableNode, "Name");

                                    //Get the data for the City and State from the HTML table
                                    //they are parsed from either the Address or Location field
                                    var cityState = GetTableCityState(wireTableNode);
                                    var city = cityState[0];
                                    var state = cityState[1];

                                    //Update the Bank Name (WireName) and City in the BankRouting table
                                    routingData.WireName = name?.ToUpper();
                                    routingData.WireCity = city;
                                    //Only update the State in the BankRouting table if
                                    //the parsed state abbreviation exists in the StateName table
                                    if (StatesDic.ContainsKey(state))
                                        routingData.WireStateId = StatesDic[state];
                                    tableNum++;
                                    isParsingSuccessful = true;
                                }
                            }
                        }

                        if (isParsingSuccessful)
                        {
                            //Set the time that we got the raw response from the external source
                            routingData.LastUpdated = DateTime.UtcNow;
                            //Save the routingData that we parsed and filled from the external source
                            //to the BankRouting table
                            await db.SaveChangesAsync();
                        }
                    }
                }
                catch (Exception)
                {
                    //If an error occurs while attempting to retrieve data from the external source
                    //then swallow it.  If we already had it saved in the BankRouting table from a previous
                    //retrieval then it will return the old routingData from the table without updating it
                    //If an error occurs while attempting to retrieve data from the external source
                    //and we do not have it saved in the BankRouting table yet then routingData will be null and
                    //the user can type in the bank name, city, and state manually
                }
            }

            return Ok(routingData);
        }

        private Dictionary<string, int>? statesDic = null;
        private Dictionary<string, int> StatesDic
        {
            get
            {
                if (statesDic == null)
                    statesDic = (from q in db.Territories select new { q.Id, q.Abbreviation }).ToDictionary(x => x.Abbreviation, x => x.Id);
                return statesDic;
            }
        }

        private string? GetTableNodeData(HtmlNode tableNode, string fieldName)
        {
            string? result = null;
            foreach (var fieldNode in tableNode.SelectNodes(".//b"))
            {
                if (fieldNode.InnerText == fieldName + ":")
                {
                    var tableRow = fieldNode.ParentNode.ParentNode;
                    var dataTd = tableRow.SelectSingleNode(".//td[2]");

                    if (dataTd.ChildNodes.Count == 1 && !string.IsNullOrWhiteSpace(dataTd.InnerText))
                        result = dataTd.InnerText;
                    else
                    {
                        foreach (var dataNodeWithinTd in dataTd.SelectNodes(".//*"))
                        {
                            if ((fieldName == "Address" || fieldName == "Location") && StatesDic.Keys.Any(x => dataNodeWithinTd.InnerText.Contains(x)))
                                result = dataNodeWithinTd.InnerText;
                            else if (fieldName != "Address")
                                result = dataNodeWithinTd.InnerText;
                        }
                    }
                    //exit for since we already found the field that we were looking for
                    break;
                }
            }
            return result;
        }

        private string[] GetTableCityState(HtmlNode tableNode)
        {
            var result = new string[2] { "", "" };
            try
            {
                var address = GetTableNodeData(tableNode, "Address");
                if (string.IsNullOrWhiteSpace(address))
                    address = GetTableNodeData(tableNode, "Location");

                if (!string.IsNullOrWhiteSpace(address))
                    result = GetCityState(address);

            }
            catch (Exception)
            {
                //If we fail to parse the address or the city/state for any reason then simply
                //swallow the error and return a string array with two empty strings.  In that case
                //we will at least still update the bank name (I.E. AchName or WireName or both)
            }

            return result;
        }

        public string[] GetCityState(string address)
        {
            address = address.Replace(".", "");
            address = address.Replace(",", ", ");
            address = address.Replace("  ", " ");
            string[] result = new string[2] { "", "" };
            Regex addressPattern = new(@"(?<city>[A-Za-z',.\s]+) (?<state>([A-Z]{2}|[A-Z]{2},))");

            MatchCollection matches = addressPattern.Matches(address);

            if (matches.Any())
            {
                result[0] = matches[0].Groups["city"].Value.ReplaceLastOccurrence(",", "").Trim();
                result[1] = matches[0].Groups["state"].Value.ReplaceLastOccurrence(",", "").Trim();
            }

            return result;
        }
    }
}
