using System.Runtime.Serialization.Json;
using Fast.Logic;
using Fast.Models;
using Fast.Shared.Logic.FileService;
using Fast.Web.Models;
using Microsoft.AspNetCore.OData.Formatter;
using Microsoft.AspNetCore.OData.Routing.Controllers;

namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class MainContractController : ODataController
{
    public enum SaveType
    {
        New = 1,
        Normal = 2
    }

    public enum InfoType
    {
        Internal = 1,
        Counterparty = 2,
        Main = 3
    }

    private readonly MyDbContext db;
    private readonly AuthorizationHelper authHelper;
    private readonly IWebHostEnvironment env;
    private readonly IFileService fileService;
    private readonly string templatesFolderName = "ContractTemplates";
    private readonly string signaturesFolderName = "Signatures";
    private readonly string uploadFolderName = "ContractDocs";

    public MainContractController(MyDbContext context, IWebHostEnvironment env)
    {
        db = context;
        this.env = env;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);

        // Create FileService with configuration
        var config = new FileServiceConfig(env.ContentRootPath);
        fileService = new FileService(config);
    }

    [Permission("Main Contract", PermissionType.View)]
    [Route("/odata/GetMainContractItems")]
    public IActionResult GetItems(ODataQueryOptions<MainContractListItem> queryOptions, bool isExport)
    {
        queryOptions = Util.GetQueryOptionsWithConvertedDates(queryOptions);
        var itemsQueryable = GetItemsInternal();
        var items = (queryOptions.ApplyTo(itemsQueryable) as IEnumerable<MainContractListItem>)?.ToList();

        if (isExport)
        {
            return File(Util.Excel.GetExportFileStream(items), "application/octet-stream");
        }

        return Ok(items);
    }

    [Permission("Main Contract", PermissionType.View)]
    [Route("[action]")]
    public IQueryable<MainContractListItem> GetItemsInternal()
    {
        IQueryable<MainContractListItem>? itemsQueryable = null;

        var minDate = DateOnly.MinValue;
        var currentDate = DateTime.Today.ToUniversalTime();
        itemsQueryable = (
            from cptyC in db.Contracts
            join intC in db.Contracts on new { cptyC.InternalEntityId, cptyC.ProductId } equals new { intC.InternalEntityId, intC.ProductId } into intContracts
            from intC in intContracts.DefaultIfEmpty()
            where intC.IsInternalSideOfContract == true
            let isTerminated = cptyC.TerminationEffectiveDate != null && cptyC.TerminationEffectiveDate <= currentDate
            let willTerminate = cptyC.TerminationNoticeDate != null || cptyC.TerminationEffectiveDate != null
            let NaesbAmendDate = cptyC.ContractNaesb != null && cptyC.ContractNaesb!.AmendmentDate != null ? cptyC.ContractNaesb!.AmendmentDate : minDate
            let ExhibitBAmendDate = cptyC.ExhibitBAmendmentDate != null ? cptyC.ExhibitBAmendmentDate : minDate
            let ExhibitCAmendDate = cptyC.ContractExhibitC != null && cptyC.ContractExhibitC!.AmendmentDate != null ? cptyC.ContractExhibitC!.AmendmentDate : minDate
            let IntExhibitCAmendDate = intC.ContractExhibitC != null && intC.ContractExhibitC!.AmendmentDate != null ? intC.ContractExhibitC!.AmendmentDate : minDate
            where cptyC.IsInternalSideOfContract == false
            select new MainContractListItem
            {
                MainContractId = cptyC.Id,
                ContractName = cptyC.ContractName,
                ContractNum = cptyC.ContractNum,
                TheirContractNum = cptyC.TheirContractNum,
                InternalEntity = cptyC.InternalEntity.Name,
                Counterparty = cptyC.Counterparty.Name,
                Product = cptyC.Product.Name,
                Status = cptyC.Status.Name ?? "",
                EffectiveDate = cptyC.EffectiveDate,
                ExecutionDate = cptyC.ExecutionDate,
                LastAmendmentDate =
                    NaesbAmendDate > ExhibitBAmendDate && NaesbAmendDate > ExhibitCAmendDate && NaesbAmendDate > IntExhibitCAmendDate ? cptyC.ContractNaesb!.AmendmentDate :
                    ExhibitCAmendDate > ExhibitBAmendDate && ExhibitCAmendDate > IntExhibitCAmendDate ? cptyC.ContractExhibitC!.AmendmentDate :
                    IntExhibitCAmendDate > ExhibitBAmendDate ? intC.ContractExhibitC!.AmendmentDate :
                    cptyC.ExhibitBAmendmentDate,
                Termination = isTerminated ? "Terminated" : willTerminate ? "Will Terminate" : "Live"
            }
        ).AsNoTracking();

        return itemsQueryable;
    }

    [Permission("Main Contract", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData()
    {
        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Main Contract", PermissionType.Modify);

        var counterparties = await DataHelper.GetCounterpartiesAsync(false, Enums.ProductCategory.All);

        var statuses = await DataHelper.GetContractStatusesAsync();

        var products = await DataHelper.GetProductsAsync();

        var entities = await DataHelper.GetInternalEntitiesAsync(false);

        var states = await DataHelper.GetTerritoriesAsync();

        var countries = await (from q in db.Countries orderby q.Id == 1 descending, q.Name select new IdName(q.Id, q.Name)).ToListAsync();

        var signers = await (from q in db.AppUsers orderby q.DisplayName select new IdName(q.Id, q.DisplayName)).ToListAsync();

        var userId = Util.GetAppUserId(User);

        var payments = await (from q in db.PaymentDateTypes orderby q.Id select new IdName(q.Id, q.Name ?? "")).ToListAsync();

        var crudePoints = (await DataHelper.GetPointsAsync(true, Enums.ProductCategory.CrudeOil)).Select(x => new IdName(x.PointId, x.PointName)).ToList();

        var result = new { hasModifyPermission, counterparties, statuses, products, entities, states, countries, signers, userId, payments, crudePoints };
        return Ok(result);
    }

    private async Task<List<PayBankItem>> GetPayBankInfoAsync(int mainContractId)
    {
        var dbItem = await (
            from q in db.Contracts
            where q.Id == mainContractId
            select new
            {
                q.InternalEntityId,
                q.ProductId,
                q.Product.CategoryId,
                HasNaesbWirePay = q.ContractNaesb != null && q.ContractNaesb.WireTransferPayment,
                HasNaesbAchPay = q.ContractNaesb != null && q.ContractNaesb.AchPayment,
                HasCptyExcAddendum = q.ContractNaesb != null && q.ContractNaesb.AddendumExhibitC != false
            }
        ).AsNoTracking().FirstAsync();

        var dbItemInt = await (
            from q in db.Contracts
            where q.IsInternalSideOfContract
                && q.InternalEntityId == dbItem.InternalEntityId
                && q.Product.CategoryId == dbItem.CategoryId
            select new { exC = q.ContractExhibitC, acc = q.ContractAccounting }
        ).AsNoTracking().FirstAsync();

        var payBankItems = new List<PayBankItem>();
        var exCInt = dbItemInt.exC;
        var accInt = dbItemInt.acc;
        var isNatGas = dbItem.CategoryId == (int)Enums.ProductCategory.NaturalGasAndLng;

        //if not nat gas then assume all accounting and exhibit c options are allowed
        var hasExCWire = (isNatGas && dbItem.HasCptyExcAddendum && dbItem.HasNaesbWirePay && exCInt != null) || (!isNatGas && exCInt != null);
        var hasExCAch = (isNatGas && dbItem.HasCptyExcAddendum && dbItem.HasNaesbAchPay && exCInt != null) || (!isNatGas && exCInt != null);
        var hasAccWire = (isNatGas && !dbItem.HasCptyExcAddendum && dbItem.HasNaesbWirePay && accInt != null) || (!isNatGas && exCInt != null);
        var hasAccACH = (isNatGas && !dbItem.HasCptyExcAddendum && dbItem.HasNaesbAchPay && accInt != null) || (!isNatGas && exCInt != null);

        if (mainContractId == 0)
        {
            return new List<PayBankItem>();
        }

        if (hasExCWire && HasPayBankInfo(exCInt!.BankName, exCInt.AbaNumWire, exCInt.AccountNum))
        {
            var itemToAdd = CreatePayBankItem(Enums.ContractPaymentType.ExCWire, exCInt.BankName!, exCInt.AccountNum!);
            payBankItems.Add(itemToAdd);
        }

        if (hasExCWire && HasPayBankInfo(exCInt!.Alt1BankName, exCInt.Alt1AbaNumWire, exCInt.Alt1AccountNum))
        {
            var itemToAdd = CreatePayBankItem(Enums.ContractPaymentType.ExCAlt1Wire, exCInt.Alt1BankName!, exCInt.Alt1AccountNum!);
            payBankItems.Add(itemToAdd);
        }

        if (hasExCWire && HasPayBankInfo(exCInt!.Alt2BankName, exCInt.Alt2AbaNumWire, exCInt.Alt2AccountNum))
        {
            var itemToAdd = CreatePayBankItem(Enums.ContractPaymentType.ExCAlt2Wire, exCInt.Alt2BankName!, exCInt.Alt2AccountNum!);
            payBankItems.Add(itemToAdd);
        }

        if (hasExCAch && HasPayBankInfo(exCInt!.BankName, exCInt.AbaNumAch, exCInt.AccountNum))
        {
            var itemToAdd = CreatePayBankItem(Enums.ContractPaymentType.ExCAch, exCInt.BankName!, exCInt.AccountNum!);
            payBankItems.Add(itemToAdd);
        }

        if (hasExCAch && HasPayBankInfo(exCInt!.Alt1BankName, exCInt.Alt1AbaNumAch, exCInt.Alt1AccountNum))
        {
            var itemToAdd = CreatePayBankItem(Enums.ContractPaymentType.ExCAlt1Ach, exCInt.Alt1BankName!, exCInt.Alt1AccountNum!);
            payBankItems.Add(itemToAdd);
        }

        if (hasExCAch && HasPayBankInfo(exCInt!.Alt2BankName, exCInt.Alt2AbaNumAch, exCInt.Alt2AccountNum))
        {
            var itemToAdd = CreatePayBankItem(Enums.ContractPaymentType.ExCAlt2Ach, exCInt.Alt2BankName!, exCInt.Alt2AccountNum!);
            payBankItems.Add(itemToAdd);
        }

        if (hasAccWire && HasPayBankInfo(accInt!.WireBankName, accInt.WireAbaNum, accInt.WireAccountNum))
        {
            var itemToAdd = CreatePayBankItem(Enums.ContractPaymentType.AccWire, accInt.WireBankName!, accInt.WireAccountNum!);
            payBankItems.Add(itemToAdd);
        }

        if (hasAccACH && HasPayBankInfo(accInt!.AchBank, accInt!.AchAbaNum, accInt.AchAccountNum))
        {
            var itemToAdd = CreatePayBankItem(Enums.ContractPaymentType.AccAch, accInt.AchBank!, accInt.AchAccountNum!);
            payBankItems.Add(itemToAdd);
        }

        payBankItems = payBankItems.OrderBy(x => x.Name).ToList();
        return payBankItems;
    }

    private readonly Lazy<Dictionary<int, ContractPaymentType>> allPayTypesLazy = new(() =>
    {
        var db = Main.CreateContext();
        var items = db.ContractPaymentTypes.ToDictionary(x => x.Id);
        return items;
    });

    private PayBankItem CreatePayBankItem(Enums.ContractPaymentType itemEnum, string bankName, string accNum)
    {
        var achOrWireStr = itemEnum switch
        {
            Enums.ContractPaymentType.ExCWire => "Wire",
            Enums.ContractPaymentType.ExCAch => "ACH",
            Enums.ContractPaymentType.ExCAlt1Wire => "Wire",
            Enums.ContractPaymentType.ExCAlt1Ach => "ACH",
            Enums.ContractPaymentType.ExCAlt2Wire => "Wire",
            Enums.ContractPaymentType.ExCAlt2Ach => "ACH",
            Enums.ContractPaymentType.AccWire => "Wire",
            Enums.ContractPaymentType.AccAch => "ACH",
            _ => throw new NotImplementedException()
        };

        var allPayTypes = allPayTypesLazy.Value;
        var payType = allPayTypes[(int)itemEnum];
        var payBankItem = new PayBankItem
        {
            Id = payType.Id,
            Name = payType.Name,
            BankName = bankName,
            AccountNum = accNum,
            DisplayText = bankName + " - " + accNum + " - " + achOrWireStr
        };

        return payBankItem;
    }

    internal static bool HasPayBankInfo(string? bankName, string? abaNum, string? accNum)
    {
        var hasBankInfo = !string.IsNullOrWhiteSpace(bankName) && !string.IsNullOrWhiteSpace(abaNum) && !string.IsNullOrWhiteSpace(accNum);
        return hasBankInfo;
    }

    [Permission("Main Contract", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetDetail(int id)
    {
        var dbItem = await (
            from q in db.Contracts
                .Include(x => x.ContractDocs)
            where q.Id == id
            select new
            {
                Ctr = q,
                ExcAmendmentDate = q.ContractExhibitC != null ? q.ContractExhibitC.AmendmentDate : null,
                NaesbAmendmentDate = q.ContractNaesb != null ? q.ContractNaesb.AmendmentDate : null,
                ExhibitbAmendmentDate = q.ExhibitBAmendmentDate
            }
        ).AsNoTracking().FirstAsync();

        var dbItemInt = await (
            from q in db.Contracts
            where q.IsInternalSideOfContract
                && q.InternalEntityId == dbItem.Ctr.InternalEntityId
                && q.ProductId == dbItem.Ctr.ProductId
            select new { ExcAmendmentDate = q.ContractExhibitC != null ? q.ContractExhibitC.AmendmentDate : null }
        ).AsNoTracking().FirstAsync();

        var latestAmendmentDate = new List<DateOnly?>() {
            dbItem.ExcAmendmentDate,
            dbItem.NaesbAmendmentDate,
            dbItem.ExhibitbAmendmentDate,
            dbItemInt.ExcAmendmentDate
        }.Max();

        MainContractDetail detail = new();
        detail.MainContractId = dbItem.Ctr.Id;
        detail.ContractName = dbItem.Ctr.ContractName;
        detail.ContractNum = dbItem.Ctr.ContractNum;
        detail.TheirContractNum = dbItem.Ctr.TheirContractNum;
        detail.ProductId = dbItem.Ctr.ProductId;
        detail.CounterpartyId = dbItem.Ctr.CounterpartyId;
        detail.InternalEntityId = dbItem.Ctr.InternalEntityId;
        detail.IsPaymentNettingAllowed = dbItem.Ctr.IsPaymentNettingAllowed;
        detail.EffectiveDate = dbItem.Ctr.EffectiveDate;
        detail.ExecutionDate = dbItem.Ctr.ExecutionDate;
        detail.TerminationEffectiveDate = dbItem.Ctr.TerminationEffectiveDate;
        detail.TerminationNoticeDate = dbItem.Ctr.TerminationNoticeDate;
        detail.Notes = dbItem.Ctr.Notes;
        detail.ContractStatusId = dbItem.Ctr.StatusId;
        detail.SignerId = dbItem.Ctr.SignerId;
        detail.LastAmendmentDate = latestAmendmentDate;
        detail.PayFromBankId = dbItem.Ctr.PayFromBankId;
        detail.PayToBankId = dbItem.Ctr.PayToBankId;
        detail.Uploads = dbItem.Ctr.ContractDocs.Select(x => new DocItemWithInactive
        { FileNameOriginal = x.FileNameOriginal, FileNameOnDisk = x.FileNameOnDisk, inactiveDate = x.InactiveDate }).ToList();

        detail.Uploads = (
            from q in detail.Uploads
            let isActive = q.inactiveDate == null
            let activeFileName = isActive ? q.FileNameOriginal : ""
            orderby
                isActive descending, //sort active files first
                activeFileName ascending, //sort active files by name
                q.inactiveDate descending, //sort inactive files by inactive date
                q.FileNameOriginal ascending //sort inactive files by name if inactive dates are the same
            select q
        ).ToList();

        detail.PayBankItems = await GetPayBankInfoAsync(id);
        return Ok(detail);
    }

    [Permission("Main Contract", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GenerateContracts(int mainContractId, bool genMain, bool genExB, bool genExC)
    {
        if (mainContractId > 0)
        {
            var productId = await db.Contracts.Where(x => x.Id == mainContractId).Select(x => x.ProductId).FirstOrDefaultAsync();
            var isNatGas = productId == 1;

            if (genMain && isNatGas)
            {
                var contractDoc = new ContractDocNaesb(fileService, uploadFolderName, templatesFolderName, signaturesFolderName, db);
                await contractDoc.Generate(mainContractId);
            }
            if (genExB)
            {
                var contractDoc = new ContractDocExB(fileService, uploadFolderName, templatesFolderName, signaturesFolderName, db);
                await contractDoc.Generate(mainContractId);
            }
            if (genExC)
            {
                var contractDoc = new ContractDocExC(fileService, uploadFolderName, templatesFolderName, signaturesFolderName, db);
                await contractDoc.Generate(mainContractId);
            }
        }

        return Ok(mainContractId);
    }

    [Permission("Main Contract", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> SaveDetail(MainContractDetail detail, SaveType saveType, InfoType infoType)
    {
        int resultId = 0;

        await db.Database.CreateExecutionStrategy().Execute(async () =>
        {
            using var dbContextTransaction = await db.Database.BeginTransactionAsync();
            Contract? dbItemCpty = null;
            Contract? dbItemInt = null;

            if (saveType != SaveType.New)
            {
                dbItemCpty = await (
                    from q in db.Contracts
                        .Include(x => x.Counterparty)
                        .ThenInclude(x => x.Country)
                        .Include(x => x.ContractAccounting)
                        .Include(x => x.ContractContact)
                        .Include(x => x.ContractExhibitBs)
                        .Include(x => x.ContractExhibitC)
                        .Include(x => x.ContractNaesb)
                        .Include(x => x.ContractCrudes)
                        .Include(x => x.ContractOther)
                        .Include(x => x.ContractDocs)
                        .Include(x => x.ContractExhibitBSelections)
                    where q.Id == detail.MainContractId
                    select q
                ).FirstOrDefaultAsync();
            }

            dbItemInt = await (
                from q in db.Contracts
                    .Include(x => x.InternalEntity)
                    .ThenInclude(x => x.Country)
                    .Include(x => x.ContractAccounting)
                    .Include(x => x.ContractContact)
                    .Include(x => x.ContractExhibitC)
                    .Include(x => x.ContractOther)
                where q.IsInternalSideOfContract
                    && q.InternalEntityId == detail.InternalEntityId
                    && q.ProductId == detail.ProductId
                select q
            ).FirstOrDefaultAsync();

            if (dbItemCpty == null) //if the counterparty item does not exist then add it
            {
                dbItemCpty = new Contract();
                db.Contracts.Add(dbItemCpty);
            }

            if (dbItemInt == null) //if the internal item does not exist then add it
            {
                dbItemInt = new Contract();
                db.Contracts.Add(dbItemInt);
            }

            if (dbItemCpty.ContractDocs != null)
                db.ContractDocs.RemoveRange(dbItemCpty.ContractDocs);

            //remove counterparty existing items so that they get completely re-inserted
            if (infoType == InfoType.Counterparty && dbItemCpty.ContractContact != null && detail.ContactInfo != null)
                db.ContractContacts.RemoveRange(dbItemCpty.ContractContact);
            if (infoType == InfoType.Counterparty && dbItemCpty.ContractAccounting != null && detail.AccountingInfo != null)
                db.ContractAccountings.RemoveRange(dbItemCpty.ContractAccounting);
            if (infoType == InfoType.Counterparty && dbItemCpty.ContractNaesb != null && detail.NaesbInfo != null)
                db.ContractNaesbs.RemoveRange(dbItemCpty.ContractNaesb);
            if (infoType == InfoType.Counterparty && dbItemCpty.ContractCrudes != null && detail.CrudeInfo != null)
                db.ContractCrudes.RemoveRange(dbItemCpty.ContractCrudes);
            if (infoType == InfoType.Counterparty && dbItemCpty.ContractExhibitC != null && detail.ExhibitCInfo != null)
                db.ContractExhibitCs.RemoveRange(dbItemCpty.ContractExhibitC);

            //remove internal entity existing items so that they get completely re-inserted
            if (infoType == InfoType.Internal && dbItemInt.ContractContact != null && detail.ContactInfo != null)
                db.ContractContacts.RemoveRange(dbItemInt.ContractContact);
            if (infoType == InfoType.Internal && dbItemInt.ContractAccounting != null && detail.AccountingInfo != null)
                db.ContractAccountings.RemoveRange(dbItemInt.ContractAccounting);
            if (infoType == InfoType.Internal && dbItemInt.ContractExhibitC != null && detail.ExhibitCInfo != null)
                db.ContractExhibitCs.RemoveRange(dbItemInt.ContractExhibitC);
            if (infoType == InfoType.Internal && dbItemInt.ContractOther != null && detail.OtherInfo != null)
                db.ContractOthers.RemoveRange(dbItemInt.ContractOther);

            var d = detail;
            dbItemCpty.ContractName = d.ContractName;
            dbItemCpty.ContractNum = d.ContractNum;
            dbItemCpty.TheirContractNum = d.TheirContractNum;
            dbItemCpty.IsInternalSideOfContract = false;
            dbItemCpty.ProductId = d.ProductId;
            dbItemCpty.CounterpartyId = d.CounterpartyId;
            dbItemCpty.InternalEntityId = d.InternalEntityId;
            dbItemCpty.StatusId = d.ContractStatusId;
            dbItemCpty.SignerId = d.SignerId;
            dbItemCpty.EffectiveDate = d.EffectiveDate?.ToUniversalTime();
            dbItemCpty.ExecutionDate = d.ExecutionDate?.ToUniversalTime();
            dbItemCpty.TerminationNoticeDate = d.TerminationNoticeDate?.ToUniversalTime();
            dbItemCpty.TerminationEffectiveDate = d.TerminationEffectiveDate?.ToUniversalTime();
            dbItemCpty.IsPaymentNettingAllowed = d.IsPaymentNettingAllowed;
            dbItemCpty.Notes = d.Notes;
            dbItemCpty.ExhibitBAmendmentDate = d.ExhibitBInfo?.ExhibitBDetail?.ExhibitBAmendmentDate;
            dbItemCpty.PayToBankId = d.PayToBankId;
            dbItemCpty.PayFromBankId = d.PayFromBankId;

            var uploadDocs = d.Uploads.Select(x => new ContractDoc
            {
                ContractId = d.MainContractId,
                FileNameOriginal = x.FileNameOriginal,
                FileNameOnDisk = x.FileNameOnDisk,
                InactiveDate = x.inactiveDate
            }).ToList();
            dbItemCpty.ContractDocs!.Clear();
            uploadDocs.ForEach(x => dbItemCpty.ContractDocs!.Add(x));

            DateTime noStartDate = new(2000, 1, 1, 0, 0, 0, DateTimeKind.Utc);
            dbItemInt.ContractNum = $"Internal:{d.InternalEntityId}:{d.ProductId}";
            dbItemInt.ContractName = $"Internal:{d.InternalEntityId}:{d.ProductId}";
            dbItemInt.TheirContractNum = null;
            dbItemInt.IsInternalSideOfContract = true;
            dbItemInt.ProductId = d.ProductId;
            dbItemInt.CounterpartyId = d.InternalEntityId;
            dbItemInt.InternalEntityId = d.InternalEntityId;
            dbItemInt.StatusId = 6; //Executed status
            dbItemInt.EffectiveDate = null;
            dbItemInt.ExecutionDate = null;
            dbItemInt.TerminationNoticeDate = null;
            dbItemInt.TerminationEffectiveDate = null;
            dbItemInt.IsPaymentNettingAllowed = true;
            dbItemInt.Notes = "Our Info Capture Side";

            if (detail.AccountingInfo != null)
            {
                ContractAccounting acc = new();
                acc.InvoiceAddressLine1 = detail.AccountingInfo.InvoiceAddressLine1;
                acc.InvoiceAddressLine2 = detail.AccountingInfo.InvoiceAddressLine2;
                acc.InvoiceCountryId = detail.AccountingInfo.InvoiceCountryId;
                acc.InvoiceCity = detail.AccountingInfo.InvoiceCity;
                acc.InvoiceStateId = detail.AccountingInfo.InvoiceStateId;
                acc.InvoiceZip = detail.AccountingInfo.InvoiceZip;
                acc.InvoiceAttn = detail.AccountingInfo.InvoiceAttn;
                acc.InvoiceTelephoneNum = detail.AccountingInfo.InvoiceTelephoneNum;
                acc.InvoiceFaxNum = detail.AccountingInfo.InvoiceFaxNum;
                acc.InvoiceEmailAddress = detail.AccountingInfo.InvoiceEmailAddress;
                acc.DoEmail = detail.AccountingInfo.DoEmail;
                acc.DoFax = detail.AccountingInfo.DoFax;
                acc.WireBankName = detail.AccountingInfo.WireBank;
                acc.WireAbaNum = detail.AccountingInfo.WireAbaNum;
                acc.WireAccountNum = detail.AccountingInfo.WireAccountNum;
                acc.WireOtherDetails = detail.AccountingInfo.WireOtherDetails;
                acc.AchBank = detail.AccountingInfo.AchBank;
                acc.AchAbaNum = detail.AccountingInfo.AchAbaNum;
                acc.AchAccountNum = detail.AccountingInfo.AchAccountNum;
                acc.AchOtherDetails = detail.AccountingInfo.AchOtherDetails;
                acc.ChecksAttn = detail.AccountingInfo.ChecksAttn;
                acc.ChecksAddressLine1 = detail.AccountingInfo.ChecksAddressLine1;
                acc.ChecksAddressLine2 = detail.AccountingInfo.ChecksAddressLine2;
                acc.ChecksCountryId = detail.AccountingInfo.ChecksCountryId;
                acc.ChecksCity = detail.AccountingInfo.ChecksCity;
                acc.ChecksStateId = detail.AccountingInfo.ChecksStateId;
                acc.ChecksZip = detail.AccountingInfo.ChecksZip;

                if (infoType == InfoType.Counterparty)
                    dbItemCpty.ContractAccounting = acc;
                else if (infoType == InfoType.Internal)
                    dbItemInt.ContractAccounting = acc;
            }

            if (detail.ContactInfo != null)
            {
                ContractContact contact = new();
                contact.CommercialAddressLine1 = detail.ContactInfo.CommercialAddressLine1;
                contact.CommercialAddressLine2 = detail.ContactInfo.CommercialAddressLine2;
                contact.CommercialCountryId = detail.ContactInfo.CommercialCountryId;
                contact.CommercialCity = detail.ContactInfo.CommercialCity;
                contact.CommercialStateId = detail.ContactInfo.CommercialStateId;
                contact.CommercialZip = detail.ContactInfo.CommercialZip;
                contact.CommercialAttn = detail.ContactInfo.CommercialAttn;
                contact.CommercialTelephoneNum = detail.ContactInfo.CommercialTelephoneNum;
                contact.CommercialFaxNum = detail.ContactInfo.CommercialFaxNum;
                contact.CommercialEmailAddress = detail.ContactInfo.CommercialEmailAddress;
                contact.SchedulingAddressLine1 = detail.ContactInfo.SchedulingAddressLine1;
                contact.SchedulingAddressLine2 = detail.ContactInfo.SchedulingAddressLine2;
                contact.SchedulingCountryId = detail.ContactInfo.SchedulingCountryId;
                contact.SchedulingCity = detail.ContactInfo.SchedulingCity;
                contact.SchedulingStateId = detail.ContactInfo.SchedulingStateId;
                contact.SchedulingZip = detail.ContactInfo.SchedulingZip;
                contact.SchedulingAttn = detail.ContactInfo.SchedulingAttn;
                contact.SchedulingTelephoneNum = detail.ContactInfo.SchedulingTelephoneNum;
                contact.SchedulingFaxNum = detail.ContactInfo.SchedulingFaxNum;
                contact.SchedulingEmailAddress = detail.ContactInfo.SchedulingEmailAddress;
                contact.NoticesAddressLine1 = detail.ContactInfo.NoticesAddressLine1;
                contact.NoticesAddressLine2 = detail.ContactInfo.NoticesAddressLine2;
                contact.NoticesCountryId = detail.ContactInfo.NoticesCountryId;
                contact.NoticesCity = detail.ContactInfo.NoticesCity;
                contact.NoticesStateId = detail.ContactInfo.NoticesStateId;
                contact.NoticesZip = detail.ContactInfo.NoticesZip;
                contact.NoticesAttn = detail.ContactInfo.NoticesAttn;
                contact.NoticesTelephoneNum = detail.ContactInfo.NoticesTelephoneNum;
                contact.NoticesFaxNum = detail.ContactInfo.NoticesFaxNum;
                contact.NoticesEmailAddress = detail.ContactInfo.NoticesEmailAddress;
                contact.CreditAddressLine1 = detail.ContactInfo.CreditAddressLine1;
                contact.CreditAddressLine2 = detail.ContactInfo.CreditAddressLine2;
                contact.CreditCountryId = detail.ContactInfo.CreditCountryId;
                contact.CreditCity = detail.ContactInfo.CreditCity;
                contact.CreditStateId = detail.ContactInfo.CreditStateId;
                contact.CreditZip = detail.ContactInfo.CreditZip;
                contact.CreditAttn = detail.ContactInfo.CreditAttn;
                contact.CreditTelephoneNum = detail.ContactInfo.CreditTelephoneNum;
                contact.CreditFaxNum = detail.ContactInfo.CreditFaxNum;
                contact.CreditEmailAddress = detail.ContactInfo.CreditEmailAddress;
                contact.ConfirmationsAddressLine1 = detail.ContactInfo.ConfirmationsAddressLine1;
                contact.ConfirmationsAddressLine2 = detail.ContactInfo.ConfirmationsAddressLine2;
                contact.ConfirmationsCountryId = detail.ContactInfo.ConfirmationsCountryId;
                contact.ConfirmationsCity = detail.ContactInfo.ConfirmationsCity;
                contact.ConfirmationsStateId = detail.ContactInfo.ConfirmationsStateId;
                contact.ConfirmationsZip = detail.ContactInfo.ConfirmationsZip;
                contact.ConfirmationsAttn = detail.ContactInfo.ConfirmationsAttn;
                contact.ConfirmationsTelephoneNum = detail.ContactInfo.ConfirmationsTelephoneNum;
                contact.ConfirmationsFaxNum = detail.ContactInfo.ConfirmationsFaxNum;
                contact.ConfirmationsEmailAddress = detail.ContactInfo.ConfirmationsEmailAddress;
                contact.DoEmail = detail.ContactInfo.DoEmail;
                contact.DoFax = detail.ContactInfo.DoFax;
                contact.CombineEmail = detail.ContactInfo.CombineEmail;

                if (infoType == InfoType.Counterparty)
                    dbItemCpty.ContractContact = contact;
                else if (infoType == InfoType.Internal)
                    dbItemInt.ContractContact = contact;
            }

            if (detail.ExhibitCInfo != null)
            {
                ContractExhibitC c = new();
                c.BankName = detail.ExhibitCInfo.BankName;
                c.CityState = detail.ExhibitCInfo.CityState;
                c.AbaNumWire = detail.ExhibitCInfo.AbaNumWire;
                c.AbaNumAch = detail.ExhibitCInfo.AbaNumAch;
                c.AccountName = detail.ExhibitCInfo.AccountName;
                c.ForFurtherCreditTo = detail.ExhibitCInfo.ForFurtherCreditTo;
                c.CreditName = detail.ExhibitCInfo.CreditName;
                c.AccountNum = detail.ExhibitCInfo.AccountNum;
                c.Alt1BankName = detail.ExhibitCInfo.Alt1BankName;
                c.Alt1CityState = detail.ExhibitCInfo.Alt1CityState;
                c.Alt1AbaNumWire = detail.ExhibitCInfo.Alt1AbaNumWire;
                c.Alt1AbaNumAch = detail.ExhibitCInfo.Alt1AbaNumAch;
                c.Alt1AccountName = detail.ExhibitCInfo.Alt1AccountName;
                c.Alt1ForFurtherCreditTo = detail.ExhibitCInfo.Alt1ForFurtherCreditTo;
                c.Alt1CreditName = detail.ExhibitCInfo.Alt1CreditName;
                c.Alt1AccountNum = detail.ExhibitCInfo.Alt1AccountNum;
                c.Alt2BankName = detail.ExhibitCInfo.Alt2BankName;
                c.Alt2CityState = detail.ExhibitCInfo.Alt2CityState;
                c.Alt2AbaNumWire = detail.ExhibitCInfo.Alt2AbaNumWire;
                c.Alt2AbaNumAch = detail.ExhibitCInfo.Alt2AbaNumAch;
                c.Alt2AccountName = detail.ExhibitCInfo.Alt2AccountName;
                c.Alt2ForFurtherCreditTo = detail.ExhibitCInfo.Alt2ForFurtherCreditTo;
                c.Alt2CreditName = detail.ExhibitCInfo.Alt2CreditName;
                c.Alt2AccountNum = detail.ExhibitCInfo.Alt2AccountNum;
                c.CheckPaymentAddressLine1 = detail.ExhibitCInfo.CheckPaymentAddressLine1;
                c.CheckPaymentAddressLine2 = detail.ExhibitCInfo.CheckPaymentAddressLine2;
                c.CheckPaymentCountryId = detail.ExhibitCInfo.CheckPaymentCountryId;
                c.CheckPaymentCity = detail.ExhibitCInfo.CheckPaymentCity;
                c.CheckPaymentStateId = detail.ExhibitCInfo.CheckPaymentStateId;
                c.CheckPaymentZip = detail.ExhibitCInfo.CheckPaymentZip;
                c.AmendmentDate = detail.ExhibitCInfo.AmendmentDate;

                if (infoType == InfoType.Counterparty)
                    dbItemCpty.ContractExhibitC = c;
                else if (infoType == InfoType.Internal)
                    dbItemInt.ContractExhibitC = c;
            }

            var isNatGas = dbItemCpty.ProductId == (int)Enums.Product.NaturalGas;
            var hasInternalAccountingInfo = infoType == InfoType.Internal && detail.AccountingInfo != null;
            var hasCounterpartyNaesbInfo = infoType == InfoType.Counterparty && detail.NaesbInfo != null;
            var hasSavedInternalAccountingInfo = dbItemInt.ContractAccounting != null;
            dbItemCpty.ContractNaesb ??= Naesb2006Controller.GetNewNaesbInfo();

            //if we're saving on the internal side then force the NAESB payment options to match the internal accounting info selections
            if (isNatGas && hasInternalAccountingInfo)
            {
                dbItemCpty.ContractNaesb!.WireTransferPayment = detail.AccountingInfo!.IsWireChecked;
                dbItemCpty.ContractNaesb.AchPayment = detail.AccountingInfo.IsAchChecked;
                dbItemCpty.ContractNaesb.CheckPayment = detail.AccountingInfo.IsChecksChecked;
            }
            else if (isNatGas && hasCounterpartyNaesbInfo)
            {
                ContractNaesb naesb = new();
                naesb.TransactionProcedureTypeId = detail.NaesbInfo!.TransactionProcedureTypeId;
                naesb.ConfirmDeadline = detail.NaesbInfo.ConfirmDeadline;
                naesb.ConfirmingPartyTypeId = detail.NaesbInfo.ConfirmingPartyTypeId;
                naesb.PerformanceObligationTypeId = detail.NaesbInfo.PerformanceObligationTypeId;
                naesb.SpotPricePublicationTypeId = detail.NaesbInfo.SpotPricePublicationTypeId;
                naesb.TaxesOptionId = detail.NaesbInfo.TaxesOptionId;
                naesb.PaymentDateOptionId = detail.NaesbInfo.PaymentDateOptionId;
                naesb.WireTransferPayment = detail.NaesbInfo.WireTransferPayment;
                naesb.AchPayment = detail.NaesbInfo.AchPayment;
                naesb.CheckPayment = detail.NaesbInfo.CheckPayment;
                naesb.NettingTypeId = detail.NaesbInfo.NettingTypeId;
                naesb.AdditionalEventTypeId = detail.NaesbInfo.AdditionalEventTypeId;
                naesb.EarlyTerminationTypeId = detail.NaesbInfo.EarlyTerminationTypeId;
                naesb.OtherAgreementSetoffsTypeId = detail.NaesbInfo.OtherAgreementSetoffsTypeId;
                naesb.ChoiceOfLaw = detail.NaesbInfo.ChoiceOfLaw;
                naesb.ConfidentialityTypeId = detail.NaesbInfo.ConfidentialityTypeId;
                naesb.AddendumExhibitB = detail.NaesbInfo.AddendumExhibitB;
                naesb.AddendumExhibitC = detail.NaesbInfo.AddendumExhibitC;
                naesb.AddendumExhibitD = detail.NaesbInfo.AddendumExhibitD;
                naesb.AddendumExhibitE = detail.NaesbInfo.AddendumExhibitE;
                naesb.AddendumExhibitF = detail.NaesbInfo.AddendumExhibitF;
                naesb.AddendumExhibitG = detail.NaesbInfo.AddendumExhibitG;
                naesb.AddendumExhibitH = detail.NaesbInfo.AddendumExhibitH;
                naesb.AddendumExhibitI = detail.NaesbInfo.AddendumExhibitI;
                naesb.AddendumExhibitJ = detail.NaesbInfo.AddendumExhibitJ;
                naesb.AddendumExhibitK = detail.NaesbInfo.AddendumExhibitK;
                naesb.AddendumExhibitL = detail.NaesbInfo.AddendumExhibitL;
                naesb.PaymentDateTypeId = detail.NaesbInfo.PaymentDateTypeId;
                naesb.SpotPricePublicationOther = detail.NaesbInfo.SpotPricePublicationOther;
                naesb.AmendmentDate = detail.NaesbInfo.AmendmentDate;
                dbItemCpty.ContractNaesb = naesb;
            }

            if (infoType == InfoType.Counterparty)
                AddContractCrudeInfo(dbItemCpty, detail);

            if (infoType == InfoType.Internal && detail.OtherInfo != null)
            {
                ContractOther otherInt = new();
                otherInt.LcLanguage = detail.OtherInfo.LcLanguage;
                otherInt.AgentClause = detail.OtherInfo.AgentClause;
                dbItemInt.ContractOther = otherInt;
            }

            await db.SaveChangesAsync();
            resultId = dbItemCpty.Id;
            var mainContractId = dbItemCpty.Id;
            await dbContextTransaction.CommitAsync();

            if (infoType == InfoType.Counterparty && detail.ExhibitBInfo != null)
                await SaveExhibitB(mainContractId, detail.ExhibitBInfo);
        });

        return Ok(resultId);
    }

    [Permission("Main Contract", PermissionType.Modify)]
    [Route("[action]")]
    public static void AddContractCrudeInfo(Contract dbItem, MainContractDetail detail)
    {
        var isCrude = dbItem.ProductId == (int)Enums.Product.CrudeOil;
        if (!isCrude || detail.CrudeInfo == null) return;
        foreach (var calc in detail.CrudeInfo!.CalcItems)
        {
            var dbCrudeItem = new ContractCrude();
            dbCrudeItem.PipeLossAllowance = calc.PipeLossAllowance;
            dbCrudeItem.Differential = calc.Differential;
            dbCrudeItem.IsDiffBeforePipeLoss = calc.IsDiffBeforePipeLoss;
            dbCrudeItem.EffectiveDate = calc.EffectiveDate;
            foreach (var point in calc.PointIds)
            {
                var dbCrudePointItem = new ContractCrudePoint();
                dbCrudePointItem.PointId = point;
                dbCrudeItem.ContractCrudePoints.Add(dbCrudePointItem);
            }
            dbItem.ContractCrudes!.Add(dbCrudeItem);
        }
    }

    public async Task SaveExhibitB(int mainContractId, MainContractExhibitBInfo info)
    {
        var itemSelected = false;
        var subItemSelected = false;
        await db.Database.CreateExecutionStrategy().Execute(async () =>
        {
            using var dbContextTransaction = await db.Database.BeginTransactionAsync();

            if (info.ExhibitBItemsToDelete != null && info.ExhibitBItemsToDelete.Count > 0)
            {
                foreach (var item in info.ExhibitBItemsToDelete)
                {
                    db.ContractExhibitBs?.RemoveRange(db.ContractExhibitBs.Where(x => x.Id == item.Id));
                    db.ContractExhibitBs?.RemoveRange(db.ContractExhibitBs.Where(x => x.ParentId == item.Id));
                }
            }
            db.ContractExhibitBSelections?.RemoveRange(db.ContractExhibitBSelections.Where(x => x.ContractId == mainContractId));
            db.ContractExhibitBs?.RemoveRange(db.ContractExhibitBs.Where(x => x.ContractId == mainContractId));

            if (info.ExhibitBDetail != null && info.ExhibitBDetail.Items.Count > 0)
            {
                foreach (var ExhibitBItem in info.ExhibitBDetail.Items)
                {
                    var globalExhibitItem = db.ContractExhibitBs?.Where(x => x.Id == ExhibitBItem.Id && x.ContractId == null).FirstOrDefault();

                    if (info.ExhibitBSelectionDetail != null)
                    {
                        if (globalExhibitItem != null && info.ExhibitBSelectionDetail.ExibhitBSelectedIds.Contains(ExhibitBItem.Id))
                        {
                            ContractExhibitBSelection selection = new();
                            selection.ContractId = mainContractId;
                            selection.ContractExhibitBId = ExhibitBItem.Id;
                            db.ContractExhibitBSelections?.Add(selection);
                        }
                        if (globalExhibitItem == null && info.ExhibitBSelectionDetail.ExibhitBSelectedIds.Contains(ExhibitBItem.Id))
                        {
                            itemSelected = true;
                        }
                        else
                        {
                            itemSelected = false;
                        }
                    }

                    if (globalExhibitItem != null)
                    {
                        globalExhibitItem.Title = ExhibitBItem.Title;
                        globalExhibitItem.Body = ExhibitBItem.Body;
                        globalExhibitItem.InactiveDate = ExhibitBItem.InactiveDate;
                        globalExhibitItem.Ordinal = ExhibitBItem.Ordinal;
                    }
                    else
                    {
                        var newExBItem = new ContractExhibitB();
                        newExBItem.Title = ExhibitBItem.Title;
                        newExBItem.Body = ExhibitBItem.Body;
                        newExBItem.Ordinal = ExhibitBItem.Ordinal;
                        newExBItem.ParentId = ExhibitBItem.ParentId;
                        newExBItem.ContractId = ExhibitBItem.IsContractSpecific ? mainContractId : null;
                        newExBItem.InactiveDate = ExhibitBItem.InactiveDate;
                        db.ContractExhibitBs?.Add(newExBItem);

                        await db.SaveChangesAsync();

                        ExhibitBItem.Id = newExBItem.Id;

                        if (itemSelected == true)
                        {
                            ContractExhibitBSelection selection = new();
                            selection.ContractId = mainContractId;
                            selection.ContractExhibitBId = newExBItem.Id;
                            db.ContractExhibitBSelections?.Add(selection);
                        }
                    }

                    var childItems = ExhibitBItem.Items;
                    foreach (var childItem in childItems)
                    {
                        var globalChildItem = db.ContractExhibitBs?.Where(x => x.Id == childItem.Id && x.ContractId == null).FirstOrDefault();

                        if (info.ExhibitBSelectionDetail != null)
                        {
                            if (globalChildItem != null && info.ExhibitBSelectionDetail.ExibhitBSelectedIds.Contains(childItem.Id))
                            {
                                ContractExhibitBSelection selection = new();
                                selection.ContractId = mainContractId;
                                selection.ContractExhibitBId = childItem.Id;
                                db.ContractExhibitBSelections?.Add(selection);
                            }
                            if (globalChildItem == null && info.ExhibitBSelectionDetail.ExibhitBSelectedIds.Contains(childItem.Id))
                            {
                                subItemSelected = true;
                            }
                            else
                            {
                                subItemSelected = false;
                            }
                        }

                        if (globalChildItem != null)
                        {
                            globalChildItem.Title = childItem.Title;
                            globalChildItem.Body = childItem.Body;
                            globalChildItem.InactiveDate = childItem.InactiveDate;
                            globalChildItem.Ordinal = childItem.Ordinal;
                        }
                        else
                        {
                            var newChildItem = new ContractExhibitB();
                            newChildItem.Title = childItem.Title;
                            newChildItem.Body = childItem.Body;
                            newChildItem.Ordinal = childItem.Ordinal;
                            newChildItem.ParentId = ExhibitBItem.Id;
                            newChildItem.ContractId = childItem.IsContractSpecific ? mainContractId : null;
                            newChildItem.InactiveDate = childItem.InactiveDate;
                            db.ContractExhibitBs?.Add(newChildItem);

                            await db.SaveChangesAsync();

                            if (subItemSelected == true)
                            {
                                ContractExhibitBSelection selection = new();
                                selection.ContractId = mainContractId;
                                selection.ContractExhibitBId = newChildItem.Id;
                                db.ContractExhibitBSelections?.Add(selection);
                            }
                        }
                    }
                }
            }
            await db.SaveChangesAsync();
            await dbContextTransaction.CommitAsync();
        });
    }

    [Permission("Main Contract", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> UploadDoc(IEnumerable<IFormFile> files, [FromODataUri] string metaData)
    {
        var docItems = new List<DocItemWithInactive>();
        MemoryStream ms = new(Encoding.UTF8.GetBytes(metaData));
        var serializer = new DataContractJsonSerializer(typeof(ChunkMetaData));
        var chunkData = serializer.ReadObject(ms) as ChunkMetaData;
        string? fileNameOnDisk = null;

        if (files != null)
        {
            foreach (var file in files)
            {
                if (file.Length > 0 && chunkData != null)
                {
                    var newFileName = Util.String.GetNewGuid() + Path.GetExtension(chunkData.FileName);
                    fileNameOnDisk = await fileService.UploadFileAsync(uploadFolderName, newFileName, file);
                }
            }
        }

        if (fileNameOnDisk != null && chunkData != null && chunkData.ChunkIndex == chunkData.TotalChunks - 1)
        {
            docItems.Add(new DocItemWithInactive
            {
                FileNameOriginal = chunkData.FileName,
                FileNameOnDisk = fileNameOnDisk
            });
            return Ok(docItems);
        }

        return Ok();
    }

    [Permission("Main Contract", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> DownloadDoc(string fileNameOnDisk)
    {
        try
        {
            var fileResponse = await fileService.DownloadFileAsync(uploadFolderName, fileNameOnDisk);
            return File(fileResponse.Stream, fileResponse.ContentType, fileResponse.FileName);
        }
        catch (Exception ex)
        {
            const string messagePrefix = "Upload Doc";
            var bytes = Util.GetExceptionFilesBytes(ex, messagePrefix);
            return File(bytes, "text/plain");
        }
    }

    [Permission("Main Contract", PermissionType.Modify)]
    [Route("[action]/{id}")]
    public async Task<IActionResult> DeleteDetail(int id)
    {
        await db.ContractExhibitBSelections.Where(x => x.ContractId == id).ForEachAsync(item => db.ContractExhibitBSelections.Remove(item));
        Contract dbItem = await db.Contracts.Where(x => x.Id == id).FirstAsync();
        db.Contracts.Remove(dbItem);
        await db.SaveChangesAsync();

        return Ok();
    }

    [Permission("Main Contract", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> GetNewContractNum()
    {
        List<string> allContractNums = (await db.Contracts.Select(x => x.ContractNum).ToListAsync());
        List<string> numericContractNums = allContractNums.Where(x => x.All(char.IsDigit)).ToList();
        int maxNum = numericContractNums.Count > 0 ? numericContractNums.Max(x => int.Parse(x)) : 1000;
        int newContractNum = maxNum + 1;

        return Ok(newContractNum.ToString());
    }
}

