using Fast.Logic.GasControl;
using Microsoft.AspNetCore.OData.Routing.Controllers;

namespace Fast.Web.Controllers
{
    [Authorize]
    [ApiController]
    [Route("api/[controller]")]
    public class GasTrackController : ODataController
    {
        private readonly AuthorizationHelper authHelper;
        public GasTrackController()
        {
            authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
        }

        [Permission("GasTrack Export", PermissionType.View)]
        [Route("[action]")]
        public async Task<IActionResult> GetRequiredData()
        {
            var hasGasTrackStandardPermission = await authHelper.IsAuthorizedAsync(User, "GasTrack Export", PermissionType.Standard);

            var result = new { hasGasTrackStandardPermission };
            return Ok(result);
        }

        [Permission("GasTrack Export", PermissionType.Standard)]
        [Route("[action]")]
        public async Task<IActionResult> Export(DateOnly startDate, DateOnly endDate)
        {
            try
            {
                var gasExporter = new GasTrackExporter();
                var gasExportList = await gasExporter.CreateExportList(startDate, endDate);

                var fileCreator = new GasTrackFileCreator();
                var memStream = fileCreator.CreateMemoryStream(gasExportList);
                return File(memStream, "application/octet-stream");
            }
            catch (Exception ex)
            {
                string messagePrefix = $"GasTrack export failed";
                messagePrefix += $"\r\nStart Date: {startDate:MM/dd/yyyy}";
                messagePrefix += $"\r\n  End Date: {endDate:MM/dd/yyyy}";
                var bytes = Util.GetExceptionFilesBytes(ex, messagePrefix);
                return File(bytes, "text/plain");
            }
        }
    }
}
