using Fast.Models;

namespace Fast.Web.Controllers;


[Authorize]
[ApiController]
[Route("api/[controller]")]
public class ExhibitBInfoController : Controller
{
    private readonly MyDbContext db;

    private readonly AuthorizationHelper authHelper;

    public ExhibitBInfoController(MyDbContext context)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
    }

    [Permission("Main Contract", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData()
    {
        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Main Contract", PermissionType.Modify);
        var result = new { hasModifyPermission };
        return Ok(result);
    }

    [Permission("Main Contract", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetExhibitBInfo(int contractId)
    {
        MainContractExhibitBInfo info = new();
        if (contractId == 0)
        {
            MainContractExhibitBDetail b = new();
            b.Items = await GetExhibitItems(0);
            info.ExhibitBDetail = b;

            return Ok(info);
        }
        else
        {
            var dbItem = await (
                from q in db.Contracts
                where q.Id == contractId
                select new
                {
                    q.ExhibitBAmendmentDate,
                    q.ContractExhibitBSelections
                }
            ).AsNoTracking().FirstAsync();

            MainContractExhibitBDetail b = new();
            b.Items = await GetExhibitItems(contractId);
            b.ExhibitBAmendmentDate = dbItem.ExhibitBAmendmentDate;
            info.ExhibitBDetail = b;

            MainContractExhibitBSelectionDetail bSel = new();
            foreach (var selection in dbItem.ContractExhibitBSelections)
            {
                bSel.ExibhitBSelectedIds.Add(selection.ContractExhibitBId);
            }
            bSel.ContractId = contractId;
            info.ExhibitBSelectionDetail = bSel;

            return Ok(info);
        }
    }
    private async Task<List<MainContractExhibitBItem>> GetExhibitItems(int contractId)
    {
        var dbAllItems = await db.ContractExhibitBs.Where(x => x.ContractId == null || x.ContractId == contractId).AsNoTracking().ToListAsync();
        var dbAllItemsByParentId = dbAllItems.OrderBy(x => x.Ordinal).ToLookup(x => x.ParentId.GetValueOrDefault());
        var dbParentItemsForContract = dbAllItems.Where(x => x.ParentId == null).OrderBy(x => x.Ordinal).ToList();

        List<MainContractExhibitBItem> result = new();
        foreach (var dbParentItem in dbParentItemsForContract)
        {
            MainContractExhibitBItem parentItem = new();
            parentItem.Id = dbParentItem.Id;
            parentItem.Title = dbParentItem.Title;
            parentItem.Body = dbParentItem.Body;
            parentItem.Ordinal = dbParentItem.Ordinal;
            parentItem.IsContractSpecific = dbParentItem.ContractId != null;
            parentItem.Items = new();
            parentItem.ParentId = dbParentItem.Id;
            parentItem.InactiveDate = dbParentItem.InactiveDate;

            var dbChildItems = dbAllItemsByParentId[dbParentItem.Id];
            foreach (var dbChildItem in dbChildItems)
            {
                MainContractExhibitBItem childItem = new();
                childItem.Id = dbChildItem.Id;
                childItem.Title = dbChildItem.Title;
                childItem.Body = dbChildItem.Body;
                childItem.Ordinal = dbChildItem.Ordinal;
                childItem.IsContractSpecific = dbChildItem.ContractId != null;
                childItem.InactiveDate = dbChildItem.InactiveDate;
                parentItem.Items.Add(childItem);
                parentItem.ParentId = dbParentItem.Id;
            }
            result.Add(parentItem);
        }

        return result;
    }

}
