using Fast.Models;

namespace Fast.Web.Controllers
{
    [Authorize]
    [ApiController]
    [Route("api/[controller]")]

    public class EntityInfoController : Controller
    {
        private readonly MyDbContext db;
        private readonly AuthorizationHelper authHelper;
        public EntityInfoController(MyDbContext context)
        {
            db = context;
            authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
        }

        [Permission("Main Contract", PermissionType.View)]
        [Route("[action]")]
        public async Task<IActionResult> GetRequiredData()
        {
            var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Main Contract", PermissionType.Modify);
            var result = new { hasModifyPermission };
            return Ok(result);
        }

        [Permission("Main Contract", PermissionType.View)]
        [Route("[action]")]
        public async Task<IActionResult> GetEntityInfo(int entityId)
        {
            var dbItem = await (
                from q in db.Counterparties
                .Include(x => x.State)
                .Include(x => x.Country)
                .Include(x => x.Jurisdiction)
                .Include(x => x.BusinessType)
                .Include(x => x.CounterpartyDuns)
                .Include(x => x.CounterpartyRelationships)
                .ThenInclude(x => x.BusinessRelationship)
                where q.Id == entityId
                select q
            ).AsNoTracking().FirstOrDefaultAsync() ?? new Counterparty();

            MainContractEntityDetail entity = new();
            entity.CompanyName = dbItem.Name;
            entity.AddressLine1 = dbItem.AddressLine1;
            entity.AddressLine2 = dbItem.AddressLine2;
            entity.Country = dbItem.Country?.Name;
            entity.City = dbItem.City;
            entity.State = dbItem.State?.Name;
            entity.Zip = dbItem.Zip;
            entity.DunsNum = string.Join(" • ", dbItem.CounterpartyDuns.Select(x => x.Duns).ToList());
            entity.FederalTaxId = dbItem.FederalTaxId;
            entity.Jurisdiction = dbItem.Jurisdiction?.Name;
            entity.BusinessType = dbItem.BusinessType?.Name;
            entity.Relationships = string.Join(" • ", dbItem.CounterpartyRelationships.Select(x => x.BusinessRelationship.Name).ToList());

            return Ok(entity);
        }
    }
}
