using System.Text.RegularExpressions;
using Fast.Logic;
using Fast.Models;
using static Fast.Shared.Logic.Util.String;

namespace Fast.Web.Controllers
{
    [Authorize]
    [ApiController]
    [Route("api/[controller]")]
    public class DealFilterController : ControllerBase
    {
        private readonly AuthorizationHelper authHelper;
        private readonly MyDbContext db;
        public DealFilterController(MyDbContext context)
        {
            authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
            db = context;
        }

        private async Task<bool> CheckUserPermission(PermissionType permissionType)
        {
            var hasPermission = await authHelper.IsAuthorizedAsync(User, "Deal", permissionType);
            var hasAccountingPermission = await authHelper.IsAuthorizedAsync(User, "Deal For Accounting", permissionType);
            return hasPermission || hasAccountingPermission;
        }

        [Route("[action]")]
        public async Task<IActionResult> GetFilters()
        {
            if (!await CheckUserPermission(PermissionType.View))
                return Forbid();

            var userId = Util.GetAppUserId(User);
            DealFilterHelper.AddDefaultFiltersIfNeeded(db, userId);

            var filters = await (
                from q in db.DealFilters
                where q.UserId == userId
                select new
                {
                    q.Id,
                    q.Name,
                    q.IsSelected
                }
            ).ToListAsync();

            return Ok(filters);
        }

        [Route("[action]")]
        public async Task<IActionResult> AddFilter(string name)
        {
            if (!await CheckUserPermission(PermissionType.View))
                return Forbid();

            var userId = Util.GetAppUserId(User);
            var newFilter = new DealFilter
            {
                Name = name,
                UserId = userId,
                IsSelected = false,
                Columns = "DealNum",
                State = DealFilterHelper.GetDefaultStateJson()
            };

            db.DealFilters.Add(newFilter);
            await db.SaveChangesAsync();

            return Ok(newFilter.Id);
        }

        public class FilterSaveValue
        {
            public List<FilterParameterItem> filterValues = new();
            public string[] columnPropsOrdered = Array.Empty<string>();
        }

        [Route("[action]")]
        public async Task<IActionResult> SaveFilter(int? filterId, FilterSaveValue filterSaveValue)
        {
            if (!await CheckUserPermission(PermissionType.View))
                return Forbid();

            if (filterId.HasValue)
            {
                var paramsToRemove = db.DealFilterParameters.Where(x => x.FilterId == filterId).ToList();
                if (paramsToRemove.Any())
                    db.DealFilterParameters.RemoveRange(paramsToRemove);

                var dbParams = FilterParameterItem.ConvertParamItemsToDbParams<DealFilterParameter>(filterSaveValue.filterValues);
                db.DealFilterParameters.AddRange(dbParams);
                bool hasAnyColumnProps = filterSaveValue.columnPropsOrdered.Length > 0;
                var dealFilter = db.DealFilters.Where(x => x.Id == filterId).First();
                var allColumnDisplayInfos = GetDisplayInfos<DealListItem>().ToDictionary(x => x.PropName);
                var updatedColPropsOrdered = new List<string>();

                foreach (var col in filterSaveValue.columnPropsOrdered)
                {
                    var colSplit = col.Split(":");
                    if (colSplit.Length == 2 && (colSplit[1] == "NaN"))
                    {
                        var propName = colSplit[0];
                        var updatedCol = propName + ":" + allColumnDisplayInfos[propName].Width;
                        updatedColPropsOrdered.Add(updatedCol);
                    }
                    else
                    {
                        updatedColPropsOrdered.Add(col);
                    }
                }
                dealFilter.Columns = hasAnyColumnProps ? string.Join(",", updatedColPropsOrdered) : null;
                FixDealFilterStateJson(dealFilter);
                db.SaveChanges();
            }

            var userId = Util.GetAppUserId(User);
            await SetFilterSelectedInternal(userId, filterId);

            return Ok();
        }

        private static void FixDealFilterStateJson(DealFilter dealFilter)
        {
            HashSet<string> columnNames = new();
            if (!string.IsNullOrEmpty(dealFilter.Columns))
            {
                var cols = dealFilter.Columns.Split(",").ToList();
                foreach (var col in cols)
                    columnNames.Add(col.Split(":")[0]);
            }

            List<string> stringsToRemove = new();
            string fieldPattern = @",?{[^{]*""field"":""(.+?)"".*?}";
            Regex regex = new(fieldPattern);
            if (dealFilter.State != null)
            {
                foreach (Match match in regex.Matches(dealFilter.State))
                {
                    string fieldName = match.Groups[1].Value;
                    if (!columnNames.Contains(fieldName))
                        stringsToRemove.Add(match.Value);
                }

                string newState = dealFilter.State;
                foreach (string str in stringsToRemove)
                    newState = newState.Replace(str, "");

                string emptyFilterPattern = @",?{""filters"":\[\].*?}";
                for (int i = 0; i < 3; i++)
                    newState = Regex.Replace(newState, emptyFilterPattern, "");

                string colonCommaPattern = @":,";
                newState = newState.Replace(colonCommaPattern, ":null,");

                string bracketCommaPattern = @"[,";
                newState = newState.Replace(bracketCommaPattern, "[");

                dealFilter.State = newState;
            }
        }

        [Route("[action]")]
        public async Task<IActionResult> GetFilterData(int filterId)
        {
            if (!await CheckUserPermission(PermissionType.View))
                return Forbid();

            List<FilterParameterItem> filterParams = await (
                from q in db.DealFilterParameters
                where q.FilterId == filterId
                select FilterParameterItem.ConvertDbParamToParamItem(q)
            ).ToListAsync();

            var filter = db.DealFilters.Where(x => x.Id == filterId).Select(x => new { x.Columns, x.State }).FirstOrDefault();
            var selectedColumns = new List<string>();
            if (filter != null && !string.IsNullOrWhiteSpace(filter.Columns))
                selectedColumns = filter.Columns.Split(",").ToList();
            var state = filter?.State;

            var result = new { filterParams, selectedColumns, state };

            return Ok(result);
        }

        [Route("[action]")]
        public async Task<IActionResult> GetAllColumnDisplayInfos()
        {
            if (!await CheckUserPermission(PermissionType.View))
                return Forbid();

            List<DisplayInfo> allColumnDisplayInfos = GetDisplayInfos<DealListItem>().ToList();
            return Ok(allColumnDisplayInfos);
        }

        [Route("[action]")]
        public async Task<IActionResult> SetFilterSelected(int? filterId)
        {
            if (!await CheckUserPermission(PermissionType.View))
                return Forbid();

            var userId = Util.GetAppUserId(this.User);
            await SetFilterSelectedInternal(userId, filterId);
            return Ok();
        }

        private async Task SetFilterSelectedInternal(int userId, int? filterId)
        {
            if (filterId.HasValue)
            {
                var filters = db.DealFilters.Where(x => x.UserId == userId).ToList();
                bool isDistributionView = filters.Where(x => x.Id == filterId).First().Name == "Distribution View";
                if (!isDistributionView)
                {
                    foreach (var filter in filters)
                    {
                        if (filter.Id == filterId)
                            filter.IsSelected = true;
                        else
                            filter.IsSelected = false;
                    }

                    await db.SaveChangesAsync();
                }
            }
        }

        [Route("[action]")]
        public async Task<IActionResult> RenameFilter(int id, string name)
        {
            if (!await CheckUserPermission(PermissionType.View))
                return Forbid();
                
            db.DealFilters.Where(x => x.Id == id).First().Name = name;
            db.SaveChanges();

            return Ok(name);
        }

        [Route("[action]/{id}")]
        public async Task<IActionResult> DeleteFilter(int id)
        {
            if (!await CheckUserPermission(PermissionType.View))
                return Forbid();
            var item = db.DealFilters.Where(x => x.Id == id).First();
            db.Entry(item).State = EntityState.Deleted;
            db.SaveChanges();

            return Ok();
        }
    }
}
