using System.Collections.Concurrent;
using System.Diagnostics;
using System.Net.Mail;
using System.Xml.Schema;
using Fast.Logic;
using Fast.Models;
using Fast.Shared.Logic.FileService;
using Fast.Shared.Logic.Valuation;
using Microsoft.AspNetCore.OData.Routing.Controllers;
using Microsoft.Extensions.FileProviders;
using Serilog;
using static Fast.Logic.DealHelper;
using static Fast.Shared.Logic.Util.String;
using DealDetail = Fast.Models.DealDetail;

namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class DealController : ODataController
{
    private readonly MyDbContext db;
    private readonly FileService fileService;
    private readonly string templatesFolderName = "DealTemplates";
    private readonly string confirmsFolderName = "DealConfirms";
    private readonly string ticketsFolderName = "DealTickets";
    private readonly string signaturesFolderName = "Signatures";
    private readonly string logosFolderName = "Logos";
    private readonly AuthorizationHelper authHelper;
    private readonly IWebHostEnvironment env;

    public DealController(MyDbContext context, IWebHostEnvironment env)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
        this.env = env;

        var config = new FileServiceConfig(env.ContentRootPath);
        fileService = new FileService(config);
    }

    private async Task<bool> HasViewPermission()
    {
        var hasDealPermission = await authHelper.IsAuthorizedAsync(User, "Deal", PermissionType.View);
        var hasAccountingPermission = await authHelper.IsAuthorizedAsync(User, "Deal For Accounting", PermissionType.View);
        if (hasDealPermission || hasAccountingPermission)
            return true;

        return false;
    }

    private async Task<bool> HasModifyPermission(SaveType saveType, int? newDealPurpose, int? savedDealPurpose)
    {
        bool hasSavePermission = await HasDealSavePermission(saveType, newDealPurpose, savedDealPurpose);
        return hasSavePermission;
    }

    [Route("/odata/GetDealItems")]
    public async Task<IActionResult> GetDealItems(ODataQueryOptions<DealListItem> queryOptions, int filterId, bool isExport)
    {
        if (!await HasViewPermission())
            return Forbid();

        var dealFilterHelper = new DealFilterHelper(db, filterId);
        queryOptions = Util.GetQueryOptionsWithTicketSort(queryOptions, out bool hasDealNumFilter);
        queryOptions = Util.GetQueryOptionsWithConvertedDates(queryOptions);
        var columnNames = GetColumnsNames(filterId);
        var itemsQueryable = GetDealItemsInternal(columnNames);
        itemsQueryable = await dealFilterHelper.ApplyDealFilters(itemsQueryable, hasDealNumFilter);
        itemsQueryable = queryOptions.ApplyTo(itemsQueryable) as IQueryable<DealListItem>;
        var items = itemsQueryable == null ? null : await itemsQueryable.ToListAsync();

        if (isExport)
        {
            var ms = Stopwatch.StartNew();
            var value = File(Util.Excel.GetExportFileStream(items, columnNames), "application/octet-stream");
            Debug.WriteLine($"Deal export: {ms.ElapsedMilliseconds} ms ({items?.Count ?? 0} items)");
            return value;
        }
        else
            return Ok(items);
    }

    private List<string> GetColumnsNames(int filterId)
    {
        string columns = db.DealFilters.Where(x => x.Id == filterId).Select(x => x.Columns ?? "").First();
        var columnNames = new List<string>();
        if (columns != null && columns.Length > 0)
        {
            var cols = columns.Split(",").ToList();
            foreach (var col in cols)
                columnNames.Add(col.Split(":")[0]);
        }

        return columnNames;
    }

    private IQueryable<DealListItem> GetDealItemsInternal(List<string> columnNames)
    {
        IQueryable<DealListItem>? itemsQueryable = null;
        HashSet<string> columnsHashSet = columnNames.ToHashSet();

        itemsQueryable = (
            from d in db.Deals
            join dc in db.VwDealConfirmationInfos on d.TicketNum equals dc.TicketNum into j1
            from dc in j1.DefaultIfEmpty()
            join dd in db.VwDealOverviewInfos on d.Id equals dd.DealId into j2
            from dd in j2.DefaultIfEmpty()
            select new DealListItem
            {
                Id = d.Id,
                DealNum = columnsHashSet.Contains("DealNum") ? d.TicketNum ?? "" : "",
                TicketNum = d.TicketNum ?? "",
                TicketNumber = d.TicketNumber,
                TicketPrefix = d.TransactionType == null ? "" : d.TransactionType.DealTicketPrefix ?? "",
                AccountingMonth = columnsHashSet.Contains("AccountingMonth") ? d.AccountingMonth : null,
                Basis = columnsHashSet.Contains("Basis") ? d.Basis : null,
                Book = columnsHashSet.Contains("Book") && d.Book != null ? d.Book.Name ?? "" : "",
                BookId = d.BookId,
                Broker = columnsHashSet.Contains("Broker") && d.Broker != null ? d.Broker.Name ?? "" : "",
                BrokerId = d.BrokerId,
                BrokerAccount = columnsHashSet.Contains("BrokerAccount") && d.BrokerAccount != null ? d.BrokerAccount.Name ?? "" : "",
                BrokerAccountId = d.BrokerAccountId,
                BuySell = columnsHashSet.Contains("BuySell") && d.BuyButtonNavigation != null ? d.BuyButtonNavigation.Name ?? "" : "",
                BuyButton = d.BuyButton,
                Contact = columnsHashSet.Contains("Contact") && d.Contact != null ? d.Contact.DisplayName : "",
                Contracts = columnsHashSet.Contains("Contracts") ? d.NumOfContracts : null,
                Counterparty = columnsHashSet.Contains("Counterparty") && d.Counterparty != null ? d.Counterparty.Name : "",
                CounterpartyId = d.CounterpartyId,
                DealPurpose = columnsHashSet.Contains("DealPurpose") && d.DealPurpose != null ? d.DealPurpose.Name ?? "" : "",
                DealPurposeId = d.DealPurposeId,
                DealStatus = columnsHashSet.Contains("DealStatus") && d.DealStatus != null ? d.DealStatus.Name ?? "" : "",
                DealStatusId = d.DealStatusId,
                DealType = columnsHashSet.Contains("DealType") && d.PhysicalDealType != null ? d.PhysicalDealType.Name ?? "" : "",
                DealTypeId = d.PhysicalDealTypeId,
                EndDate = columnsHashSet.Contains("EndDate") ? d.EndDate : null,
                FixedPrice = columnsHashSet.Contains("FixedPrice") ? (d.FixedPriceButton == 1 ? d.FixedPrice : null) : null,
                ForceMajeure = columnsHashSet.Contains("ForceMajeure") ? d.FmLanguage == null || d.FmLanguage == "" ? "No" : "Yes" : "",
                Hypothetical = columnsHashSet.Contains("Hypothetical") ? d.HypotheticalId == 0 ? "Real" : "Hypohetical" : "",
                HypotheticalId = d.HypotheticalId,
                InternalEntity = columnsHashSet.Contains("InternalEntity") && d.InternalEntity != null ? d.InternalEntity.Name : "",
                InternalEntityId = d.InternalEntityId,
                Pipeline = columnsHashSet.Contains("Pipeline") && d.Pipeline != null ? d.Pipeline.Name ?? "" : "",
                PipelineId = d.PipelineId,
                PipelineSourceDelivery = columnsHashSet.Contains("PipelineSourceDelivery") && d.PipelineSourceDelivery != null ? d.PipelineSourceDelivery.Name ?? "" : "",
                PipelineSourceDeliveryId = d.PipelineSourceDeliveryId,
                Point = columnsHashSet.Contains("Point") && d.Point != null ? d.Point.Name ?? "" : "",
                PointSourceDelivery = columnsHashSet.Contains("PointSourceDelivery") ? dd.DealPoints ?? "" : "",
                PointId = d.PointId,
                Portfolio = columnsHashSet.Contains("Portfolio") && d.Portfolio != null ? d.Portfolio.Name ?? "" : "",
                PortfolioId = d.PortfolioId,
                PremDisc = columnsHashSet.Contains("PremDisc") ? d.PremiumOrDiscount : null,
                PriceIndex = columnsHashSet.Contains("PriceIndex") && d.PriceIndex != null ? d.PriceIndex.Name ?? "" : "",
                PriceIndexId = d.PriceIndexId,
                Product = columnsHashSet.Contains("Product") ? d.Product.Name : "",
                ProductId = d.ProductId,
                Region = columnsHashSet.Contains("Region") && d.Region != null ? d.Region.Name ?? "" : "",
                RegionId = d.RegionId,
                StartDate = columnsHashSet.Contains("StartDate") ? d.StartDate : null,
                Strategy = columnsHashSet.Contains("Strategy") && d.Strategy != null ? d.Strategy.Name ?? "" : "",
                StrategyId = d.StrategyId,
                Trader = columnsHashSet.Contains("Trader") && d.Trader != null ? d.Trader.DisplayName ?? "" : "",
                TraderId = d.TraderId,
                TradeDate = columnsHashSet.Contains("TradeDate") ? d.TradingDate : null,
                TransactionType = columnsHashSet.Contains("TransactionType") && d.TransactionType != null ? d.TransactionType.Name ?? "" : "",
                TransactionTypeId = d.TransactionTypeId,
                VariableVolume = columnsHashSet.Contains("VariableVolume") ? d.IsVariableVolume ? "Yes" : "No" : "",
                Volume = columnsHashSet.Contains("Volume") ? d.Volume : null,
                SendEmail = columnsHashSet.Contains("SendEmail") ? dc.SendEmail ?? "" : "",
                EmailAddress = columnsHashSet.Contains("EmailAddress") ? dc.EmailAddress ?? "" : "",
                EmailDate = columnsHashSet.Contains("EmailDate") ? dc.EmailDate : null,
                EmailedBy = columnsHashSet.Contains("EmailedBy") ? dc.EmailedBy ?? "" : "",
                SendFax = columnsHashSet.Contains("SendFax") ? dc.SendFax ?? "" : "",
                FaxNumber = columnsHashSet.Contains("FaxNumber") ? dc.FaxNumber ?? "" : "",
                FaxDate = columnsHashSet.Contains("FaxDate") ? dc.FaxDate : null,
                FaxedBy = columnsHashSet.Contains("FaxedBy") ? dc.FaxedBy ?? "" : "",
                Created = d.Created.GetValueOrDefault(),
                Modified = d.Modified,
                Distributed = dc.Distributed
            }
        ).AsNoTracking();

        return itemsQueryable;
    }

    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData()
    {
        if (!await HasViewPermission())
            return Forbid();

        var userId = Util.GetAppUserId(User);
        var sw = Stopwatch.StartNew();
        bool hasDealModifyPermission = false;
        bool hasDealForAccountingPermission = false;
        bool hasDealSaveSourceDelivery = false;
        List<ContactInfoItem>? contacts = null;
        List<DisplayInfo>? allColumnDisplayInfos = null;
        List<DealFilter>? filters = null;
        List<BookInfo>? books = null;
        List<IdNameActive>? regions = null, strategies = null, portfolios = null;
        List<DealHelper.CounterpartyInfo>? counterparties = null;
        List<IdNameActive>? internalEntities = null;
        List<TraderInfo>? traders = null;
        List<IdName>? dealStatuses = null;
        List<IdName>? dealPurposes = null;
        List<IdName>? dealTypes = null;
        List<IndexInfo>? priceIndexes = null;
        List<PipeInfo>? pipelines = null;
        List<PipeInfo>? pipelineSourceDeliveries = null;
        List<DealHelper.PointInfo>? points = null;
        List<DealHelper.PointInfo>? pointSourceDeliveries = null;
        List<DealHelper.ContractInfo>? contracts = null;
        List<IdName>? volumeTypes = null;
        List<IdName>? fuelCalculationTypes = null;
        List<ForceMajeureInfo>? forceMajeures = null;
        List<IdName>? transactionTypes = null;
        List<IdName>? products = null;
        List<IdName>? brokers = null;
        List<BrokerAccountInfo>? brokerAccounts = null;
        List<IdName>? deliveryMethods = null;
        int dealModifyTimeLag = 0;
        var paidOnTypes = new List<IdName>
        {
            new() { Id = (int)Enums.PaidOnType.Actuals, Name = Enums.PaidOnType.Actuals.ToString() },
            new() { Id = (int)Enums.PaidOnType.Nominations, Name = Enums.PaidOnType.Nominations.ToString() }
        };

        List<Task> tasks;
        tasks = new List<Task>
            {
                Task.Run(async () => { hasDealModifyPermission = await authHelper.IsAuthorizedAsync(User, "Deal", PermissionType.Modify); }),
                Task.Run(async () => { hasDealForAccountingPermission = await authHelper.IsAuthorizedAsync(User, "Deal For Accounting", PermissionType.Modify); }),
                Task.Run(() => { allColumnDisplayInfos = GetDisplayInfos<DealListItem>(); }),
                Task.Run(() => { filters = GetFilters(userId); }),
                Task.Run(() => { regions = GetRegions(); }),
                Task.Run(() => { strategies = GetStrategies(); }),
                Task.Run(() => { portfolios = GetPortfolios(); }),
                Task.Run(() => { books = GetBookInfos(); }),
                Task.Run(() => { counterparties = GetCounterparties(); }),
                Task.Run(() => { internalEntities = GetInternalEntities(); }),
                Task.Run(() => { contacts = DealHelper.GetTradingContacts(); }),
                Task.Run(() => { traders = GetTraders(); }),
                Task.Run(() => { dealStatuses = GetDealStatuses(); }),
                Task.Run(() => { dealPurposes = GetDealPurposes(); }),
                Task.Run(() => { dealTypes = GetDealTypes(); }),
                Task.Run(() => { priceIndexes = GetPriceIndexes(); }),
                Task.Run(async () => { pipelines = await DataHelper.GetPipelinesAsync(true); }),
                Task.Run(() => { points = GetPoints(); }),
                Task.Run(() => { contracts = GetContracts(); }),
                Task.Run(() => { volumeTypes = GetVolumeTypes(); }),
                Task.Run(() => { fuelCalculationTypes = GetFuelCalculationTypes(); }),
                Task.Run(() => { forceMajeures = GetForceMajeures(); }),
                Task.Run(() => { dealModifyTimeLag = GetDealModifyTimeLag(); }),
                Task.Run(() => { transactionTypes = GetTransactionTypes(); }),
                Task.Run(async () => { products = await DataHelper.GetProductsAsync(); }),
                Task.Run(() => { brokers = GetBrokers(); }),
                Task.Run(() => { brokerAccounts = GetBrokerAccounts(); }),
                Task.Run(() => { deliveryMethods = GetDeliveryMethods(); })
            };
        await Task.WhenAll(tasks);

        hasDealSaveSourceDelivery = hasDealModifyPermission || hasDealForAccountingPermission || await authHelper.IsAuthorizedAsync(User, "Deal Save Source/Delivery", PermissionType.Standard);

        tasks = new List<Task>
            {
                Task.Run(() => { pipelineSourceDeliveries = pipelines.Copy(); }),
                Task.Run(() => { pointSourceDeliveries = points.Copy(); })
            };
        await Task.WhenAll(tasks);

        var seconds = sw.Elapsed.TotalSeconds;

        var result = new
        {
            hasDealModifyPermission,
            hasDealForAccountingPermission,
            hasDealSaveSourceDelivery,
            allColumnDisplayInfos,
            filters,
            regions,
            strategies,
            portfolios,
            books,
            counterparties,
            internalEntities,
            contacts,
            traders,
            dealStatuses,
            dealPurposes,
            dealTypes,
            priceIndexes,
            pipelines,
            points,
            pipelineSourceDeliveries,
            pointSourceDeliveries,
            contracts,
            volumeTypes,
            fuelCalculationTypes,
            forceMajeures,
            dealModifyTimeLag,
            transactionTypes,
            products,
            brokers,
            brokerAccounts,
            deliveryMethods,
            paidOnTypes
        };
        return Ok(result);
    }

    [Route("[action]")]
    public async Task<IActionResult> GetContacts()
    {
        if (!await HasViewPermission())
            return Forbid();

        var result = DealHelper.GetTradingContacts();
        return Ok(result);
    }

    [Route("[action]")]
    public async Task<IActionResult> GetDealDetail(int dealId)
    {
        if (!await HasViewPermission())
            return Forbid();

        char[] separator = { ',' };

        var dealDetail = await (
            from d in db.Deals
            where d.Id == dealId
            select new DealDetail
            {
                Id = d.Id,
                DealNum = d.TicketNum ?? "",
                AccountingMonth = d.AccountingMonth,
                Basis = d.Basis,
                BookId = d.BookId,
                BrokerAccountId = d.BrokerAccountId,
                BrokerId = d.BrokerId,
                IsBuy = d.BuyButton == 1,
                Comments = d.Comments ?? "",
                ContactId = d.ContactId,
                NumOfContracts = d.NumOfContracts,
                CostBasisInc = d.CostBasisInc,
                CostFuelCalculationTypeId = d.CostFuelCalculationTypeId ?? 2,
                CostPremInc = d.CostPremInc,
                CounterpartyId = d.CounterpartyId,
                DealPurposeId = d.DealPurposeId,
                DealStatusId = d.DealStatusId,
                EndDate = d.EndDate,
                FixedPrice = d.FixedPriceButton == 1 ? d.FixedPrice : null,
                IsFixedPrice = d.FixedPriceButton == 1,
                Fmlanguage = d.FmLanguage ?? "",
                HedgeFee = d.HedgeFee,
                HypotheticalId = d.HypotheticalId,
                InternalEntityId = d.InternalEntityId,
                InternalMemo = d.InternalMemo ?? "",
                IsVariableVolume = d.IsVariableVolume,
                ContractNumber = d.NettingContractNumber ?? "",
                DealTypeId = d.PhysicalDealTypeId,
                PipelineId = d.PipelineId,
                PipelineSourceDeliveryId = d.PipelineSourceDeliveryId,
                PointId = d.PointId,
                PortfolioId = d.PortfolioId,
                PremiumOrDiscount = d.PremiumOrDiscount,
                PriceIndexId = d.PriceIndexId,
                PriceIndexId2 = d.PriceIndexId2,
                RegionId = d.RegionId,
                StartDate = d.StartDate,
                StrategyId = d.StrategyId,
                TraderId = d.TraderId,
                TradingDate = d.TradingDate,
                TransactionTypeId = d.TransactionTypeId,
                UpdateNotes = d.UpdateNotes ?? "",
                Volume = d.Volume,
                VolumeTypeId = d.VolumeTypeId,
                WaspNum = d.WaspNum ?? "",
                Created = d.Created == null ? null : d.Created.Value.ToLocalTime(),
                Modified = d.Modified == null ? null : d.Modified.Value.ToLocalTime(),
                VolumeC2 = d.VolumeC2,
                VolumeC3 = d.VolumeC3,
                VolumeIc4 = d.VolumeIc4,
                VolumeNc4 = d.VolumeNc4,
                VolumeC5p = d.VolumeC5P,
                FixedPriceC2 = d.FixedPriceC2,
                FixedPriceC3 = d.FixedPriceC3,
                FixedPriceIc4 = d.FixedPriceIc4,
                FixedPriceNc4 = d.FixedPriceNc4,
                FixedPriceC5p = d.FixedPriceC5P,
                PriceIndexIdC2 = d.PriceIndexIdC2,
                PriceIndexIdC3 = d.PriceIndexIdC3,
                PriceIndexIdIc4 = d.PriceIndexIdIc4,
                PriceIndexIdNc4 = d.PriceIndexIdNc4,
                PriceIndexIdC5p = d.PriceIndexIdC5P,
                BasisC2 = d.BasisC2,
                BasisC3 = d.BasisC3,
                BasisIc4 = d.BasisIc4,
                BasisNc4 = d.BasisNc4,
                BasisC5p = d.BasisC5P,
                PremDiscC2 = d.PremDiscC2,
                PremDiscC3 = d.PremDiscC3,
                PremDiscIc4 = d.PremDiscIc4,
                PremDiscNc4 = d.PremDiscNc4,
                PremDiscC5p = d.PremDiscC5P,
                ProductId = d.ProductId,
                InSpecMarketingFee = d.InSpecMarketingFee,
                InSpecMarketingFeeTypeId = d.InSpecMarketingFeeTypeId,
                OutSpecMarketingFee = d.OutSpecMarketingFee,
                OutSpecMarketingFeeTypeId = d.OutSpecMarketingFeeTypeId,
                SuperiorFee = d.SuperiorFee,
                SuperiorFeeTypeId = d.SuperiorFeeTypeId,
                DeliveryModeId = d.DeliveryModeId,
                PaidOnId = d.PaidOnId,
                IsNetback = d.IsNetback,
                DeductTransport = d.DeductTransport
            }
        ).AsNoTracking().FirstAsync();

        var noteInfo = await (
            from d in db.Deals
            where d.Id == dealId
            select new
            {
                Created = d.Created.GetValueOrDefault().ToLocalTime(),
                CreatedName = d.CreatedByNavigation == null ? "" : d.CreatedByNavigation.DisplayName,
                Modified = d.Modified.GetValueOrDefault().ToLocalTime(),
                ModifiedName = d.ModifiedByNavigation == null ? "" : d.ModifiedByNavigation.DisplayName
            }
        ).AsNoTracking().FirstAsync();

        dealDetail.UpdateNotes = GetUpdateNotes(noteInfo.Created, noteInfo.CreatedName, noteInfo.Modified, noteInfo.ModifiedName);

        if (dealDetail.IsVariableVolume)
        {
            var dealVolumes = await (
                           from d in db.DealVolumes
                           where d.DealId == dealId
                           select new DealDetailVolume
                           {
                               PhysicalVolume = d.PhysicalVolume,
                               StartDate = d.StartDate
                           }
                        ).AsNoTracking().ToListAsync();

            dealDetail.DealVolumes = dealVolumes;
        }

        var sourceDeliveryPointItems = await (
            from d in db.PointSourceDeliveries
            where d.DealId == dealId
            select new SourceDeliveryPointItem
            {
                DealId = d.DealId,
                PointId = d.PointId,
                PointVolume = d.PointVolume ?? 0,
                PointName = d.Point.Name ?? "",

            }
        ).AsNoTracking().ToListAsync() ?? new List<SourceDeliveryPointItem>();

        foreach (var item in sourceDeliveryPointItems)
        {
            var newItem = new SourceDeliveryPointItem();
            newItem.DealId = item.DealId;
            newItem.PointId = item.PointId;
            newItem.PointName = item.PointName ?? "";
            newItem.PointVolume = item.PointVolume;
            dealDetail.SourceDeliveryPointItems.Add(newItem);
        }

        return Ok(dealDetail);
    }

    public enum SaveType
    {
        New = 1,
        Normal = 2
    }

    public class SaveDealResult
    {
        public int DealId;
        public string? DealNum;
    }

    [Route("[action]")]
    public async Task<IActionResult> UniqueDealCheck(DealDetail dealDetail, SaveType saveType)
    {
        int newDealPurpose = dealDetail.DealPurposeId.GetValueOrDefault();
        int? savedDealPurpose = await db.Deals.Where(d => d.Id == dealDetail.Id).Select(d => d.DealPurposeId).FirstOrDefaultAsync();
        if (!await HasModifyPermission(saveType, newDealPurpose, savedDealPurpose))
            return Forbid();

        string? ticketNums = null;

        if ((Enums.TransactionType)dealDetail.TransactionTypeId.GetValueOrDefault() != Enums.TransactionType.PhysicalNGL)
        {
            var buyButton = dealDetail.IsBuy ? 1 : -1;
            var fixedPriceButton = dealDetail.IsFixedPrice ? 1 : 0;

            var similarDeals = (
                from d in db.Deals
                where d.BuyButton.HasValue && d.BuyButton.Value == buyButton &&
                d.CounterpartyId.GetValueOrDefault() == dealDetail.CounterpartyId.GetValueOrDefault() &&
                d.PipelineId.GetValueOrDefault() == dealDetail.PipelineId.GetValueOrDefault() &&
                d.PointId.GetValueOrDefault() == dealDetail.PointId.GetValueOrDefault() &&
                d.StartDate == dealDetail.StartDate &&
                d.EndDate == dealDetail.EndDate &&
                d.NumOfContracts.GetValueOrDefault() == dealDetail.NumOfContracts.GetValueOrDefault() &&
                d.Volume.GetValueOrDefault() == dealDetail.Volume.GetValueOrDefault() &&
                d.FixedPriceButton.GetValueOrDefault() == fixedPriceButton &&
                d.FixedPrice.GetValueOrDefault() == dealDetail.FixedPrice.GetValueOrDefault() &&
                d.Basis.GetValueOrDefault() == dealDetail.Basis.GetValueOrDefault() &&
                d.PriceIndexId.GetValueOrDefault() == dealDetail.PriceIndexId.GetValueOrDefault() &&
                d.PriceIndexId2.GetValueOrDefault() == dealDetail.PriceIndexId2.GetValueOrDefault() &&
                d.PremiumOrDiscount.GetValueOrDefault() == dealDetail.PremiumOrDiscount.GetValueOrDefault() &&
                (saveType == SaveType.New || (saveType == SaveType.Normal && d.Id != dealDetail.Id))
                select d.TicketNum
            ).ToList();

            if (similarDeals.Any())
                ticketNums = string.Join(", ", similarDeals);
        }

        return Ok(ticketNums);
    }

    //only requires the view permission but the appropriate save security actions are checked within the method
    [Route("[action]")]
    public async Task<IActionResult> SaveSourceDelivery(DealDetail dealDetail)
    {
        var hasDealSaveSourceDelivery = await authHelper.IsAuthorizedAsync(User, "Deal Save Source/Delivery", PermissionType.Standard);
        int? savedDealPurpose = await db.Deals.Where(d => d.Id == dealDetail.Id).Select(d => d.DealPurposeId).FirstOrDefaultAsync();
        var hasModifyPermission = await HasModifyPermission(SaveType.Normal, null, savedDealPurpose);

        //if we don't have both permissions then throw an error but if we have at least one of them then continue
        if (!hasDealSaveSourceDelivery && !hasModifyPermission)
            return Forbid();

        var dbItem = await (
            from q in db.Deals
                .Include(x => x.PointSourceDeliveries)
            where q.Id == dealDetail.Id
            select q
        ).FirstAsync();
        var d = dealDetail;

        db.PointSourceDeliveries.RemoveRange(dbItem.PointSourceDeliveries);

        if (dealDetail.SourceDeliveryPointItems.Count != 0 && d.TransactionTypeId == 1 || d.TransactionTypeId == 11)
        {
            foreach (var pointItem in dealDetail.SourceDeliveryPointItems)
            {
                var pointsItemToSave = new PointSourceDelivery
                {
                    DealId = dealDetail.Id,
                    PointVolume = pointItem.PointVolume,
                    PointId = pointItem.PointId
                };
                db.PointSourceDeliveries.Add(pointsItemToSave);
            }
        }

        dbItem.PipelineSourceDeliveryId = d.TransactionTypeId == 1 || d.TransactionTypeId == 11 ? d.PipelineSourceDeliveryId : null;

        await db.SaveChangesAsync();
        return Ok();
    }


    [Route("[action]")]
    public async Task<IActionResult> SaveDealDetail(DealDetail dealDetail, SaveType saveType)
    {
        int newDealPurpose = dealDetail.DealPurposeId.GetValueOrDefault();
        int? savedDealPurpose = await db.Deals.Where(d => d.Id == dealDetail.Id).Select(d => d.DealPurposeId).FirstOrDefaultAsync();
        if (!await HasModifyPermission(saveType, newDealPurpose, savedDealPurpose))
            return Forbid();

        int resultId = 0;
        string ticketNum = "";
        int userId = 0;

        await db.Database.CreateExecutionStrategy().Execute(async () =>
        {
            using var dbContextTransaction = await db.Database.BeginTransactionAsync();
            Deal? dbItem = null;
            userId = Util.GetAppUserId(User);
            var time = DateTime.UtcNow;

            var d = dealDetail;
            bool hasDealNum = !string.IsNullOrWhiteSpace(d.DealNum);


            if (saveType != SaveType.New)
            {
                dbItem = (
                    from q in db.Deals
                        .Include(x => x.DealVolumes)
                        .Include(x => x.PointSourceDeliveries)
                    where q.Id == dealDetail.Id
                    select q
                ).First();

                if (hasDealNum)
                    dbItem.TicketNum = dealDetail.DealNum ?? "";
                else
                    dbItem.TicketNum = await Util.GetNewDealNumAsync(dealDetail.TransactionTypeId.GetValueOrDefault(), db);

                dbItem.TraderId = dealDetail.TraderId;
                dbItem.TradingDate = dealDetail.TradingDate;
            }
            if (dbItem == null) //if the item does not exist then add it
            {
                dbItem = new Deal
                {
                    CreatedBy = userId,
                    Created = time,
                    TraderId = userId,
                    TradingDate = DateOnly.FromDateTime(time.Date),
                    ProductId = d.ProductId,
                };
                dbItem.TicketNum = await Util.GetNewDealNumAsync(dealDetail.TransactionTypeId.GetValueOrDefault(), db);
                db.Deals.Add(dbItem);
                await db.SaveChangesAsync();
            }
            else
            {
                //remove existing items so that they get completely re-inserted
                db.DealVolumes.RemoveRange(dbItem.DealVolumes);
                db.PointSourceDeliveries.RemoveRange(dbItem.PointSourceDeliveries);
            }

            if (d.TransactionTypeId.GetValueOrDefault() == (int)Enums.TransactionType.PhysicalGas)
                dbItem.DeliveryModeId = null;

            dbItem.ProductId = d.ProductId;
            dbItem.AccountingMonth = d.AccountingMonth;
            dbItem.Basis = d.Basis;
            dbItem.BookId = d.BookId;
            dbItem.BrokerAccountId = d.BrokerAccountId;
            dbItem.BrokerId = d.BrokerId;
            dbItem.BuyButton = d.IsBuy ? 1 : -1;
            dbItem.Comments = d.Comments;
            dbItem.ContactId = null;
            dbItem.ContactId = d.ContactId;
            dbItem.NumOfContracts = d.NumOfContracts;
            dbItem.CostBasisInc = d.CostBasisInc;
            dbItem.CostFuelCalculationTypeId = d.CostFuelCalculationTypeId ?? 2;
            dbItem.CostPremInc = d.CostPremInc;
            dbItem.CounterpartyId = d.CounterpartyId;
            dbItem.DealPurposeId = d.DealPurposeId;
            dbItem.DealStatusId = d.DealStatusId;
            dbItem.EndDate = d.EndDate;
            dbItem.FixedPrice = d.IsFixedPrice ? d.FixedPrice : null;
            dbItem.FixedPriceButton = d.IsFixedPrice ? 1 : 0;
            dbItem.FmLanguage = saveType == SaveType.New && hasDealNum ? null : d.Fmlanguage;
            dbItem.HedgeFee = d.TransactionTypeId == 3 && d.DealPurposeId == 9 ? d.HedgeFee : null;
            dbItem.HypotheticalId = d.HypotheticalId;
            dbItem.InternalEntityId = d.InternalEntityId;
            dbItem.InternalMemo = d.InternalMemo;

            //per Bryce via Teams chat on 2022-07-26
            //If we're doing a "save new" on an existing deal then variable volume should automatically be set to "no" on the new deal
            if (saveType == SaveType.New && hasDealNum)
                dbItem.IsVariableVolume = false;
            else
                dbItem.IsVariableVolume = d.IsVariableVolume;

            dbItem.NettingContractNumber = string.IsNullOrWhiteSpace(d.ContractNumber) ? null : d.ContractNumber;
            dbItem.PhysicalDealTypeId = d.DealTypeId;
            dbItem.PipelineId = d.PipelineId;
            dbItem.PipelineSourceDeliveryId = d.TransactionTypeId == 1 || d.TransactionTypeId == 11 ? d.PipelineSourceDeliveryId : null;
            dbItem.PointId = d.PointId;
            dbItem.PortfolioId = d.PortfolioId;
            dbItem.PremiumOrDiscount = d.PremiumOrDiscount;
            dbItem.PriceIndexId = !d.IsFixedPrice ? d.PriceIndexId : null;
            dbItem.PriceIndexId2 = !d.IsFixedPrice ? d.PriceIndexId2 : null;
            dbItem.RegionId = d.RegionId;
            dbItem.StartDate = d.StartDate;
            dbItem.StrategyId = d.StrategyId;
            dbItem.TransactionTypeId = d.TransactionTypeId;
            dbItem.Volume = d.Volume;
            dbItem.VolumeTypeId = d.VolumeTypeId;
            dbItem.WaspNum = d.TransactionTypeId == 3 && d.DealPurposeId == 9 ? d.WaspNum : null;
            dbItem.VolumeC2 = d.VolumeC2;
            dbItem.VolumeC3 = d.VolumeC3;
            dbItem.VolumeIc4 = d.VolumeIc4;
            dbItem.VolumeNc4 = d.VolumeNc4;
            dbItem.VolumeC5P = d.VolumeC5p;
            dbItem.FixedPriceC2 = d.FixedPriceC2;
            dbItem.FixedPriceC3 = d.FixedPriceC3;
            dbItem.FixedPriceIc4 = d.FixedPriceIc4;
            dbItem.FixedPriceNc4 = d.FixedPriceNc4;
            dbItem.FixedPriceC5P = d.FixedPriceC5p;
            dbItem.PriceIndexIdC2 = d.PriceIndexIdC2;
            dbItem.PriceIndexIdC3 = d.PriceIndexIdC3;
            dbItem.PriceIndexIdIc4 = d.PriceIndexIdIc4;
            dbItem.PriceIndexIdNc4 = d.PriceIndexIdNc4;
            dbItem.PriceIndexIdC5P = d.PriceIndexIdC5p;
            dbItem.BasisC2 = d.BasisC2;
            dbItem.BasisC3 = d.BasisC3;
            dbItem.BasisIc4 = d.BasisIc4;
            dbItem.BasisNc4 = d.BasisNc4;
            dbItem.BasisC5P = d.BasisC5p;
            dbItem.PremDiscC2 = d.PremDiscC2;
            dbItem.PremDiscC3 = d.PremDiscC3;
            dbItem.PremDiscIc4 = d.PremDiscIc4;
            dbItem.PremDiscNc4 = d.PremDiscNc4;
            dbItem.PremDiscC5P = d.PremDiscC5p;
            dbItem.InSpecMarketingFee = d.InSpecMarketingFee;
            dbItem.InSpecMarketingFeeTypeId = d.InSpecMarketingFeeTypeId;
            dbItem.OutSpecMarketingFee = d.OutSpecMarketingFee;
            dbItem.OutSpecMarketingFeeTypeId = d.OutSpecMarketingFeeTypeId;
            dbItem.SuperiorFee = d.SuperiorFee;
            dbItem.SuperiorFeeTypeId = d.SuperiorFeeTypeId;
            dbItem.DeliveryModeId = d.TransactionTypeId == 11 ? d.DeliveryModeId : null;
            dbItem.PaidOnId = d.PaidOnId;
            dbItem.IsNetback = d.IsNetback;
            dbItem.DeductTransport = d.DeductTransport;

            dbItem.ModifiedBy = userId;
            dbItem.Modified = time;

            var createdName = (from q in db.AppUsers where q.Id == dbItem.CreatedBy select q.DisplayName).First();
            var modifiedName = (from q in db.AppUsers where q.Id == dbItem.ModifiedBy select q.DisplayName).First();
            dbItem.UpdateNotes = GetUpdateNotes(dbItem.Created.GetValueOrDefault(), createdName, dbItem.Modified.Value, modifiedName);

            d.SourceDeliveryPointItems ??= new List<SourceDeliveryPointItem>();
            foreach (var pointItem in d.SourceDeliveryPointItems)
            {
                var pointsItemToSave = new PointSourceDelivery();
                pointsItemToSave.DealId = dbItem.Id;
                pointsItemToSave.PointVolume = dbItem.IsVariableVolume ? null : pointItem.PointVolume;
                pointsItemToSave.PointId = pointItem.PointId;
                db.PointSourceDeliveries.Add(pointsItemToSave);
            }

            if (dbItem.IsVariableVolume && d.DealVolumes != null)
            {
                foreach (var dealVolume in d.DealVolumes)
                {
                    if (dealVolume.StartDate.HasValue && dealVolume.PhysicalVolume.HasValue)
                    {
                        DateOnly volDate = dealVolume.StartDate.Value;
                        if (d.VolumeTypeId == (int)Enums.VolumeType.Monthly)
                            volDate = Util.Date.FirstDayOfMonth(volDate);
                        var dv = new DealVolume
                        {
                            StartDate = volDate,
                            EndDate = volDate, //EndDate is not in UI
                            PhysicalVolume = dealVolume.PhysicalVolume.Value
                        };
                        dbItem.DealVolumes.Add(dv);
                    }
                }
            }

            db.SaveChanges();

            if (d.TransactionTypeId == (int)Enums.TransactionType.Futures && d.DealPurposeId == (int)Enums.DealPurpose.Trigger && !string.IsNullOrWhiteSpace(d.WaspNum))
                await InsertGasDealForFuturesTrigger(dbItem);

            resultId = dbItem.Id;
            ticketNum = dbItem.TicketNum ?? "";

            await dbContextTransaction.CommitAsync();
        });

        //If accounting deal purpose, generate Tickets but not confirms
        bool isAccountingDeal = IsAccountingDeal(dealDetail.DealPurposeId);
        bool skipConfirms = isAccountingDeal;
        GenerateDealDocuments(dealDetail, ticketNum, userId, skipConfirms);

        var saveResult = new SaveDealResult { DealId = resultId, DealNum = ticketNum };
        return Ok(saveResult);
    }

    private async Task<bool> HasDealSavePermission(SaveType saveType, int? newDealPurpose, int? savedDealPurpose)
    {
        var hasDealModifyPermission = await authHelper.IsAuthorizedAsync(User, "Deal", PermissionType.Modify);
        if (hasDealModifyPermission)
            return true;

        var hasDealForAccountingPermission = await authHelper.IsAuthorizedAsync(User, "Deal For Accounting", PermissionType.Modify);
        var isSaveNew = saveType == SaveType.New;

        if (!hasDealForAccountingPermission)
            return false;

        //User has Deal For Accounting Permission but not Deal Modify Permission
        var isPurposeForAccounting = !IsAccountingDeal(newDealPurpose);
        if (isPurposeForAccounting)
            return false;

        var isModifyNonAccountingDeal = !isSaveNew && !IsAccountingDeal(savedDealPurpose);
        if (isModifyNonAccountingDeal)
            return false;

        return true;
    }

    public static bool IsAccountingDeal(int? dealPurposeId)
    {
        //The new dealPurposeId will be null for deletes, so we need to allow it.
        if (dealPurposeId == null)
            return true;

        int[] AccountingPurposeIds = { (int)Enums.DealPurpose.PTR, (int)Enums.DealPurpose.Cashout, (int)Enums.DealPurpose.LU, (int)Enums.DealPurpose.Imbalance };

        return AccountingPurposeIds.Contains(dealPurposeId.Value);
    }

    private void GenerateDealDocuments(DealDetail dealDetail, string ticketNum, int userId, bool skipConfirms)
    {
        if (string.IsNullOrWhiteSpace(ticketNum))
            return;

        var transactionTypeId = dealDetail.TransactionTypeId.GetValueOrDefault();

        // Capture fileService for use in background task
        var fs = fileService;

        _ = Task.Run(async () =>
        {
            try
            {
                using var db = Main.CreateContext();
                switch (transactionTypeId)
                {
                    case (int)Enums.TransactionType.PhysicalGas:
                        if (!skipConfirms)
                        {
                            var confirmDocGas = new ConfirmDocPhysicalGas(fs, confirmsFolderName, templatesFolderName, signaturesFolderName, logosFolderName, db);
                            await confirmDocGas.Generate(ticketNum, userId);
                        }

                        var ticketDocGas = new TicketDocPhysicalGas(fs, ticketsFolderName, templatesFolderName, signaturesFolderName, logosFolderName, db);
                        await ticketDocGas.Generate(ticketNum, userId);
                        break;

                    case (int)Enums.TransactionType.Futures:
                        var ticketDocFutures = new TicketDocFutures(fs, ticketsFolderName, templatesFolderName, signaturesFolderName, logosFolderName, db);
                        await ticketDocFutures.Generate(ticketNum, userId);
                        break;
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, "Failed to generate deal documents for {TicketNum}", ticketNum);
            }
        });
    }


    private async Task InsertGasDealForFuturesTrigger(Deal futuresDeal)
    {
        Deal? gasDeal = await (
            from q in db.Deals
            where q.TransactionTypeId == (int)Enums.TransactionType.PhysicalGas &&
            q.WaspNum == futuresDeal.WaspNum
            orderby q.Id descending
            select q
        ).FirstOrDefaultAsync();

        if (gasDeal != null)
            return; // gas deal already exists

        WASPHelper waspHelper = new(db, DateOnly.FromDateTime(DateTime.Today));
        var isBuy = futuresDeal.BuyButton == 1;
        var waspVolPriceResult = waspHelper.GetWaspVolAndPrice(futuresDeal.TicketNum ?? "", futuresDeal.WaspNum ?? "", futuresDeal.ProductId, futuresDeal.NumOfContracts, futuresDeal.FixedPrice, futuresDeal.BrokerId, isBuy, futuresDeal.AccountingMonth, futuresDeal.StartDate);

        gasDeal = await GetCopiedDealAsync(futuresDeal, db);
        gasDeal.TransactionTypeId = (int)Enums.TransactionType.PhysicalGas;
        gasDeal.ProductId = (int)Enums.Product.NaturalGas;
        gasDeal.BuyButton = -futuresDeal.BuyButton;
        gasDeal.PhysicalDealTypeId = (int)Enums.DealType.Firm;
        gasDeal.FixedPriceButton = 1;
        gasDeal.NumOfContracts = waspVolPriceResult.Volume;
        gasDeal.Volume = waspVolPriceResult.PhysicalVolume;
        gasDeal.FixedPrice = waspVolPriceResult.Price;
        gasDeal.PipelineSourceDeliveryId = gasDeal.PipelineId;

        if (gasDeal.PointId.HasValue)
            gasDeal.PointSourceDeliveries.Add(new PointSourceDelivery { PointId = gasDeal.PointId.GetValueOrDefault() });

        db.Deals.Add(gasDeal);
        await db.SaveChangesAsync();

        // Capture fileService for use in background task
        var fs = fileService;

        _ = Task.Run(async () =>
        {
            try
            {
                using var db = Main.CreateContext();
                var confirmDocGas = new ConfirmDocPhysicalGas(fs, confirmsFolderName, templatesFolderName, signaturesFolderName, logosFolderName, db);
                var confirmResultGas = await confirmDocGas.Generate(gasDeal.TicketNum, gasDeal.ModifiedBy.GetValueOrDefault());
                var ticketDocGas = new TicketDocPhysicalGas(fs, ticketsFolderName, templatesFolderName, signaturesFolderName, logosFolderName, db);
                var ticketResultGas = await ticketDocGas.Generate(gasDeal.TicketNum, gasDeal.ModifiedBy.GetValueOrDefault());
            }
            catch (Exception ex)
            {
                Log.Error(ex, "Failed to generate trigger documents for {TicketNum}", gasDeal.TicketNum);
            }
        });
    }

    [Route("[action]/{id}")]
    public async Task<IActionResult> DeleteDeal(int id)
    {
        int? savedDealPurpose = await db.Deals.Where(d => d.Id == id).Select(d => d.DealPurposeId).FirstOrDefaultAsync();
        if (!await HasModifyPermission(SaveType.Normal, null, savedDealPurpose))
            return Forbid();

        var gasNomMarkets = db.GasMarkets.Where(x => x.DealId == id).ToList();
        db.GasMarkets.RemoveRange(gasNomMarkets);

        var gasNomSupplies = db.GasSupplies.Where(x => x.DealId == id).ToList();
        db.GasSupplies.RemoveRange(gasNomSupplies);

        var crudeNomMarkets = db.CrudeMarkets.Where(x => x.DealId == id).ToList();
        db.CrudeMarkets.RemoveRange(crudeNomMarkets);

        var crudeNomSupplies = db.CrudeSupplies.Where(x => x.DealId == id).ToList();
        db.CrudeSupplies.RemoveRange(crudeNomSupplies);

        Deal dbItem = db.Deals.Where(x => x.Id == id).First();
        db.Entry(dbItem).State = EntityState.Deleted;

        db.SaveChanges();

        return Ok();
    }

    [Route("[action]")]
    public async Task<IActionResult> SaveGridSettings(int filterId, [FromBody] string stateJson)
    {
        if (!await HasViewPermission())
            return Forbid();

        var filter = db.DealFilters.Where(x => x.Id == filterId).First();
        filter.State = stateJson;
        db.SaveChanges();

        return Ok(filter);
    }

    [Route("[action]")]
    public async Task<IActionResult> DownloadConfirm(string ticketNum)
    {
        if (!await HasViewPermission())
            return Forbid();

        return await DownloadFile(ticketNum, FileType.Confirm);
    }

    [Route("[action]")]
    public async Task<IActionResult> DownloadTicket(string ticketNum)
    {
        if (!await HasViewPermission())
            return Forbid();

        return await DownloadFile(ticketNum, FileType.Ticket);
    }

    [Route("[action]")]
    public async Task<IActionResult> GetWaspNums(int counterpartyId, DateOnly startDate, bool addNewWaspNum)
    {
        if (!await HasViewPermission())
            return Forbid();

        WASPHelper waspHelper = new(db, DateOnly.FromDateTime(DateTime.Today));
        var waspNums = waspHelper.GetWaspNums((int)Enums.DealPurpose.Trigger, counterpartyId, startDate, (int)Enums.TransactionType.Futures, addNewWaspNum);
        return Ok(waspNums);
    }

    [Route("[action]")]
    public async Task<IActionResult> GetWaspVolAndPrice(string dealNum, string waspNum, int productId, int? numOfContracts, double? fixedPrice, int? brokerId, bool isBuy, DateOnly? accountingMonth, DateOnly? startDate)
    {
        if (!await HasViewPermission())
            return Forbid();

        WASPHelper waspHelper = new(db, DateOnly.FromDateTime(DateTime.Today));
        var result = waspHelper.GetWaspVolAndPrice(dealNum, waspNum, productId, numOfContracts, fixedPrice, brokerId, isBuy, accountingMonth, startDate);
        return Ok(result);
    }

    [Permission("Deal", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> Distribute([FromBody] string[] ticketNums)
    {
        ConcurrentBag<string> concurrentMessages = new();
        try
        {
            ticketNums = ticketNums.OrderBy(x => x.TicketSort()).ToArray();
            var userId = Util.GetAppUserId(User);
            var dealConfirms = await (
                from dc in db.DealConfirmations
                join d in db.Deals on dc.TicketNum equals d.TicketNum into j1
                from d in j1.DefaultIfEmpty()
                join dci in db.VwDealConfirmationInfos on d.TicketNum equals dci.TicketNum into j2
                from dci in j2.DefaultIfEmpty()
                where ticketNums.Contains(dc.TicketNum) && dc.ConfirmFile != null
                select new
                {
                    dc.TicketNum,
                    ConfirmFile = dc.ConfirmFile,
                    Counterparty = d.Counterparty == null ? "" : d.Counterparty.Name,
                    dci.SendEmail,
                    dci.SendFax,
                    dci.EmailAddress,
                    dci.FaxNumber,
                    dci.Attention,
                    dci.CombineEmail
                }
            ).ToListAsync();

            var notification = db.Notifications.First(x => x.Id == (int)Enums.NotificationType.Confirmations);

            var groupedEmailItems = (
                from q in dealConfirms
                where q.SendEmail == "yes"
                group q by new { Combine = q.CombineEmail ?? false, q.Counterparty, q.EmailAddress } into g
                select g
            ).ToList();

            var faxItems = dealConfirms.Where(x => x.SendFax == "yes").ToList();
            var noDistItems = dealConfirms.Where(x => x.SendEmail == "no" && x.SendFax == "no").ToList();
            foreach (var item in noDistItems)
                concurrentMessages.Add($"Error processing {item.TicketNum}: contract is not configured for email or fax");

            await Parallel.ForEachAsync(groupedEmailItems, new ParallelOptions { MaxDegreeOfParallelism = 2 }, async (group, _) =>
            {
                var groupTicketNums = string.Join(",", group.Select(x => x.TicketNum));
                try
                {
                    if (string.IsNullOrWhiteSpace(group.Key.EmailAddress))
                    {
                        concurrentMessages.Add($"Error processing {groupTicketNums}: missing email address");
                        return;
                    }

                    List<Tuple<string, string>> localFiles = new();
                    string subFolder = "DealConfirms";
                    string mergeTempFolder = Path.Combine(Path.GetTempPath(), "MergeTemp");
                    Directory.CreateDirectory(mergeTempFolder);

                    foreach (var item in group)
                    {
                        if (string.IsNullOrWhiteSpace(item.ConfirmFile))
                        {
                            concurrentMessages.Add($"Error processing {groupTicketNums}: missing confirm file");
                            return;
                        }

                        var fileResponse = await fileService.DownloadFileAsync(subFolder, item.ConfirmFile);

                        string tempPath = Path.Combine(mergeTempFolder, $"{Guid.NewGuid():N}.pdf");
                        await using (var localFs = new FileStream(tempPath, FileMode.Create, FileAccess.Write))
                            await fileResponse.Stream.CopyToAsync(localFs);

                        localFiles.Add(Tuple.Create(item.TicketNum, tempPath));
                    }

                    // Read all local files into byte arrays for merging
                    var pdfBytesList = new List<(string fileName, byte[] bytes)>();
                    foreach (var (ticketNum, tempPath) in localFiles)
                    {
                        var bytes = await System.IO.File.ReadAllBytesAsync(tempPath);
                        pdfBytesList.Add(($"{ticketNum}.pdf", bytes));
                    }

                    var combinedFileStream = await Shared.Logic.Package.PdfConverter.MergePdfsFromBytes(pdfBytesList);
                    string subject = $"{notification.Subject} - {groupTicketNums}";
                    string body = notification.Body ?? "";
                    string attachName = $"{subject}.pdf";
                    var attachment = new Attachment(combinedFileStream, attachName, "application/pdf");

                    using var newContext = Main.CreateContext();
                    var emailer = new Util.Email(newContext, Enums.NotificationType.Confirmations);
                    string sentTo = emailer.SendEmail(group.Key.EmailAddress, subject, body, attachment);
                    concurrentMessages.Add($"Sent {groupTicketNums} to {sentTo}");

                    foreach (var item in group)
                    {
                        var confirm = newContext.DealConfirmations.First(x => x.TicketNum == item.TicketNum);
                        confirm.EmailedBy = userId;
                        confirm.EmailDate = DateTime.UtcNow;
                    }
                    newContext.SaveChanges();
                }
                catch (Exception ex)
                {
                    concurrentMessages.Add($"Error processing {groupTicketNums}: {GetExceptionMessage(ex)}");
                }
            });

            await Parallel.ForEachAsync(faxItems, new ParallelOptions { MaxDegreeOfParallelism = 2 }, async (faxItem, _) =>
            {
                try
                {
                    if (string.IsNullOrWhiteSpace(faxItem.FaxNumber))
                    {
                        concurrentMessages.Add($"Error processing {faxItem.TicketNum}: missing fax number");
                        return;
                    }

                    string subFolder = "DealConfirms";
                    var fileResponse = await fileService.DownloadFileAsync(subFolder, faxItem.ConfirmFile);
                    string tempFolder = Path.Combine(env.ContentRootPath, "DealConfirms", "MergeTempFax");
                    Directory.CreateDirectory(tempFolder);
                    var tempPath = Path.Combine(tempFolder, $"{Guid.NewGuid():N}.pdf");
                    await using (var fs = new FileStream(tempPath, FileMode.Create, FileAccess.Write))
                        await fileResponse.Stream.CopyToAsync(fs);

                    var emailer = new Util.Email(db, Enums.NotificationType.Confirmations);
                    string subject = $"{notification.Subject} - {faxItem.TicketNum}";
                    string recipient = string.IsNullOrWhiteSpace(faxItem.Attention)
                        ? faxItem.Counterparty
                        : $"{faxItem.Attention} @ {faxItem.Counterparty}";
                    using var stream = System.IO.File.OpenRead(tempPath);
                    var attachment = new Attachment(stream, $"{faxItem.TicketNum}.pdf");
                    string sentTo = emailer.SendFaxViaEmail(faxItem.FaxNumber, recipient, subject, attachment);
                    concurrentMessages.Add($"Sent {faxItem.TicketNum} to {sentTo}");
                    using var newContext = Main.CreateContext();
                    var rec = newContext.DealConfirmations.First(x => x.TicketNum == faxItem.TicketNum);
                    rec.FaxedBy = userId;
                    rec.FaxDate = DateTime.UtcNow;
                    newContext.SaveChanges();
                }
                catch (Exception ex)
                {
                    concurrentMessages.Add($"Error processing {faxItem.TicketNum}: {GetExceptionMessage(ex)}");
                }
            });
        }
        catch (Exception ex)
        {
            concurrentMessages.Add(GetExceptionMessage(ex));
        }

        var ordered = concurrentMessages.OrderByDescending(x => x.Contains("Error"));
        bool hasErrors = ordered.Any(x => x.Contains("Error"));
        var sb = new StringBuilder();
        sb.AppendLine(hasErrors ? "Some files failed. See details below." : "All files sent successfully.");
        foreach (var msg in ordered)
            sb.AppendLine(msg);
        return Ok(new { hasErrors, message = sb.ToString() });
    }


    [Route("[action]")]
    public async Task<IActionResult> MergeFiles(string type, [FromBody] string[] ticketNums)
    {
        if (!await HasViewPermission())
            return Forbid();

        string newFileName = $"{DateTime.Now:yyyyMMdd_HHmmss}-merged-{type}.pdf";
        string tempFolder = Path.Combine(Path.GetTempPath(), "MergedFiles");
        Directory.CreateDirectory(tempFolder);

        try
        {
            ticketNums = ticketNums.OrderBy(x => x.TicketSort()).ToArray();

            var dealFiles = db.DealConfirmations
                .Where(x => ticketNums.Contains(x.TicketNum))
                .Select(x => new { x.TicketNum, x.ConfirmFile, x.TicketFile })
                .ToList();

            string subFolder = type switch
            {
                "confirms" => "DealConfirms",
                "tickets" => "DealTickets",
                _ => throw new Exception("Unknown merge file type")
            };

            var tempFileTuples = new List<Tuple<string, string>>();

            foreach (var item in dealFiles)
            {
                string? fileName = type == "confirms" ? item.ConfirmFile : item.TicketFile;
                if (string.IsNullOrWhiteSpace(fileName)) continue;

                var fileResponse = await fileService.DownloadFileAsync(subFolder, fileName);
                string tempPath = Path.Combine(tempFolder, $"{Guid.NewGuid():N}.pdf");

                await using (var localFs = new FileStream(tempPath, FileMode.Create, FileAccess.Write))
                    await fileResponse.Stream.CopyToAsync(localFs);

                tempFileTuples.Add(Tuple.Create(item.TicketNum, tempPath));
            }

            if (!tempFileTuples.Any())
                throw new FileNotFoundException("No valid PDF files to merge.");

            // Read all temp files into byte arrays for merging
            var pdfBytesList = new List<(string fileName, byte[] bytes)>();
            foreach (var (ticketNum, tempPath) in tempFileTuples)
            {
                var bytes = await System.IO.File.ReadAllBytesAsync(tempPath);
                pdfBytesList.Add(($"{ticketNum}.pdf", bytes));
            }

            var mergedStream = await Shared.Logic.Package.PdfConverter.MergePdfsFromBytes(pdfBytesList);
            mergedStream.Position = 0;

            foreach (var f in tempFileTuples)
            {
                try { System.IO.File.Delete(f.Item2); } catch { }
            }

            return File(mergedStream, "application/pdf", newFileName);
        }
        catch (Exception ex)
        {
            var bytes = Util.GetExceptionFilesBytes(ex);
            return File(bytes, "text/plain");
        }
    }

    private enum FileType
    {
        Confirm,
        Ticket
    }

    private async Task<IActionResult> DownloadFile(string ticketNum, FileType fileType)
    {
        try
        {
            string subFolder = fileType == FileType.Confirm ? "DealConfirms" : "DealTickets";
            string fileName = db.DealConfirmations.Where(x => x.TicketNum == ticketNum).Select(x => fileType == FileType.Confirm ? x.ConfirmFile : x.TicketFile).FirstOrDefault() ?? "";

            if (string.IsNullOrWhiteSpace(fileName))
                throw new FileNotFoundException();

            var fileResponse = await fileService.DownloadFileAsync(subFolder, fileName);
            return File(fileResponse.Stream, fileResponse.ContentType, fileResponse.FileName);
        }
        catch (Exception ex)
        {
            var bytes = Util.GetExceptionFilesBytes(ex, $"Deal #: {ticketNum}");
            return File(bytes, "text/plain");
        }
    }
    private static int GetDefaultProductIdForTransaction(int transactionTypeId, int productId)
    {
        if (transactionTypeId == (int)Enums.TransactionType.PhysicalGas)
            return (int)Enums.Product.NaturalGas;
        else if (transactionTypeId == (int)Enums.TransactionType.PhysicalNGL)
            return (int)Enums.Product.UnspecifiedNgl;
        else if (transactionTypeId == (int)Enums.TransactionType.Futures)
            return productId;
        else if (transactionTypeId == (int)Enums.TransactionType.PhysicalCrudeOil)
            return (int)Enums.Product.CrudeOil;
        else
            return productId;
    }
}
