using Fast.Models;

namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class CrudeInfoController : Controller
{
    private readonly MyDbContext db;
    private readonly AuthorizationHelper authHelper;
    public CrudeInfoController(MyDbContext context)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
    }

    [Permission("Main Contract", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData()
    {
        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Main Contract", PermissionType.Modify);
        var states = await DataHelper.GetTerritoriesAsync();
        var countries = await (from q in db.Countries orderby q.Id == 1 descending, q.Name select new IdName(q.Id, q.Name)).ToListAsync();
        var crudePoints = (await DataHelper.GetPointsAsync(true, Enums.ProductCategory.CrudeOil)).Select(x => new IdName(x.PointId, x.PointName)).ToList();
        var result = new { hasModifyPermission, states, countries, crudePoints };
        return Ok(result);
    }

    [Permission("Main Contract", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetCrudeInfo(int contractId)
    {
        var dbItem = await (
            from q in db.Contracts
            where q.Id == contractId
            select q.ContractCrudes
        ).AsNoTracking().FirstOrDefaultAsync() ?? new List<ContractCrude>();

        MainContractCrudeDetail crude = new();
        var crudePoints = db.ContractCrudePoints.ToList();
        foreach (var dbCrudeItem in dbItem.OrderByDescending(x => x.EffectiveDate))
        {
            MainContractCrudeCalcItem calcItem = new();
            var matchingPoints = crudePoints.Where(x => x.ContractCrudeId == dbCrudeItem.Id).Select(x => x.PointId).ToList();
            if (matchingPoints.Count > 0)
            {
                calcItem.PointIds = matchingPoints;
            }
            calcItem.PipeLossAllowance = dbCrudeItem.PipeLossAllowance;
            calcItem.Differential = dbCrudeItem.Differential;
            calcItem.IsDiffBeforePipeLoss = dbCrudeItem.IsDiffBeforePipeLoss;
            calcItem.EffectiveDate = dbCrudeItem.EffectiveDate;
            crude.CalcItems.Add(calcItem);
        }

        return Ok(crude);
    }
}

