using Fast.Logic;
using Fast.Models;
using Microsoft.AspNetCore.OData.Routing.Controllers;

namespace Fast.Web.Controllers


{
    [Authorize]
    [ApiController]
    [Route("api/[controller]")]
    public class ContactController : ODataController
    {
        private readonly MyDbContext db;
        private readonly AuthorizationHelper authHelper;

        public ContactController(MyDbContext context)
        {
            db = context;
            authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
        }

        [Permission("Contact", PermissionType.View)]
        [Route("/odata/GetContactItems")]
        public IActionResult GetItems(ODataQueryOptions<ContactListItem> queryOptions, bool isExport)
        {
            queryOptions = Util.GetQueryOptionsWithConvertedDates(queryOptions);
            var itemsQueryable = GetItemsInternal();
            var items = (queryOptions.ApplyTo(itemsQueryable) as IEnumerable<ContactListItem>)?.ToList();

            if (isExport)
                return File(Util.Excel.GetExportFileStream(items), "application/octet-stream");
            else
                return Ok(items);
        }

        private IQueryable<ContactListItem> GetItemsInternal()
        {
            IQueryable<ContactListItem>? itemsQueryable = null;

            itemsQueryable = (
                from q in db.Contacts
                join oi in db.VwContactOverviewInfos on q.Id equals oi.ContactId into j1
                from oi in j1.DefaultIfEmpty()
                select new ContactListItem
                {
                    Id = q.Id,
                    DisplayName = q.DisplayName,
                    FullName = q.FullName ?? "",
                    Title = q.Title ?? "",
                    City = q.City ?? "",
                    State = q.State == null ? "" : q.State.Name ?? "",
                    Zip = q.Zip ?? "",
                    Country = q.Country == null ? "" : q.Country.Name,
                    ContactCounterparties = oi.ContactCounterparties ?? "",
                    ContactTypes = oi.ContactTypes ?? ""
                }
            );

            return itemsQueryable;
        }

        [Permission("Contact", PermissionType.View)]
        [Route("[action]")]
        public async Task<IActionResult> GetRequiredData()
        {
            var userId = Util.GetAppUserId(User);

            var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Contact", PermissionType.Modify);
            var contactTypes = (from q in db.ContactTypes orderby q.Id select new IdName(q.Id, q.Name)).ToList();
            var counterparties = DealHelper.GetNewCounterparties();
            var states = (from q in db.Territories orderby q.Name select new IdName(q.Id, q.Name ?? "")).ToList();
            var countries = (from q in db.Countries orderby q.Id == 1 descending, q.Name select new { q.Id, q.Name }).ToList();

            var result = new { hasModifyPermission, contactTypes, counterparties, states, countries };
            return Ok(result);
        }

        [Permission("Contact", PermissionType.View)]
        [Route("[action]")]
        public async Task<IActionResult> GetDetail(int id)
        {
            var detail = await (
                from q in db.Contacts
                where q.Id == id
                select new ContactDetail
                {
                    Id = q.Id,
                    DisplayName = q.DisplayName,
                    FullName = q.FullName ?? "",
                    Title = q.Title ?? "",
                    AddressLine1 = q.AddressLine1 ?? "",
                    AddressLine2 = q.AddressLine2 ?? "",
                    City = q.City ?? "",
                    StateId = q.StateId,
                    Zip = q.Zip ?? "",
                    CountryId = q.CountryId,
                    OfficePhone = q.OfficePhone ?? "",
                    CellPhone = q.CellPhone ?? "",
                    AltPhone = q.AltPhone ?? "",
                    Fax = q.Fax ?? "",
                    Email = q.Email ?? "",
                    Notes = q.Notes ?? ""
                }
            ).AsNoTracking().FirstAsync();

            var contactCounterparties = (
               from q in db.ContactCounterparties
               where q.ContactId == id
               orderby q.Id
               select new IdDateOnly
               {
                   Id = q.CounterpartyId,
                   Date = q.InactiveDate
               }
            ).ToList();
            detail.ContactCounterparties = contactCounterparties;

            var contactTypeIds = (
                from q in db.ContactToContactTypes
                where q.ContactId == id
                orderby q.Id
                select q.ContactTypeId
            ).ToList();
            detail.ContactTypeIds = contactTypeIds;

            return Ok(detail);
        }

        [Permission("Contact", PermissionType.Modify)]
        [Route("[action]")]
        public async Task<IActionResult> SaveDetail(ContactDetail detail, Enums.SaveType saveType)
        {
            int resultId = 0;

            await db.Database.CreateExecutionStrategy().Execute(async () =>
            {
                using var dbContextTransaction = await db.Database.BeginTransactionAsync();
                Contact? dbItem = null;
                if (saveType != Enums.SaveType.New)
                {
                    dbItem = await (
                        from q in db.Contacts
                            .Include(x => x.ContactToContactTypes)
                            .Include(x => x.ContactCounterparties)
                        where q.Id == detail.Id
                        select q
                    ).FirstOrDefaultAsync();
                }

                if (dbItem == null) //if the item does not exist then add it
                {
                    dbItem = new Contact();
                    db.Contacts.Add(dbItem);
                }
                else
                {
                    //remove existing items so that they get completely re-inserted
                    db.ContactToContactTypes.RemoveRange(dbItem.ContactToContactTypes);
                    db.ContactCounterparties.RemoveRange(dbItem.ContactCounterparties);
                }

                var d = detail;
                dbItem.DisplayName = d.DisplayName;
                dbItem.FullName = d.FullName;
                dbItem.Title = d.Title;
                dbItem.AddressLine1 = d.AddressLine1;
                dbItem.AddressLine2 = d.AddressLine2;
                dbItem.City = d.City;
                dbItem.StateId = d.StateId;
                dbItem.Zip = d.Zip;
                dbItem.CountryId = d.CountryId;
                dbItem.OfficePhone = d.OfficePhone;
                dbItem.CellPhone = d.CellPhone;
                dbItem.AltPhone = d.AltPhone;
                dbItem.Fax = d.Fax;
                dbItem.Email = d.Email;
                dbItem.Notes = d.Notes;
                await db.SaveChangesAsync();

                if (d.ContactCounterparties != null)
                {
                    foreach (var contactCounterparty in d.ContactCounterparties)
                    {
                        dbItem.ContactCounterparties.Add(new ContactCounterparty { CounterpartyId = contactCounterparty.Id, InactiveDate = contactCounterparty.Date });
                        await db.SaveChangesAsync(); //call save changes after each addition so that the order inserted is preserved
                    }
                }

                if (d.ContactTypeIds != null)
                {
                    foreach (var contactTypeId in d.ContactTypeIds)
                    {
                        dbItem.ContactToContactTypes.Add(new ContactToContactType { ContactTypeId = contactTypeId });
                        await db.SaveChangesAsync(); //call save changes after each addition so that the order inserted is preserved
                    }
                }

                resultId = dbItem.Id;
                await dbContextTransaction.CommitAsync();
            });

            return Ok(resultId);
        }

        [Permission("Contact", PermissionType.Modify)]
        [Route("[action]/{id}")]
        public IActionResult DeleteDetail(int id)
        {
            Contact dbItem = db.Contacts.Where(x => x.Id == id).First();
            db.Entry(dbItem).State = EntityState.Deleted;

            db.SaveChanges();

            return Ok();
        }
    }
}
