namespace Fast.Web.Controllers;
using Fast.Models;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class AccountingInfoController : Controller
{
    private readonly MyDbContext db;

    private readonly AuthorizationHelper authHelper;
    public AccountingInfoController(MyDbContext context)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
    }

    [Permission("Main Contract", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData()
    {
        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Main Contract", PermissionType.Modify);
        var states = await DataHelper.GetTerritoriesAsync();
        var countries = await (from q in db.Countries orderby q.Id == 1 descending, q.Name select new IdName(q.Id, q.Name)).ToListAsync();
        var result = new { hasModifyPermission, states, countries };
        return Ok(result);
    }

    [Permission("Main Contract", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetAccountingInfo(int contractId, bool getInternalSide, int entityId, int productId)
    {
        ContractAccounting dbItem;
        if (getInternalSide)
        {
            dbItem = await (
                from q in db.Contracts
                where q.IsInternalSideOfContract
                    && q.InternalEntityId == entityId
                    && q.ProductId == productId
                select q.ContractAccounting
            ).AsNoTracking().FirstOrDefaultAsync() ?? new ContractAccounting();
        }
        else
        {
            dbItem = await (
               from q in db.Contracts
               where q.Id == contractId
               select q.ContractAccounting
           ).AsNoTracking().FirstOrDefaultAsync() ?? new ContractAccounting();
        }

        var naesbData = await (
            from q in db.ContractNaesbs
            where q.ContractId == contractId
            select new
            {
                q.WireTransferPayment,
                q.AchPayment,
                q.CheckPayment,
                q.AddendumExhibitC
            }
        ).AsNoTracking().FirstOrDefaultAsync();

        MainContractAccountingDetail d = new();
        d.InvoiceAddressLine1 = dbItem.InvoiceAddressLine1;
        d.InvoiceAddressLine2 = dbItem.InvoiceAddressLine2;
        d.InvoiceCountryId = dbItem.InvoiceCountryId;
        d.InvoiceCity = dbItem.InvoiceCity;
        d.InvoiceStateId = dbItem.InvoiceStateId;
        d.InvoiceZip = dbItem.InvoiceZip;
        d.InvoiceAttn = dbItem.InvoiceAttn;
        d.InvoiceTelephoneNum = dbItem.InvoiceTelephoneNum;
        d.InvoiceFaxNum = dbItem.InvoiceFaxNum;
        d.InvoiceEmailAddress = dbItem.InvoiceEmailAddress;
        d.DoEmail = dbItem.DoEmail;
        d.DoFax = dbItem.DoFax;
        d.WireBank = dbItem.WireBankName;
        d.WireAbaNum = dbItem.WireAbaNum;
        d.WireAccountNum = dbItem.WireAccountNum;
        d.WireOtherDetails = dbItem.WireOtherDetails;
        d.AchBank = dbItem.AchBank;
        d.AchAbaNum = dbItem.AchAbaNum;
        d.AchAccountNum = dbItem.AchAccountNum;
        d.AchOtherDetails = dbItem.AchOtherDetails;
        d.ChecksAttn = dbItem.ChecksAttn;
        d.ChecksAddressLine1 = dbItem.ChecksAddressLine1;
        d.ChecksAddressLine2 = dbItem.ChecksAddressLine2;
        d.ChecksCountryId = dbItem.ChecksCountryId;
        d.ChecksCity = dbItem.ChecksCity;
        d.ChecksStateId = dbItem.ChecksStateId;
        d.ChecksZip = dbItem.ChecksZip;
        if (naesbData != null)
        {
            d.IsExhibitCChecked = naesbData.AddendumExhibitC;
            d.IsWireChecked = naesbData.WireTransferPayment;
            d.IsAchChecked = naesbData.AchPayment;
            d.IsChecksChecked = naesbData.CheckPayment;
        }

        return Ok(d);
    }
}
