import { inject, Injectable } from '@angular/core';
import { HttpClient } from '@angular/common/http';
import { OdataGridHelper } from '../_shared/utils/odata-grid-helper';
import { map, Observable } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { GridDataResult } from '@progress/kendo-angular-grid';
import * as util from '../_shared/utils/util';

export interface Item {
  PaymentInstructionsId: number;
  Counterparty: string;
  AccountHolder: string;
  EffectiveDate: string;
  BankName: string;
  Relationships: string;
}

export interface Detail {
  paymentInstructionsId: number;
  counterpartyId: number;
  effectiveDate: Date;
  customerNum: string;
  vendorNum: string;
  bankName: string;
  bankCity: string;
  stateId: number;
  accountHolder: number;
  bankWireNum: string;
  bankAchNum: string;
  accountNum: string;
  swiftCode: string;
  abaWireCity: string;
  abaWireStateId: number;
  abaAchCity: string;
  abaAchStateId: number;
  fcTo: string;
  fcAccountNum: string;
  documents: util.DocItem[];
  notes: string;
  ccd: boolean;
  ctx: boolean;
  ppd: boolean;
  tax: boolean;
  wire: boolean;
}

export interface RequiredData {
  hasModifyPermission: boolean;
  counterparties: CounterpartyItem[];
  states: util.IdName[];
}

export interface CounterpartyItem {
  id: number;
  name: string;
  customerNum: string;
  vendorNum: string;
}

export interface RoutingData {
  routingNum: string;
  achName: string;
  achCity: string;
  achStateId: number;
  wireName: string;
  wireCity: string;
  wireStateId: number;
}

export enum SaveType {
  New = 1,
  Normal = 2
}

@Injectable({
  providedIn: 'root'
})
export class PaymentInstructionsService extends OdataGridHelper {
  private baseUrl = `${window.location.origin}/api/PaymentInstructions`;
  http = inject(HttpClient);

  requiredData$ = this.http.get<RequiredData>(`${this.baseUrl}/GetRequiredData`)

  getItems(state: State, showInactive: boolean): Observable<GridDataResult> {
    const url = `GetPaymentInstructionsItems?showInactive=${showInactive}`;
    return this.fetch(url, state, null, false);
  }

  getDetail(id: number): Observable<Detail> {
    const url = `${this.baseUrl}/GetDetail?id=${id}`;
    return this.http.get<Detail>(url);
  }

  getRoutingData(routingNum: string): Observable<RoutingData> {
    const url = `${this.baseUrl}/GetRoutingData?routingNum=${routingNum}`;
    return this.http.get<RoutingData>(url);
  }

  saveDetail(detail: Detail, saveType: SaveType): Observable<number> {
    const url = `${this.baseUrl}/SaveDetail?saveType=${saveType}`;
    detail = util.convertDatesToDateOnlyStrings(detail, ['effectiveDate']);
    return this.http.put<number>(url, detail);
  }

  deleteDetail(id: number) {
    const url = `${this.baseUrl}/DeleteDetail/${id}`;
    return this.http.delete(url);
  }

  sendNotification(id: number) {
    const url = `${this.baseUrl}/SendNotification/${id}`;
    return this.http.post(url, null);
  }

  exportItems(state: State, fileNameOriginal: string) {
    return this.fetchExport(`GetPaymentInstructionsItems`, state, null, fileNameOriginal);
  }

  downloadDoc(fileNameOriginal: string, fileNameOnDisk: string): Observable<{ fileBlob: Blob, fileName: string }> {
    const url = `${this.baseUrl}/DownloadDoc?fileNameOnDisk=${fileNameOnDisk}`;
    return this.http.get(url, { observe: 'response', responseType: 'blob' }).pipe(
      map(result => {
        const fileBlob = result.body;
        const fileName: string = result.body.type === "text/plain" ? "Error.txt" : fileNameOriginal;
        return { fileBlob, fileName };
      })
    );
  }
}
