import { ChangeDetectionStrategy, ChangeDetectorRef, Component, HostListener, OnInit, ViewChild, ViewEncapsulation, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { catchError, filter, map, retry, shareReplay, switchMap, tap } from 'rxjs/operators';
import { BehaviorSubject, combineLatest, Observable, of, Subject, } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { Detail, Item, PaymentInstructionsService, RequiredData, RoutingData, SaveType, } from './payment-instructions.service';
import { MessageService, PromptSettings, promptAction } from '../_shared/services/message.service';
import { AbstractControl, Validators } from '@angular/forms';
import { ToastService } from '../_shared/services/fast-toast.service';
import { TooltipDirective } from '@progress/kendo-angular-tooltip';
import * as util from '../_shared/utils/util';
import { ErrorEvent, SuccessEvent, UploadComponent, UploadProgressEvent } from '@progress/kendo-angular-upload';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';
import { ActivatedRoute, Router } from '@angular/router';
import { GridDataResult } from '@progress/kendo-angular-grid';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { KENDO_MASKEDTEXTBOX } from '@progress/kendo-angular-inputs';
import { SelectableSettings } from '@progress/kendo-angular-grid';
import { FastGridComponent } from '../_shared/elements/fast-grid.component';

function ValidateBankNum(cityOrStateFieldName: string, bankNumFieldName: string) {
  return (c: AbstractControl): { [key: string]: boolean } | null => {
    if (c.parent) {
      const bankNum = c.parent.get(bankNumFieldName).value;
      const cityOrState = c.parent.get(cityOrStateFieldName).value;
      const isCityOrStateFilledIn = !util.isNullOrWhitespace(cityOrState);
      const isNumFilledIn = !util.isNullOrWhitespace(bankNum);
      if (!isCityOrStateFilledIn && isNumFilledIn) {
        return { 'FieldRequired': true };
      }
      return null;
    } else return null;
  }
}

@Component({
  selector: 'payment-instructions',
  templateUrl: './payment-instructions.component.html',
  encapsulation: ViewEncapsulation.None,
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON, KENDO_MASKEDTEXTBOX]
})

export class PaymentInstructionsComponent implements OnInit {
  @ViewChild("grid") fastGrid: FastGridComponent;
  @ViewChild('tooltipGrid') tooltipGrid: TooltipDirective;
  @ViewChild('docUploadTarget') docUploadElem: UploadComponent;
  util = util;
  detailForm: util.FormModel<Detail>;
  detailInitialValues: Detail;
  showInactive: boolean = false;
  gridScrollPosition: util.GridScrollPosition = { topPos: 0, leftPos: 0 };
  localRequiredData: RequiredData;
  hasModifyPermission = false;
  mySelection: number[] = [];
   selectableSettings: SelectableSettings = {
      checkboxOnly: false,
      mode: 'single',
      enabled: true
    }
  manualChangesTriggering: boolean = false;
  docSaveUrl = `${window.location.origin}/api/PaymentInstructions/UploadDoc`;
  gridLoading$ = new BehaviorSubject<boolean>(true)
  refreshItems$ = new BehaviorSubject<string>(null)
  refreshRequiredData$ = new BehaviorSubject(null)
  refreshRoutingData$ = new BehaviorSubject<[string, boolean]>([null, false])
  detailLoading$ = new BehaviorSubject<boolean>(true)
  refreshDetail$ = new BehaviorSubject<number>(null)
  detailOpened$ = new BehaviorSubject<boolean>(false)
  exporting$ = new BehaviorSubject<boolean>(false)
  exportClicked$ = new Subject()
  downloadDoc$ = new Subject<util.DocItem>()
  save$ = new Subject<SaveType>()
  delete$ = new Subject()
  sendNotification$ = new Subject<number>()
  state: State = {
    filter: null,
    group: null,
    skip: 0,
    sort: [{ field: 'Counterparty', dir: 'asc' }],
    take: 50
  };

  ngOnInit(): void {
    this.activatedRoute.queryParamMap.subscribe(params => {
      const detailParam: string = params.get('detail');
      this.loadDetailParam(detailParam);
    });
  }

  loadDetailParam(detailParam: string): void {
    if (util.isNumber(detailParam))
      this.openDetail(parseInt(detailParam))
  }

  hasAchNumChanged$: Observable<string>;
  hasWireNumChanged$: Observable<string>;
  routingData$: Observable<[RoutingData, boolean]>;
  requiredData$: Observable<RequiredData>;
  items$: Observable<GridDataResult>;
  exportAction$: Observable<{ fileBlob: Blob; fileName: string; }>;
  detail$: Observable<Detail>;
  saveResult$: Observable<number>;
  deleteResult$: Observable<object>;
  sendNotificationResult$: Observable<object>;
  downloadDocResult$: Observable<{ fileBlob: Blob; fileName: string; }>;
  title$: Observable<string>;
  counterpartyChanged$: Observable<number>;

  private service = inject(PaymentInstructionsService);
  private messageService = inject(MessageService);
  private titleService = inject(Title);
  private fb = inject(CustomFormBuilder);
  private ref = inject(ChangeDetectorRef);
  private notify = inject(ToastService);
  private activatedRoute = inject(ActivatedRoute);
  private router = inject(Router);

  constructor() {
    this.detailForm = this.getDetailForm();
    this.detailInitialValues = this.detailForm.value as Detail;

    this.requiredData$ = this.refreshRequiredData$.pipe(
      tap(() => this.detailLoading$.next(true)),
      switchMap(refreshType => {
        return combineLatest([this.service.requiredData$, of(refreshType)]);
      }),
      map(([requiredData, refreshType]) => {
        this.localRequiredData = requiredData;
        if (refreshType === util.RefreshType.SelfOnly)
          this.detailLoading$.next(false);
        return requiredData;
      }),
      tap((requiredData) => {
        this.hasModifyPermission = requiredData.hasModifyPermission;
        // util.focusInputTarget()
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.items$ = this.refreshItems$.pipe(
      tap(() => {
        this.gridLoading$.next(true);
      }),
      switchMap(() => {
        return this.service.getItems(this.state, this.showInactive);
      }),
      tap(() => {
        this.gridLoading$.next(false);
        util.goToSavedGridScrollPos(this.fastGrid.kendoGridElement , this.gridScrollPosition);
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService);
      }), retry(10)
    )

    this.exportAction$ = this.exportClicked$.pipe(
      tap(() => {
        this.exporting$.next(true);
      }),
      switchMap(() => {
        return this.service.exportItems(this.state, 'PaymentInstructions.xlsx');
      }),
      tap(res => {
        util.openOrSaveFile(res.fileBlob, res.fileName);
        this.exporting$.next(false);
      }),
      shareReplay(1),
      catchError(err => {
        this.exporting$.next(false);
        return util.handleError(err, this.messageService);
      }), retry(10)
    )

    this.detail$ = this.refreshDetail$.pipe(
      filter(id => id !== null),
      switchMap(id => {
        this.detailLoading$.next(true);
        this.detailForm.disable();
        this.detailForm.reset();
        if (id === 0)
          return of(this.detailInitialValues);
        else
          return this.service.getDetail(id);
      }),
      map(result => {
        const detail: Detail = result;
        if (detail) {
          util.convertToDates(detail);
          this.detailForm.setValue(detail);
        }
        return detail;
      }),
      tap(() => {
        this.detailFinishedLoading();
      }),
      shareReplay(1),
      catchError(err => {
        this.closeDetail(false);
        return util.handleError(err, this.messageService)
      }), retry(5)
    )

    this.saveResult$ = this.save$.pipe(
      switchMap(saveType => {
        this.detailLoading$.next(true);
        this.detailForm.disable();
        const itemToSave: Detail = this.detailForm.value as Detail;
        return this.service.saveDetail(itemToSave, saveType);
      }),
      tap(saveResult => {
        this.notify.success('save successful');
        this.closeDetail(false);
        this.refreshItems$.next(null);
        this.mySelection = [saveResult];
        this.notificationPrompt(saveResult);
      }),
      shareReplay(1),
      catchError(err => {
        this.detailFinishedLoading();
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.deleteResult$ = this.delete$.pipe(
      switchMap(() => {
        this.detailLoading$.next(true);
        this.detailForm.disable();
        const itemToDelete: Detail = this.detailForm.getRawValue();
        return this.service.deleteDetail(itemToDelete.paymentInstructionsId);
      }),
      tap(() => {
        this.notify.success('delete successful');
        this.detailFinishedLoading();
        this.closeDetail(false);
        this.refreshItems$.next(null);
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.sendNotificationResult$ = this.sendNotification$.pipe(
      switchMap((id: number) => {
        return this.service.sendNotification(id);
      }),
      tap(() => {
        this.notify.success('notification sent');
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.downloadDocResult$ = this.downloadDoc$.pipe(
      filter(docItem => {
        return docItem && docItem.fileNameOnDisk !== null;
      }),
      switchMap(fileNamePair => {
        return this.service.downloadDoc(fileNamePair.fileNameOriginal, fileNamePair.fileNameOnDisk);
      }),
      tap(res => {
        util.openOrSaveFile(res.fileBlob, res.fileName);
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService);
      }), retry(10)
    )

    this.title$ = of('Payment Instructions').pipe(
      tap((title) => util.trySetTitle(this.titleService, title))
    )

    this.counterpartyChanged$ = this.detailForm.get('counterpartyId').valueChanges.pipe(
      filter(() => {
        return !this.detailLoading$.value || this.manualChangesTriggering;
      }),
      tap((counterpartyId: number) => {
        const foundCounterparty = this.localRequiredData.counterparties.find(x => {
          return x.id == counterpartyId
        });
        const customerNum = !util.isNullOrWhitespace(foundCounterparty?.customerNum) ? foundCounterparty.customerNum : 'None';
        const vendorNum = !util.isNullOrWhitespace(foundCounterparty?.vendorNum) ? foundCounterparty.vendorNum : 'None';
        this.detailForm.patchValue({
          customerNum: customerNum,
          vendorNum: vendorNum
        });
        this.manualChangesTriggering = false;
      }),
      shareReplay(1),
    )

    this.hasAchNumChanged$ = this.detailForm.get('bankAchNum').valueChanges.pipe(
      filter(() => !this.detailLoading$.value),
      tap((bankAchNum: string) => {
        const isAch = true;
        const achNumCtrl = this.detailForm.get('bankAchNum');
        const isValid = achNumCtrl.valid && !util.isNullOrWhitespace(achNumCtrl.value);
        if (isValid)
          this.refreshRoutingData$.next([bankAchNum, isAch]);
        else {
          this.detailForm.controls['abaAchCity'].updateValueAndValidity();
          this.detailForm.controls['abaAchStateId'].updateValueAndValidity();
        }
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(1)
    )

    this.hasWireNumChanged$ = this.detailForm.get('bankWireNum').valueChanges.pipe(
      filter(() => !this.detailLoading$.value),
      tap((bankWireNum: string) => {
        const isAch = false;
        const wireNumCtrl = this.detailForm.get('bankWireNum');
        const isValid = wireNumCtrl.valid && !util.isNullOrWhitespace(wireNumCtrl.value);
        if (isValid)
          this.refreshRoutingData$.next([bankWireNum, isAch]);
        else {
          this.detailForm.controls['abaWireCity'].updateValueAndValidity();
          this.detailForm.controls['abaWireStateId'].updateValueAndValidity();
        }
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(1)
    )

    this.routingData$ = this.refreshRoutingData$.pipe(
      filter(([routingNum,]) => {
        return !util.isNullOrWhitespace(routingNum);
      }),
      tap(() => {
        this.detailLoading$.next(true);
      }),
      switchMap(([routingNum, isAch]) => {
        return combineLatest([this.service.getRoutingData(routingNum), of(isAch)]);
      }),
      tap(([routingData, isAch]) => {
        this.detailLoading$.next(false);
        if (isAch) {
          if (!util.isNullOrWhitespace(routingData?.achCity))
            this.detailForm.patchValue({ abaAchCity: routingData.achCity });
          if (routingData?.achStateId)
            this.detailForm.patchValue({ abaAchStateId: routingData.achStateId });
          if (!util.isNullOrWhitespace(routingData?.achName))
            this.detailForm.patchValue({ bankName: routingData.achName });

          this.detailForm.controls['abaAchCity'].updateValueAndValidity();
          this.detailForm.controls['abaAchStateId'].updateValueAndValidity();
        }
        else {
          if (!util.isNullOrWhitespace(routingData?.wireCity))
            this.detailForm.patchValue({ abaWireCity: routingData.wireCity });
          if (routingData?.wireStateId)
            this.detailForm.patchValue({ abaWireStateId: routingData.wireStateId });
          if (!util.isNullOrWhitespace(routingData?.wireName))
            this.detailForm.patchValue({ bankName: routingData.wireName });

          this.detailForm.controls['abaWireCity'].updateValueAndValidity();
          this.detailForm.controls['abaWireStateId'].updateValueAndValidity();
        }
      }),
      shareReplay(1),
      catchError(err => {
        this.detailLoading$.next(false);
        this.detailForm.controls['abaAchCity'].updateValueAndValidity();
        this.detailForm.controls['abaAchStateId'].updateValueAndValidity();
        this.detailForm.controls['abaWireCity'].updateValueAndValidity();
        this.detailForm.controls['abaWireStateId'].updateValueAndValidity();
        return util.handleError(err, this.messageService)
      }), retry(1)
    )
  }

  @HostListener('window:resize') onResize() {
    //this function is empty but for some reason it helps the window to resize faster
  };

  getDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<Detail> = fb.group({
      paymentInstructionsId: fb.ctr(0, Validators.required),
      counterpartyId: fb.ctr(null, Validators.required),
      effectiveDate: fb.ctr(util.currentDate.toDate(), Validators.required),
      customerNum: fb.ctr(null),
      vendorNum: fb.ctr(null),
      bankName: fb.ctr(null),
      bankCity: fb.ctr(null),
      stateId: fb.ctr(null),
      accountHolder: fb.ctr(null),
      bankWireNum: fb.ctr(null, [
        Validators.pattern(/^\d{9}$/)
      ]),
      bankAchNum: fb.ctr(null, [
        Validators.pattern(/^\d{9}$/)
      ]),
      accountNum: fb.ctr(null),
      swiftCode: fb.ctr(null),
      fcTo: fb.ctr(null),
      fcAccountNum: fb.ctr(null),
      documents: fb.ctr([]),
      notes: fb.ctr(null),
      abaAchCity: fb.ctr(null, [ValidateBankNum('abaAchCity', 'bankAchNum')]),
      abaAchStateId: fb.ctr(null, fb.ctr(ValidateBankNum('abaAchStateId', 'bankAchNum'))),
      abaWireCity: fb.ctr(null, fb.ctr(ValidateBankNum('abaWireCity', 'bankWireNum'))),
      abaWireStateId: fb.ctr(null, fb.ctr(ValidateBankNum('abaWireStateId', 'bankWireNum'))),
      ccd: fb.ctr(false),
      ctx: fb.ctr(false),
      ppd: fb.ctr(false),
      tax: fb.ctr(false),
      wire: fb.ctr(false)
    });

    return fg;
  }

  openDetail(id: number): void {
    util.hideTooltips([this.tooltipGrid]);
    util.saveGridScrollPos(this.fastGrid.kendoGridElement, this.gridScrollPosition);
    this.refreshDetail$.next(id);
    this.detailOpened$.next(true);
  }

  onDetailClosing() {
    util.onDetailChanging(this.detailForm, this.messageService, this.closeDetail, this.save);
  }

  closeDetail = (isFromInterface: boolean) => {
    this.detailOpened$.next(false);
    this.router.navigate([], { relativeTo: this.activatedRoute, queryParams: {} });
    if (isFromInterface)
      util.goToSavedGridScrollPos(this.fastGrid.kendoGridElement, this.gridScrollPosition);
  }

  detailFinishedLoading(): void {
    this.manualChangesTriggering = true;
    this.detailForm.enable();
    this.detailLoading$.next(false);
    // util.focusInputTarget();
  }

  add(): void {
    this.openDetail(0);
  }

  edit(dataItem: Item): void {
    this.openDetail(dataItem.PaymentInstructionsId);
  }

  save = (saveType: util.SaveType) => {
    this.detailForm.markAllAsTouched();

    const wire = this.detailForm.get('bankWireNum');
    const ach = this.detailForm.get('bankAchNum');
    const errorMessages = [];

    if (wire.hasError('pattern')) {
      errorMessages.push("ABA Wire # must contain 9 digits");
    }

    if (ach.hasError('pattern')) {
      errorMessages.push("ABA ACH # must contain 9 digits");
    }

    if (this.detailForm.valid) {
      this.save$.next(saveType);
    } else {
      this.notify.error(errorMessages.length ? errorMessages.join(' & ') : 'validation failed');
    }
  }

  delete(): void {
    const deleteConfirmSettings: PromptSettings = {
      title: "Please confirm",
      content: "Are you sure you want to delete this item?",
      type: "Yes-No"
    }

    this.messageService.prompt(deleteConfirmSettings).then((result) => {
      if (result === promptAction.Yes) {
        this.delete$.next(null);
      }
    })
  }

  notificationPrompt(id: number): void {
    const notificationPromptSettings: PromptSettings = {
      title: "Send Notification",
      content: "Would you like to send a new data notification?",
      type: "Yes-No"
    };

    this.messageService.prompt(notificationPromptSettings).then((result) => {
      if (result === promptAction.Yes) {
        this.sendNotification$.next(id);
      }
    })
  }

  docComplete() {
    this.docUploadElem.clearFiles();
  }

  downloadDoc(doc: util.DocItem) {
    this.downloadDoc$.next(doc)
  }

  removeDoc(doc: util.DocItem) {
    const docs: util.DocItem[] = this.detailForm.get('documents').value;
    const indexToRemove = docs.findIndex(x => x.fileNameOnDisk === doc.fileNameOnDisk);
    docs.splice(indexToRemove, 1);
    this.detailForm.patchValue({ documents: docs });
    this.detailForm.markAsDirty();
  }

  docUploadSuccess(value: SuccessEvent) {
    const newdocitems: util.DocItem[] = value.response.body;
    let docs: util.DocItem[] = this.detailForm.get('documents').value;
    if (docs)
      docs = docs.concat(newdocitems);
    else
      docs = newdocitems;
    this.detailForm.patchValue({ documents: docs });
    this.detailForm.markAsDirty();
  }

  docUploadError(value: ErrorEvent) {
    util.handleError(value.response, this.messageService);
  }

  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  uploadProgress(event: UploadProgressEvent) {
    this.ref.detectChanges();
  }

  export(): void {
    this.exportClicked$.next(null);
  }

  dataStateChange(state: State): void {
    this.gridScrollPosition.topPos = 0;
    this.gridScrollPosition.leftPos = 0;
    util.fixKendoQueryFilter(state.filter);
    this.state = state;
    this.refreshItems$.next(null);
  }

  refreshDropdowns() {
    this.refreshRequiredData$.next(util.RefreshType.SelfOnly);
  }

  showInactiveChanged(showInactive: boolean) {
    this.showInactive = showInactive;
    this.refreshItems$.next(null);
  }
}
