import { Component, ChangeDetectionStrategy, output, input, ViewEncapsulation, AfterViewInit, inject } from '@angular/core';
import { tap, map, catchError, switchMap, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, combineLatest, zip, Subject, Observable } from 'rxjs';
import { MessageService } from '../../_shared/services/message.service';
import * as util from '../../_shared/utils/util'
import { OtherInfoService, RequiredData, MainContractOtherInfoDetail } from './other-info.service';
import { CustomFormBuilder } from '../../_shared/services/custom-form-builder.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../../app.config';

@Component({
  selector: 'app-other-info',
  templateUrl: './other-info.component.html',
  styleUrl: './other-info.component.scss',
  changeDetection: ChangeDetectionStrategy.OnPush,
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON]
})
export class OtherInfoComponent implements AfterViewInit {
  internalEntityId = input.required<number>();
  productId = input.required<number>();
  saveClicked = output();
  closeClicked = output();

  util = util;
  icons = util.icons;
  localRequiredData: RequiredData;
  hasModifyPermission = false;
  otherInfoDetailForm = this.getOtherInfoDetailForm();
  otherInfoInitialValues = this.otherInfoDetailForm.value as MainContractOtherInfoDetail;
  refreshRequiredData$ = new BehaviorSubject(null);
  loading$ = new BehaviorSubject<boolean>(true);
  refreshOtherInfo$ = new Subject<void>();
  otherInfo$: Observable<MainContractOtherInfoDetail>;
  loadingComplete$: Observable<[MainContractOtherInfoDetail, RequiredData]>;

  private service = inject(OtherInfoService);
  private messageService = inject(MessageService);
  private fb = inject(CustomFormBuilder);

  constructor() {
    this.otherInfo$ = this.refreshOtherInfo$.pipe(
      tap(() => {
        this.loading$.next(true);
      }),
      switchMap(() => {
        return this.service.getOtherInfo(this.internalEntityId(), this.productId());
      }),
      map(detail => {
        this.otherInfoDetailForm.setValue(detail);
        return detail;
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.loadingComplete$ = zip(this.otherInfo$, this.requiredData$).pipe(
      tap(() => {
        this.loading$.next(false);
      })
    )
  }

  ngAfterViewInit(): void {
    this.refreshOtherInfo$.next();
  }

  getOtherInfoDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<MainContractOtherInfoDetail> = fb.group({
      agentClause: fb.ctr(null),
      lcLanguage: fb.ctr(null)
    });
    return fg;
  }

  saveOtherInfo() {
    this.saveClicked.emit();
  }

  closeOtherInfo() {
    this.closeClicked.emit();
  }

  getOtherInfo(): MainContractOtherInfoDetail {
    const otherInfoItems = this.otherInfoDetailForm.value as MainContractOtherInfoDetail;
    return otherInfoItems;
  }

  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => this.loading$.next(true)),
    switchMap(refreshType => {
      return combineLatest([this.service.requiredData$, of(refreshType)]);
    }),
    map(([requiredData, refreshType]) => {
      this.localRequiredData = requiredData;
      if (refreshType === util.RefreshType.SelfOnly)
        this.loading$.next(false);
      return requiredData;
    }),
    tap((requiredData) => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )
}
