import { Component, ChangeDetectionStrategy, ChangeDetectorRef, output, input, OnInit, ViewEncapsulation, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { tap, map, catchError, switchMap, filter, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, combineLatest, zip } from 'rxjs';
import { MessageService } from '../../_shared/services/message.service';
import { DialogService } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../../_shared/services/notify.service';
import * as util from '../../_shared/utils/util'
import { Naesb2006Service, MainContractNaesbInfoDetail, RequiredData } from './naesb2006.service';
import { ActivatedRoute } from '@angular/router';
import { CustomFormBuilder } from '../../_shared/services/custom-form-builder.service';
import { PayOptions } from '../accounting-info/accounting-info.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../../app.config';

@Component({
  selector: 'app-naesb2006',
  templateUrl: './naesb2006.component.html',
  styleUrl: './naesb2006.component.scss',
  changeDetection: ChangeDetectionStrategy.OnPush,
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON]
})

export class Naesb2006Component implements OnInit {
  contractId = input.required<number>();
  saveClicked = output();
  closeClicked = output();

  private service = inject(Naesb2006Service);
  private messageService = inject(MessageService);
  private titleService = inject(Title);
  private fb = inject(CustomFormBuilder);
  private ref = inject(ChangeDetectorRef);
  private dialogService = inject(DialogService);
  private notify = inject(NotifyService);
  private activatedRoute = inject(ActivatedRoute);

  constructor() {

  }
  util = util;
  icons = util.icons;
  localRequiredData: RequiredData;
  hasModifyPermission = false;
  naesbInfoDetailForm = this.getNaesbInfoDetailForm();
  refreshRequiredData$ = new BehaviorSubject(null);
  loading$ = new BehaviorSubject<boolean>(true);
  refreshNaesb$ = new BehaviorSubject<number>(null);

  ngOnInit(): void {
    this.naesbInfoDetailForm.enable();
    this.refreshNaesb$.next(this.contractId());
  }

  getNaesbInfoDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<MainContractNaesbInfoDetail> = fb.group({
      achPayment: fb.ctr(true),
      addendumExhibitB: fb.ctr(false),
      addendumExhibitC: fb.ctr(false),
      addendumExhibitD: fb.ctr(false),
      addendumExhibitE: fb.ctr(false),
      addendumExhibitF: fb.ctr(false),
      addendumExhibitG: fb.ctr(false),
      addendumExhibitH: fb.ctr(false),
      addendumExhibitI: fb.ctr(false),
      addendumExhibitJ: fb.ctr(false),
      addendumExhibitK: fb.ctr(false),
      addendumExhibitL: fb.ctr(false),
      additionalEventTypeId: fb.ctr(1),
      otherAgreementSetoffsTypeId: fb.ctr(1),
      checkPayment: fb.ctr(false),
      choiceOfLaw: fb.ctr(null),
      confidentialityTypeId: fb.ctr(1),
      confirmDeadline: fb.ctr(2),
      confirmingPartyTypeId: fb.ctr(3),
      earlyTerminationTypeId: fb.ctr(1),
      nettingTypeId: fb.ctr(1),
      paymentDateOptionId: fb.ctr(2),
      performanceObligationTypeId: fb.ctr(2),
      spotPricePublicationTypeId: fb.ctr(1),
      taxesOptionId: fb.ctr(1),
      transactionProcedureTypeId: fb.ctr(1),
      wireTransferPayment: fb.ctr(true),
      paymentDateTypeId: fb.ctr(null),
      spotPricePublicationOther: fb.ctr(null),
      amendmentDate: fb.ctr(null),
    });
    return fg;
  }

  saveNaesbInfo() {
    this.saveClicked.emit();
  }

  closeNaesbInfo() {
    this.closeClicked.emit();
  }

  getNaesbInfo(): MainContractNaesbInfoDetail {
    const naesbInfoItems = this.naesbInfoDetailForm.value as MainContractNaesbInfoDetail;
    return naesbInfoItems;
  }

  spotPriceText() {
    util.focusInputByClassName('spotPriceOtherElem');
  }

  paymentDateText() {
    util.focusInputByClassName('paymentDateTextElem');
  }

  paymentDateBlankText() {
    util.focusInputByClassName('paymentDateBlankTextElem');
  }

  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => {
      this.loading$.next(true);
    }),
    switchMap(refreshType => {
      return combineLatest([this.service.requiredData$, of(refreshType)]);
    }),
    map(([requiredData]) => {
      this.localRequiredData = requiredData;
      return requiredData;
    }),
    tap((requiredData) => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  naesb$ = this.refreshNaesb$.pipe(
    tap(() => {
      this.loading$.next(true);
    }),
    switchMap(contractId => {
      return this.service.getNaesbInfo(contractId);
    }),
    map(detail => {
      util.convertToDates(detail, ['paymentDateOptionId', 'paymentDateTypeId']);
      this.naesbInfoDetailForm.setValue(detail);
      return detail;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  getCheckedPayOptions = (): PayOptions => {
    return {
      isWireChecked: this.naesbInfoDetailForm.get('wireTransferPayment').value,
      isAchChecked: this.naesbInfoDetailForm.get('achPayment').value,
      isChecksChecked: this.naesbInfoDetailForm.get('checkPayment').value
    }
  }

  paymentDateTypeIdChanged$ = this.naesbInfoDetailForm.get('paymentDateTypeId').valueChanges.pipe(
    filter(() => {
      return !this.loading$.value;
    }),
    tap((paymentDateTypeId: number) => {
      if (!util.isNullOrWhitespace(paymentDateTypeId)) {
        this.naesbInfoDetailForm.patchValue({ paymentDateOptionId: 4 });
      }
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(1)
  )

  spotPricePublicationOtherChanged$ = this.naesbInfoDetailForm.get('spotPricePublicationOther').valueChanges.pipe(
    filter(() => {
      return !this.loading$.value;
    }),
    tap((spotPricePublicationOther: string) => {
      if (!util.isNullOrWhitespace(spotPricePublicationOther)) {
        this.naesbInfoDetailForm.patchValue({ spotPricePublicationTypeId: 2 });
      }
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(1)
  )

  loadingComplete$ = zip(this.naesb$, this.requiredData$).pipe(
    tap(() => {
      this.loading$.next(false);
    })
  )

  filterStates$ = new BehaviorSubject<string>(null)
  states$ = this.filterStates$.pipe(util.filterIdNames(this.loading$, this.requiredData$, 'states'));

  filterCountries$ = new BehaviorSubject<string>(null)
  countries$ = this.filterCountries$.pipe(util.filterIdNames(this.loading$, this.requiredData$, 'countries'));

  filterPayments$ = new BehaviorSubject<string>(null)
  payments$ = this.filterPayments$.pipe(util.filterIdNames(this.loading$, this.requiredData$, 'payments'));

}
