import { inject, Injectable } from '@angular/core';
import { HttpClient } from '@angular/common/http';
import { OdataGridHelper } from '../_shared/utils/odata-grid-helper';
import { map, Observable } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { GridDataResult } from '@progress/kendo-angular-grid';
import * as util from '../_shared/utils/util';
import { convertDatesToDateOnlyStrings } from '../_shared/utils/util';
import { MainContractContactDetail } from './contact-info/contact-info.service';
import { MainContractAccountingDetail } from './accounting-info/accounting-info.service';
import { MainContractExhibitBInfo } from './exhibitb-info/exhibitb-info.service';
import { MainContractEntityDetail } from './entity-info/entity-info.service';
import { MainContractExhibitCDetail } from './exhibitc-info/exhibitc-info.service';
import { MainContractOtherInfoDetail } from './other-info/other-info.service';
import { MainContractCrudeDetail } from './crude-info/crude-info.service';
import { MainContractNaesbInfoDetail } from './naesb2006/naesb2006.service';

export enum InfoType {
  Internal = 1,
  Counterparty = 2,
  Main = 3
}

export interface InfoCaptureItem {
  name: string;
}

export interface Item {
  MainContractId: number;
  ContractNum: string;
  TheirContractNum: string;
  ContractName: string;
  InternalEntity: string;
  Counterparty: string;
  Product: string;
  Status: string;
  EffectiveDate: Date;
  ExecutionDate: Date;
  LastAmendment: Date;
  Termination: string;
}

export interface Detail {
  mainContractId: number;
  contractName: string;
  contractNum: string;
  theirContractNum: string;
  productId: number;
  internalEntityId: number;
  counterpartyId: number;
  contractStatusId: number;
  signerId: number;
  isPaymentNettingAllowed: boolean;
  effectiveDate: Date;
  executionDate: Date;
  terminationNoticeDate: Date;
  terminationEffectiveDate: Date;
  lastAmendmentDate: Date;
  uploads: util.DocItemWithInactive[];
  notes: string;
  contactInfo: MainContractContactDetail;
  accountingInfo: MainContractAccountingDetail;
  exhibitCInfo: MainContractExhibitCDetail;
  exhibitBInfo: MainContractExhibitBInfo;
  naesbInfo: MainContractNaesbInfoDetail;
  crudeInfo: MainContractCrudeDetail;
  entityInfo: MainContractEntityDetail;
  otherInfo: MainContractOtherInfoDetail;
  payToBankId: number;
  payFromBankId: number
  payBankItems: PaymentBankItem[];
}

export interface GenerateSelections {
  isGenMain: boolean,
  isGenExB: boolean,
  isGenExC: boolean
}

export enum SaveType {
  New = 1,
  Normal = 2
}

export enum CloseType {
  Main = 1,
  Info = 2,
  Skip = 3
}

export interface RequiredData {
  hasModifyPermission: boolean;
  counterparties: util.Entity[];
  statuses: util.IdName[];
  products: util.IdName[];
  entities: util.Entity[];
  states: util.IdName[];
  countries: util.IdName[];
  signers: util.IdName[];
  userId: number;
  payments: util.IdName[];
  crudePoints: util.IdName[];
}

export interface PaymentBankItem {
  id: number;
  name: string;
  bankName: string;
  accountNum: string;
  displayText: string;
}


@Injectable({
  providedIn: 'root'
})

export class MainContractService extends OdataGridHelper {
  private baseUrl = `${window.location.origin}/api/MainContract`;
  http = inject(HttpClient);

  requiredData$ = this.http.get<RequiredData>(`${this.baseUrl}/getRequiredData`)

  getItems(state: State): Observable<GridDataResult> {
    return this.fetch(`GetMainContractItems`, state, null, false)
  }

  getDetail(id: number): Observable<Detail> {
    const url = `${this.baseUrl}/GetDetail?id=${id}`;
    return this.http.get<Detail>(url);
  }

  getNewContractNum(): Observable<string> {
    const url = `${this.baseUrl}/GetNewContractNum`;
    return this.http.get<string>(url);
  }

  generateContracts(mainContractId: number, genMain: boolean, genExB: boolean, genExC: boolean): Observable<number> {
    const url = `${this.baseUrl}/GenerateContracts?mainContractId=${mainContractId}&genMain=${genMain}&genExB=${genExB}&genExC=${genExC}`;
    return this.http.get<number>(url);
  }

  saveDetail(detail: Detail, saveType: SaveType, infoType: InfoType): Observable<number> {
    const url = `${this.baseUrl}/SaveDetail?saveType=${saveType}&infoType=${infoType}`;
    detail = convertDatesToDateOnlyStrings(detail, ['effectiveDate', 'executionDate', 'terminationNoticeDate', 'terminationEffectiveDate', 'lastAmendmentDate']);
    detail.exhibitCInfo = convertDatesToDateOnlyStrings(detail.exhibitCInfo, ['amendmentDate']);
    detail.naesbInfo = convertDatesToDateOnlyStrings(detail.naesbInfo, ['amendmentDate']);

    if (detail.exhibitBInfo?.exhibitBDetail)
      detail.exhibitBInfo.exhibitBDetail = convertDatesToDateOnlyStrings(detail.exhibitBInfo.exhibitBDetail, ['exhibitBAmendmentDate']);

    if (detail.exhibitBInfo?.exhibitBDetail?.items)
      detail.exhibitBInfo.exhibitBDetail.items = convertDatesToDateOnlyStrings(detail.exhibitBInfo.exhibitBDetail.items, ['inactiveDate'], true);

    if (detail.exhibitBInfo?.exhibitBItemsToDelete)
      detail.exhibitBInfo.exhibitBItemsToDelete = convertDatesToDateOnlyStrings(detail.exhibitBInfo.exhibitBItemsToDelete, ['inactiveDate'], true);

    if (detail.crudeInfo?.calcItems)
      detail.crudeInfo.calcItems = convertDatesToDateOnlyStrings(detail.crudeInfo.calcItems, ['effectiveDate'], true);

    return this.http.put<number>(url, detail);
  }

  deleteDetail(id: number) {
    const url = `${this.baseUrl}/DeleteDetail/${id}`;
    return this.http.delete(url);
  }

  exportItems(state: State, fileNameOriginal: string) {
    return this.fetchExport(`GetMainContractItems`, state, null, fileNameOriginal);
  }

  downloadDoc(fileNameOriginal: string, fileNameOnDisk: string): Observable<{ fileBlob: Blob, fileName: string }> {
    const url = `${this.baseUrl}/DownloadDoc?fileNameOnDisk=${fileNameOnDisk}`;
    return this.http.get(url, { observe: 'response', responseType: 'blob' }).pipe(
      map(result => {
        const fileBlob = result.body;
        const fileName: string = result.body.type === "text/plain" ? "Error.txt" : fileNameOriginal;
        return { fileBlob, fileName };
      })
    );
  }

}
