import { Component, ChangeDetectionStrategy, ChangeDetectorRef, ViewChild, ElementRef, ViewEncapsulation, HostListener, OnInit, ViewChildren, QueryList, AfterViewInit, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { tap, take, map, catchError, switchMap, filter, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, Subject, combineLatest } from 'rxjs';
import { CompositeFilterDescriptor, State } from '@progress/kendo-data-query';
import { MessageService } from '../_shared/services/message.service';
import { AbstractControl, UntypedFormGroup, Validators } from '@angular/forms';
import { DialogService, DialogSettings } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../_shared/services/notify.service';
import { TooltipDirective } from '@progress/kendo-angular-tooltip';
import * as util from '../_shared/utils/util';
import { DrawerItem, DrawerSelectEvent, ExpansionPanelComponent, KENDO_DRAWER } from '@progress/kendo-angular-layout';
import { MainContractService, Item, Detail, SaveType, InfoType as InfoType, CloseType, GenerateSelections } from './main-contract.service';
import { ErrorEvent, SuccessEvent, UploadComponent, UploadProgressEvent } from '@progress/kendo-angular-upload';
import dayjs from 'dayjs';
import { NumericTextBoxComponent } from '@progress/kendo-angular-inputs';
import { ActivatedRoute } from '@angular/router';
import { HttpParams } from '@angular/common/http';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';
import { AccountingInfoComponent } from '../main-contract/accounting-info/accounting-info.component';
import { ContactInfoComponent } from '../main-contract/contact-info/contact-info.component';
import { CrudeInfoComponent } from '../main-contract/crude-info/crude-info.component';
import { EntityInfoComponent } from '../main-contract/entity-info/entity-info.component';
import { ExhibitBInfoComponent } from '../main-contract/exhibitb-info/exhibitb-info.component';
import { ExhibitCInfoComponent } from '../main-contract/exhibitc-info/exhibitc-info.component';
import { Naesb2006Component } from '../main-contract/naesb2006/naesb2006.component';
import { OtherInfoComponent } from '../main-contract/other-info/other-info.component';
import { MainContractContactDetail } from './contact-info/contact-info.service';
import { MainContractExhibitBDetail, MainContractExhibitBInfo } from './exhibitb-info/exhibitb-info.service';
import { MainContractNaesbInfoDetail } from './naesb2006/naesb2006.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { HesitateDirective } from '../_shared/directives/hesitate-directive';
import { DocInactivePipe } from './doc-inactive-name.pipe';
import { HoverClassDirective } from '../_shared/directives/hover-class-directive';
import { SetHasItemPipe } from './set-has-item.pipe';

function ValidateEffectiveDate(c: AbstractControl): { [key: string]: boolean } | null {
  if (c.parent) {
    return null
  } else return null;
}

@Component({
  selector: 'app-main-contract',
  templateUrl: './main-contract.component.html',
  styleUrls: ['./main-contract.component.scss'],
  encapsulation: ViewEncapsulation.None,
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON, KENDO_DRAWER, HesitateDirective, HoverClassDirective, DocInactivePipe, SetHasItemPipe,
    AccountingInfoComponent,
    ContactInfoComponent,
    CrudeInfoComponent,
    EntityInfoComponent,
    ExhibitBInfoComponent,
    ExhibitCInfoComponent,
    Naesb2006Component,
    OtherInfoComponent
  ]
})
export class MainContractComponent implements OnInit, AfterViewInit {
  @ViewChild("grid", { read: ElementRef }) kendoGridEl: ElementRef;
  @ViewChild('tooltipGrid') tooltipGrid: TooltipDirective;
  @ViewChild('docUploadTarget') docUploadElem: UploadComponent;
  @ViewChild('wirePanel') wirePanelElem: ExpansionPanelComponent;
  @ViewChild('achPanel') achPanelElem: ExpansionPanelComponent;
  @ViewChild('checkPanel') checkPanelElem: ExpansionPanelComponent;
  @ViewChild('acctInfo') acctInfoElem: AccountingInfoComponent;
  @ViewChild('contactInfo') contactInfoElem: ContactInfoComponent;
  @ViewChild('crudeInfo') crudeInfoElem: CrudeInfoComponent;
  @ViewChild('entityInfo') entityInfoElem: EntityInfoComponent;
  @ViewChild('exhibitBInfo') exhibitBInfoElem: ExhibitBInfoComponent;
  @ViewChild('exhibitCInfo') exhibitCInfoElem: ExhibitCInfoComponent;
  @ViewChild('naesbInfo') naesbInfoElem: Naesb2006Component;
  @ViewChild('otherInfo') otherInfoElem: OtherInfoComponent;
  @ViewChildren('crudeCalcElem') crudeCalcElems: QueryList<NumericTextBoxComponent>;
  @HostListener('window:resize') onResize() {
    //this function is empty but for some reason it helps the window to resize faster
  };

  entityId: number; //may be internal entity, or it may be counterparty
  contractId: number;
  productId: number;
  naesbInfoItems: MainContractNaesbInfoDetail

  private service = inject(MainContractService);
  private messageService = inject(MessageService);
  private titleService = inject(Title);
  private fb = inject(CustomFormBuilder);
  private ref = inject(ChangeDetectorRef);
  private dialogService = inject(DialogService);
  private notify = inject(NotifyService);
  private activatedRoute = inject(ActivatedRoute);


  constructor() {
  }
  
  util = util;
  icons = util.icons;
  hasModifyPermission = false;
  gridScrollPosition: util.GridScrollPosition = { topPos: 0, leftPos: 0 };
  mySelection: number[] = [];
  checkedKeys: number[] = [];
  detailForm = this.getDetailForm();
  generateContractsForm = this.getGenerateContractsForm();
  generateContractsInitialValues: GenerateSelections = this.generateContractsForm.value as GenerateSelections;
  detailInitialValues: Detail = this.detailForm.value as Detail;
  selectedInfoItem: string;
  drawerItems: string[] = [];
  infoType: InfoType = InfoType.Main;
  infoTypeEnum = InfoType; //this is needed only for the html template
  docSaveUrl = `${window.location.origin}/api/MainContract/UploadDoc`;
  refreshBankPay$ = new BehaviorSubject(null);
  downloadDoc$ = new Subject<util.DocItemWithInactive>()
  refreshRequiredData$ = new BehaviorSubject(null);
  refreshDrawerItems$ = new Subject<void>();
  detailLoading$ = new BehaviorSubject<boolean>(true);
  infoLoading$ = new BehaviorSubject<boolean>(false);
  refreshDetail$ = new Subject<number>();
  detailOpened$ = new BehaviorSubject<boolean>(false);
  inactiveDetailOpened$ = new BehaviorSubject<boolean>(false);
  infoOpened$ = new BehaviorSubject<boolean>(false);
  setInfoTitle$ = new BehaviorSubject<string>(null);
  exporting$ = new BehaviorSubject<boolean>(false);
  exportClicked$ = new Subject();
  generateContractClicked$ = new Subject();
  gridLoading$ = new BehaviorSubject<boolean>(true);
  refreshItems$ = new BehaviorSubject<string>(null);
  save$ = new Subject<[SaveType, CloseType, util.AfterCompletedAction]>();
  delete$ = new Subject();
  generateContractsOpened$ = new BehaviorSubject<boolean>(false);
  inactiveDateForm: UntypedFormGroup;
  infoNames = new Set<string>();

  state: State = {
    filter: { filters: [], logic: 'and' },
    group: null,
    skip: 0,
    sort: [{ field: 'ContractNum', dir: 'asc' }],
    take: 50
  };

  ngOnInit(): void {
    const fb = this.fb;

    this.inactiveDateForm = fb.group({
      inactiveDate: [null],
      fileNameOnDisk: [null]
    });
  }

  ngAfterViewInit(): void {
    //this reads the query params to decide whether to load into deal or distribute mode
    this.activatedRoute.queryParams.subscribe(x => {
      const params = new HttpParams({ fromObject: x });
      const filter: CompositeFilterDescriptor = { filters: [], logic: 'and' };
      const filters = filter.filters;

      const counterparty = params.get('counterparty');
      if (counterparty)
        filters.push({ field: 'Counterparty', operator: 'eq', value: counterparty });

      const internalEntity = params.get('internalEntity');
      if (internalEntity)
        filters.push({ field: 'InternalEntity', operator: 'eq', value: internalEntity });

      this.state.filter = filter;
      this.dataStateChange(this.state);
    });
  }

  getGenerateContractsForm() {
    const fb = this.fb;
    const fg: util.FormModel<GenerateSelections> = fb.group({
      isGenMain: fb.ctr(true),
      isGenExB: fb.ctr(true),
      isGenExC: fb.ctr(true)
    });
    return fg;
  }

  getDetailForm() {
    const fb = this.fb;

    const newExhibitBInfo: MainContractExhibitBInfo = {
      exhibitBDetail: null,
      exhibitBSelectionDetail: null,
      exhibitBItemsToDelete: []
    };

    const fg: util.FormModel<Detail> = fb.group({
      mainContractId: fb.ctr(0, Validators.required),
      contractName: fb.ctr(null, Validators.required),
      contractNum: fb.ctr(null, Validators.required),
      theirContractNum: fb.ctr(null),
      productId: fb.ctr(3, Validators.required),
      internalEntityId: fb.ctr(315, Validators.required),
      counterpartyId: fb.ctr(null, Validators.required),
      contractStatusId: fb.ctr(1, [Validators.required, ValidateEffectiveDate]),
      signerId: fb.ctr(null),
      isPaymentNettingAllowed: fb.ctr(true),
      effectiveDate: fb.ctr(null),
      executionDate: fb.ctr(null),
      terminationEffectiveDate: fb.ctr(null, ValidateEffectiveDate),
      terminationNoticeDate: fb.ctr(null, ValidateEffectiveDate),
      lastAmendmentDate: fb.ctr(null),
      notes: fb.ctr(null),
      uploads: fb.ctr([]),
      accountingInfo: fb.ctr(null),
      exhibitCInfo: fb.ctr(null),
      otherInfo: fb.ctr(null),
      exhibitBInfo: fb.ctr(newExhibitBInfo),
      contactInfo: fb.ctr(null),
      entityInfo: fb.ctr(null),
      crudeInfo: fb.ctr(null),
      naesbInfo: fb.ctr(null),
      payFromBankId: fb.ctr(null),
      payToBankId: fb.ctr(null),
      payBankItems: fb.ctr(null),
    });
    return fg;
  }

  getExhibitBInfoDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<MainContractExhibitBDetail> = fb.group({
      title: fb.ctr(''),
      body: fb.ctr(''),
      showInactive: fb.ctr(false),
      items: fb.ctr([]),
      exhibitBAmendmentDate: fb.ctr(null),
    });
    return fg;
  }
  refreshDropdowns() {
    this.refreshRequiredData$.next(util.RefreshType.SelfOnly);
  }

  openDetail(id: number): void {
    this.detailOpened$.next(true);
    this.tooltipGrid.hide();
    util.saveGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
    this.refreshDetail$.next(id);
    this.contractId = id;
    this.generateContractsForm.setValue(this.generateContractsInitialValues);
  }

  onDetailClosing() {
    util.onDetailChanging(this.detailForm, this.dialogService, this.closeDetail, this.save);
  }

  onInfoClosing() {
    const infoForms = this.getInfoForms();
    util.onDetailChanging(infoForms, this.dialogService, this.closeInfo, this.saveInfo, { extraActionParamValue: util.AfterCompletedAction.CloseDetail });
  }

  openInfo(infoType: InfoType): void {
    this.infoType = infoType;
    this.setInfoNames();
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid) {
      this.productId = this.detailForm.get('productId').value;
      if (this.infoType === InfoType.Internal) {

        this.entityId = this.detailForm.get('internalEntityId').value;
        this.setInfoTitle$.next('Internal Entity Info')
      }
      else if (this.infoType === InfoType.Counterparty) {
        this.entityId = this.detailForm.get('counterpartyId').value;
        this.setInfoTitle$.next('Contract Info')
      }
      this.refreshDrawerItems$.next();
    } else {
      this.notify.error("validation failed");
    }
  }

  closeInfo = () => {
    this.infoType = InfoType.Main;
    this.infoNames = new Set<string>();
    this.infoOpened$.next(false);
  }

  closeDetail = (isFromInterface: boolean) => {
    this.detailOpened$.next(false);
    if (isFromInterface)
      util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
  }

  openInactiveDetail(fileNameOnDisk: string) {
    const detail: Detail = this.detailForm.value as Detail;
    const docs = detail.uploads;
    const item = docs.find(item => item.fileNameOnDisk == fileNameOnDisk);
    if (item.inactiveDate !== null) {
      this.inactiveDateForm.reset();
      const detail: Detail = this.detailForm.value as Detail;
      const docs = detail.uploads;
      const item = docs.find(item => item.fileNameOnDisk == fileNameOnDisk);
      item.inactiveDate = null;
      this.detailForm.patchValue({ uploads: docs });
      this.detailForm.markAsDirty();
      this.sortDocs();
    }
    else {
      this.inactiveDetailOpened$.next(true);
      this.inactiveDateForm.patchValue({ inactiveDate: util.currentDate.toDate(), fileNameOnDisk: fileNameOnDisk });
    }
  }

  onInactiveDetailClosing() {
    this.inactiveDateForm.reset();
    this.inactiveDetailOpened$.next(false);
  }

  setInactiveDate() {
    const inactiveDate: Date = this.inactiveDateForm.get('inactiveDate').value;
    const fileNameOnDisk: string = this.inactiveDateForm.get('fileNameOnDisk').value;
    const detail: Detail = this.detailForm.value as Detail;
    const docs = detail.uploads;
    const item = docs.find(item => item.fileNameOnDisk == fileNameOnDisk);
    item.inactiveDate = inactiveDate;
    this.detailForm.patchValue({ uploads: docs });
    this.detailForm.markAsDirty();
    this.inactiveDetailOpened$.next(false);
    this.sortDocs();
  }

  sortDocs() {
    const detail: Detail = this.detailForm.value as Detail;
    const docs = detail.uploads;

    docs.sort(function (a, b) {
      //if a is active and b is inactive
      if (a.inactiveDate == null && b.inactiveDate !== null) {
        return -1
      }
      //if both a and b are active
      if (a.inactiveDate == null && b.inactiveDate == null) {
        //sort by name
        if (a.fileNameOriginal < b.fileNameOriginal) {
          return -1;
        }
        else if (a.fileNameOriginal > b.fileNameOriginal) {
          return 1;
        }
      }
      //if both a and b are inactive
      else if (a.inactiveDate !== null && b.inactiveDate !== null) {
        //if same inactive date
        if (dayjs(a.inactiveDate).isSame(b.inactiveDate)) {
          //sort by name
          if (a.fileNameOriginal < b.fileNameOriginal) {
            return -1;
          }
          else if (a.fileNameOriginal > b.fileNameOriginal) {
            return 1;
          }
        }
        //otherwise sort by inactive date
        else if (a.inactiveDate > b.inactiveDate) {
          return -1;
        }
        else if (b.inactiveDate > a.inactiveDate) {
          return 1;
        }
      }
    });
  }

  detailFinishedLoading(): void {
    this.detailForm.enable();
    this.detailLoading$.next(false);
    util.focusInputTarget();
  }

  infoFinishedSaving(afterCompletedAction: util.AfterCompletedAction): void {
    this.infoLoading$.next(false);
    if (afterCompletedAction === util.AfterCompletedAction.CloseDetail)
      this.closeInfo();
  }

  add(): void {
    this.openDetail(0);
  }

  edit(dataItem: Item): void {
    this.openDetail(dataItem.MainContractId);
  }

  save = (saveType: util.SaveType) => {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid)
      this.save$.next([saveType, CloseType.Main, util.AfterCompletedAction.CloseDetail]);
    else
      this.notify.error("validation failed");
  }

  pasteCommercialContact(): void {
    const contactDetail = this.contactInfoElem.getCommercialInfo() as MainContractContactDetail;
    this.acctInfoElem.pasteCommercial(contactDetail);
  }

  saveAllComponents(): void {
    this.saveInfo(util.SaveType.Normal, util.AfterCompletedAction.DoNothing);
  }

  updateAccountingExhibitC(addendumExhibitC: boolean): void {
    this.acctInfoElem.updateExhibitC(addendumExhibitC);
  }

  saveInfo = (saveType: util.SaveType, afterCompletedAction: util.AfterCompletedAction) => {
    this.save$.next([saveType, CloseType.Info, afterCompletedAction]);
  }

  saveWithoutClose = (saveType: util.SaveType) => {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid)
      this.save$.next([saveType, CloseType.Skip, util.AfterCompletedAction.DoNothing]);
    else
      this.notify.error("validation failed");
  }

  delete(): void {
    const deleteConfirmSettings: DialogSettings = {
      title: "Please confirm",
      content: "Are you sure you want to delete this item?",
      actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
      cssClass: 'utilPrompt'
    }

    this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
      if (util.getDialogAction(result) === util.dialogAction.Yes)
        this.delete$.next(null);
    });
  }

  docComplete() {
    this.docUploadElem.clearFiles();
  }

  downloadDoc(doc: util.DocItemWithInactive) {
    this.downloadDoc$.next(doc)
  }

  removeDoc(doc: util.DocItemWithInactive) {
    const docs: util.DocItemWithInactive[] = this.detailForm.get('uploads').value;
    const indexToRemove = docs.findIndex(x => x.fileNameOnDisk === doc.fileNameOnDisk);
    docs.splice(indexToRemove, 1);
    this.detailForm.patchValue({ uploads: docs });
    this.detailForm.markAsDirty();
  }

  docUploadSuccess(value: SuccessEvent) {
    const newdocitems: util.DocItemWithInactive[] = value.response.body;
    let docs: util.DocItemWithInactive[] = this.detailForm.get('uploads').value;
    if (docs)
      docs = docs.concat(newdocitems);
    else
      docs = newdocitems;
    this.detailForm.patchValue({ uploads: docs });
    this.detailForm.markAsDirty();

    this.sortDocs();
  }

  docUploadError(value: ErrorEvent) {
    util.handleError(value.response, this.messageService);
  }

  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  uploadProgress(event: UploadProgressEvent) {
    this.ref.detectChanges();
  }

  isGenerateDisabled(): boolean {
    const gen: GenerateSelections = this.generateContractsForm.value as GenerateSelections;
    const isDisabled = !gen.isGenMain && !gen.isGenExB && !gen.isGenExC;
    return isDisabled;
  }

  onGenerateContractsOpened() {
    util.onDetailChanging(this.detailForm, this.dialogService, this.openGenerateContracts, this.saveWithoutClose);
  }

  openGenerateContracts = () => {
    this.generateContractsOpened$.next(true);
  }

  closeGenerateContracts() {
    this.generateContractsOpened$.next(false);
  }

  generateContracts() {
    this.closeGenerateContracts();
    this.generateContractClicked$.next(null);
  }

  export(): void {
    this.exportClicked$.next(null);
  }

  dataStateChange(state: State): void {
    this.gridScrollPosition.topPos = 0;
    this.gridScrollPosition.leftPos = 0;
    util.fixKendoQueryFilter(state.filter);
    this.state = state;
    this.refreshItems$.next(null);
  }

  onDrawerSelection($event: DrawerSelectEvent) {
    this.selectedInfoItem = $event.item.text;
  }

  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => this.detailLoading$.next(true)),
    switchMap(refreshType => {
      return combineLatest([this.service.requiredData$, of(refreshType)]);
    }),
    map(([requiredData, refreshType]) => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
      if (refreshType === util.RefreshType.SelfOnly)
        this.detailLoading$.next(false);
      return requiredData;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  items$ = this.refreshItems$.pipe(
    tap(() => {
      this.gridLoading$.next(true);
    }),
    switchMap(() => {
      return this.service.getItems(this.state);
    }),
    tap(() => {
      this.gridLoading$.next(false);
      util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  exportAction$ = this.exportClicked$.pipe(
    tap(() => {
      this.exporting$.next(true);
    }),
    switchMap(() => {
      return this.service.exportItems(this.state, 'MainContracts.xlsx');
    }),
    tap(res => {
      util.openOrSaveFile(res.fileBlob, res.fileName);
      this.exporting$.next(false);
    }),
    shareReplay(1),
    catchError(err => {
      this.exporting$.next(false);
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  detail$ = combineLatest([this.refreshDetail$, this.requiredData$]).pipe(
    switchMap(([id, requiredData]) => {
      this.detailLoading$.next(true);
      this.detailForm.reset();
      this.detailForm.disable();
      if (id === 0)
        return combineLatest([of(id), this.service.getNewContractNum(), of(requiredData)]);
      else
        return combineLatest([of(id), of(''), of(requiredData)]);
    }),
    switchMap(([id, newContractNum, requiredData]) => {
      if (id === 0) {
        this.detailInitialValues.contractNum = newContractNum;
        this.detailInitialValues.signerId = requiredData.userId;
        return of(this.detailInitialValues);
      }
      else
        return this.service.getDetail(id);
    }),
    map(result => {
      if (result) {
        util.convertToDates(result, ["paymentDateOptionId", "paymentDateTypeId"]);
        this.detailForm.setValue(result);
      }
      return result;
    }),
    tap(() => {
      const payFromBankId = this.detailForm.get('payFromBankId').value;
      const payToBankId = this.detailForm.get('payToBankId').value;
      const payBankItems = this.detailForm.get('payBankItems').value;
      if (!payBankItems || payBankItems.findIndex(item => item.id === payToBankId) === -1) {
        this.detailForm.patchValue({ payToBankId: null });
      }
      if (!payBankItems || payBankItems.findIndex(item => item.id === payFromBankId) === -1) {
        this.detailForm.patchValue({ payFromBankId: null });
      }
      this.detailFinishedLoading();
    }),
    shareReplay(1),
    catchError(err => {
      this.closeDetail(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  saveResult$ = this.save$.pipe(
    switchMap(([saveType, closeType, afterCompletedAction]) => {
      if (closeType === CloseType.Main || closeType === CloseType.Skip) {
        this.detailLoading$.next(true);
        this.detailForm.disable();
      }
      else if (closeType === CloseType.Info) {
        this.infoLoading$.next(true);
      }

      const itemToSave: Detail = this.detailForm.value as Detail;
      itemToSave.payBankItems ??= [];

      const infoForms = this.getInfoForms();
      infoForms.forEach(infoForm => {
        if (infoForm === this.acctInfoElem?.accountingInfoDetailForm)
          itemToSave.accountingInfo = infoForm.getRawValue();

        if (infoForm === this.contactInfoElem?.contactInfoDetailForm)
          itemToSave.contactInfo = infoForm.getRawValue();

        if (infoForm === this.crudeInfoElem?.crudeInfoDetailForm)
          itemToSave.crudeInfo = infoForm.getRawValue();

        if (infoForm === this.exhibitBInfoElem?.exhibitBInfoDetailForm) {
          itemToSave.exhibitBInfo.exhibitBDetail = infoForm.getRawValue();
          itemToSave.exhibitBInfo.exhibitBSelectionDetail = this.exhibitBInfoElem.getExhibitBSelectionDetail();
          itemToSave.exhibitBInfo.exhibitBItemsToDelete = this.exhibitBInfoElem.getItemsForDeletion();
        }

        if (infoForm === this.exhibitCInfoElem?.exhibitCInfoDetailForm)
          itemToSave.exhibitCInfo = infoForm.getRawValue();

        if (infoForm === this.naesbInfoElem?.naesbInfoDetailForm)
          itemToSave.naesbInfo = infoForm.getRawValue();

        if (infoForm === this.otherInfoElem?.otherInfoDetailForm)
          itemToSave.otherInfo = infoForm.getRawValue();
      });

      return combineLatest([of(closeType), this.service.saveDetail(itemToSave, saveType, this.infoType), of(afterCompletedAction)]);
    }),
    tap(([closeType, contractId, afterCompletedAction]) => {
      const infoForms = this.getInfoForms();
      infoForms.forEach(form => {
        form.markAsPristine();
      });

      if (closeType === CloseType.Main)
        this.closeDetail(false);
      else if (closeType === CloseType.Info)
        this.infoFinishedSaving(afterCompletedAction);

      this.notify.success('save successful');
      this.contractId = contractId;
      this.refreshDetail$.next(contractId);
      this.refreshItems$.next(null);
      this.mySelection = [contractId];
    }),
    shareReplay(1),
    catchError(err => {
      this.detailFinishedLoading();
      this.infoFinishedSaving(util.AfterCompletedAction.DoNothing);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  deleteResult$ = this.delete$.pipe(
    switchMap(() => {
      this.detailLoading$.next(true);
      this.detailForm.disable();
      const itemToDelete: Detail = this.detailForm.getRawValue();
      return this.service.deleteDetail(itemToDelete.mainContractId);
    }),
    tap(() => {
      this.notify.success('delete successful');
      this.closeDetail(false);
      this.refreshItems$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      this.detailForm.enable();
      this.detailLoading$.next(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  infoDrawerItems$ = this.refreshDrawerItems$.pipe(
    filter(() => this.detailLoading$.value === false),
    tap(() => {
      this.detailLoading$.next(true);
    }),
    map(() => {
      const infoDrawerItems: Array<DrawerItem> = [];

      this.drawerItems = Array.from(this.infoNames);
      this.drawerItems.forEach(str => {
        const item: DrawerItem = { text: str, icon: 'k-i-data' }
        infoDrawerItems.push(item);
      });

      infoDrawerItems[0].selected = true;
      this.selectedInfoItem = infoDrawerItems[0].text;

      return infoDrawerItems;
    }),
    tap(() => {
      this.detailLoading$.next(false);
      this.infoOpened$.next(true);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  infoTitle$ = this.setInfoTitle$.pipe(
    map(title => {
      return title;
    })
  )

  generateContracts$ = this.generateContractClicked$.pipe(
    tap(() => {
      this.detailLoading$.next(true);
    }),
    switchMap(() => {
      const mainContractId = this.detailForm.get('mainContractId').value;
      const gen: GenerateSelections = this.generateContractsForm.value as GenerateSelections;
      return this.service.generateContracts(mainContractId, gen.isGenMain, gen.isGenExB, gen.isGenExC);
    }),
    tap(mainContractId => {
      this.notify.success('contracts generated');
      this.refreshDetail$.next(mainContractId);
    }),
    shareReplay(1),
    catchError(err => {
      this.detailLoading$.next(false);
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  contractStatusChanged$ = this.detailForm.get('contractStatusId').valueChanges.pipe(
    filter(() => {
      return !this.detailLoading$.value
    }),
    tap(() => {
      const effectiveDate = this.detailForm.get('effectiveDate');
      effectiveDate.updateValueAndValidity();
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(1)
  )

  title$ = of('Main Contract').pipe(
    tap((title) => util.trySetTitle(this.titleService, title))
  )

  downloadDocResult$ = this.downloadDoc$.pipe(
    filter(docItem => {
      return docItem && docItem.fileNameOnDisk !== null;
    }),
    switchMap(fileNamePair => {
      return this.service.downloadDoc(fileNamePair.fileNameOriginal, fileNamePair.fileNameOnDisk);
    }),
    tap(res => {
      util.openOrSaveFile(res.fileBlob, res.fileName);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  setInfoNames() {
    this.infoNames.clear();
    const productId: number = this.detailForm.get('productId').value;
    const isCrudeProduct = productId && productId === 3;
    const isNatGasProduct = productId && productId === 1;

    if (this.infoType === InfoType.Internal) {
      this.infoNames.add('Entity Info');
      this.infoNames.add('Accounting Info');
      this.infoNames.add('Contact Info');
      this.infoNames.add('Exhibit C');
      this.infoNames.add('Other Info');
    }

    if (this.infoType === InfoType.Counterparty) {
      this.infoNames.add('Entity Info');
      this.infoNames.add('Accounting Info');
      this.infoNames.add('Contact Info');
      this.infoNames.add('Exhibit B');
      this.infoNames.add('Exhibit C');
    }

    if (this.infoType === InfoType.Counterparty && isCrudeProduct)
      this.infoNames.add('Crude Oil');

    if (this.infoType === InfoType.Counterparty && isNatGasProduct)
      this.infoNames.add('NAESB 2006');
  }

  getInfoForms() {
    const infoForms: UntypedFormGroup[] = [];

    if (this.infoNames.has('Accounting Info'))
      infoForms.push(this.acctInfoElem.accountingInfoDetailForm);

    if (this.infoNames.has('Contact Info'))
      infoForms.push(this.contactInfoElem.contactInfoDetailForm);

    if (this.infoNames.has('Crude Oil'))
      infoForms.push(this.crudeInfoElem.crudeInfoDetailForm);

    if (this.infoNames.has('Entity Info'))
      infoForms.push(this.entityInfoElem.entityInfoDetailForm);

    if (this.infoNames.has('Exhibit B'))
      infoForms.push(this.exhibitBInfoElem.exhibitBInfoDetailForm);

    if (this.infoNames.has('Exhibit C'))
      infoForms.push(this.exhibitCInfoElem.exhibitCInfoDetailForm);

    if (this.infoNames.has('NAESB 2006'))
      infoForms.push(this.naesbInfoElem.naesbInfoDetailForm);

    if (this.infoNames.has('Other Info'))
      infoForms.push(this.otherInfoElem.otherInfoDetailForm)

    return infoForms;
  }

  filterCounterparties$ = new BehaviorSubject<string>(null)
  counterparties$ = this.filterCounterparties$.pipe(util.filterSpecials(this.detailLoading$, this.requiredData$, 'counterparties', 'fullName'));

  filterStatuses$ = new BehaviorSubject<string>(null)
  statuses$ = this.filterStatuses$.pipe(util.filterSpecials(this.detailLoading$, this.requiredData$, 'statuses', 'name'));

  filterProducts$ = new BehaviorSubject<string>(null)
  products$ = this.filterProducts$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'products'));

  filterEntities$ = new BehaviorSubject<string>(null)
  entities$ = this.filterEntities$.pipe(util.filterSpecials(this.detailLoading$, this.requiredData$, 'entities', 'fullName'));

  filterStates$ = new BehaviorSubject<string>(null)
  states$ = this.filterStates$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'states'));

  filterCountries$ = new BehaviorSubject<string>(null)
  countries$ = this.filterCountries$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'countries'));

  filterSigners$ = new BehaviorSubject<string>(null)
  signers$ = this.filterSigners$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'signers'));

  filterPayments$ = new BehaviorSubject<string>(null)
  payments$ = this.filterPayments$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'payments'));

  filterCrudePoints$ = new BehaviorSubject<string>(null)
  crudePoints$ = this.filterCrudePoints$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'crudePoints'));

  filterPayToBanks$ = new BehaviorSubject<string>(null)
  payToBanks$ = this.filterPayToBanks$.pipe(util.filterSpecials(this.detailLoading$, this.detail$, 'payBankItems', 'displayText'));

  filterPayFromBanks$ = new BehaviorSubject<string>(null)
  payFromBanks$ = this.filterPayFromBanks$.pipe(util.filterSpecials(this.detailLoading$, this.detail$, 'payBankItems', 'displayText'));
}
