
import { Component, ChangeDetectionStrategy, ChangeDetectorRef, output, input, OnInit, ViewEncapsulation, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { tap, map, catchError, switchMap, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, combineLatest, zip } from 'rxjs';
import { MessageService } from '../../_shared/services/message.service';
import { DialogService } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../../_shared/services/notify.service';
import * as util from '../../_shared/utils/util'
import { ActivatedRoute } from '@angular/router';
import { CustomFormBuilder } from '../../_shared/services/custom-form-builder.service';
import { ExhibitCInfoService, MainContractExhibitCDetail, RequiredData } from './exhibitc-info.service';
import { InfoType } from '../main-contract.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../../app.config';
import { KENDO_MASKEDTEXTBOX } from '@progress/kendo-angular-inputs';

@Component({
  selector: 'app-exhibitc-info',
  templateUrl: './exhibitc-info.component.html',
  styleUrl: './exhibitc-info.component.scss',
  changeDetection: ChangeDetectionStrategy.OnPush,
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON, KENDO_MASKEDTEXTBOX]
})

export class ExhibitCInfoComponent implements OnInit {
  contractId = input.required<number>();
  entityId = input.required<number>();
  productId = input.required<number>();
  entityType = input.required<InfoType>();
  saveClicked = output();
  closeClicked = output();

  private service = inject(ExhibitCInfoService);
  private messageService = inject(MessageService);
  private titleService = inject(Title);
  private fb = inject(CustomFormBuilder);
  private ref = inject(ChangeDetectorRef);
  private dialogService = inject(DialogService);
  private notify = inject(NotifyService);
  private activatedRoute = inject(ActivatedRoute);

  constructor() {

  }
  util = util;
  icons = util.icons;
  localRequiredData: RequiredData;
  hasModifyPermission = false;
  exhibitCInfoDetailForm = this.getExhibitCInfoDetailForm();
  refreshRequiredData$ = new BehaviorSubject(null);
  loading$ = new BehaviorSubject<boolean>(true);
  refreshExhibitCInfo$ = new BehaviorSubject<void>(null);

  ngOnInit(): void {
    this.refreshExhibitCInfo$.next();
  }

  getExhibitCInfoDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<MainContractExhibitCDetail> = fb.group({
      abaNumAch: fb.ctr(null),
      abaNumWire: fb.ctr(null),
      accountName: fb.ctr(null),
      accountNum: fb.ctr(null),
      alt1AbaNumAch: fb.ctr(null),
      alt1AbaNumWire: fb.ctr(null),
      alt1AccountName: fb.ctr(null),
      alt1AccountNum: fb.ctr(null),
      alt1BankName: fb.ctr(null),
      alt1CityState: fb.ctr(null),
      alt1CreditName: fb.ctr(null),
      alt1ForFurtherCreditTo: fb.ctr(null),
      alt2AbaNumAch: fb.ctr(null),
      alt2AccountName: fb.ctr(null),
      alt2AbaNumWire: fb.ctr(null),
      alt2AccountNum: fb.ctr(null),
      alt2BankName: fb.ctr(null),
      alt2CityState: fb.ctr(null),
      alt2CreditName: fb.ctr(null),
      alt2ForFurtherCreditTo: fb.ctr(null),
      checkPaymentAddressLine1: fb.ctr(null),
      checkPaymentAddressLine2: fb.ctr(null),
      checkPaymentCountryId: fb.ctr(1),
      bankName: fb.ctr(null),
      checkPaymentCity: fb.ctr(null),
      checkPaymentStateId: fb.ctr(null),
      checkPaymentZip: fb.ctr(null),
      cityState: fb.ctr(null),
      creditName: fb.ctr(null),
      forFurtherCreditTo: fb.ctr(null),
      amendmentDate: fb.ctr(null),
    });
    return fg;
  }

  saveExhibitCInfo() {
    this.saveClicked.emit();
  }

  closeExhibitCInfo() {
    this.closeClicked.emit();
  }

  getExhibitCInfo(): MainContractExhibitCDetail {
    const exhibitCItems = this.exhibitCInfoDetailForm.value as MainContractExhibitCDetail;
    return exhibitCItems;
  }

  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => this.loading$.next(true)),
    switchMap(refreshType => {
      return combineLatest([this.service.requiredData$, of(refreshType)]);
    }),
    map(([requiredData]) => {
      this.localRequiredData = requiredData;
      this.loading$.next(false);
      return requiredData;
    }),
    tap((requiredData) => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  exhibitC$ = this.refreshExhibitCInfo$.pipe(
    tap(() => {
      this.loading$.next(true);
    }),
    switchMap(() => {
      const getInternalSide: boolean = this.entityType() === InfoType.Internal;
      return this.service.getExhibitCInfo(this.contractId(), getInternalSide, this.entityId(), this.productId());
    }),
    map(detail => {
      util.convertToDates(detail);
      this.exhibitCInfoDetailForm.setValue(detail);
      return detail;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  loadingComplete$ = zip(this.exhibitC$, this.requiredData$).pipe(
    tap(() => {
      this.loading$.next(false);
    })
  )

  filterStates$ = new BehaviorSubject<string>(null)
  states$ = this.filterStates$.pipe(util.filterIdNames(this.loading$, this.requiredData$, 'states'));

  filterCountries$ = new BehaviorSubject<string>(null)
  countries$ = this.filterCountries$.pipe(util.filterIdNames(this.loading$, this.requiredData$, 'countries'));
}
