import { Component, ChangeDetectionStrategy, ChangeDetectorRef, output, input, OnInit, ViewEncapsulation, inject } from '@angular/core';
import { tap, map, catchError, switchMap, filter, shareReplay, retry, take } from 'rxjs/operators';
import { of, BehaviorSubject, combineLatest, Subject, zip } from 'rxjs';
import { MessageService } from '../../_shared/services/message.service';
import { DialogService, DialogSettings } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../../_shared/services/notify.service';
import * as util from '../../_shared/utils/util';
import { ExhibitBInfoService, MainContractExhibitBDetail, RequiredData, MainContractExhibitBItem, MainContractExhibitBInfo, MainContractExhibitBSelectionDetail } from './exhibitb-info.service';
import { CustomFormBuilder } from '../../_shared/services/custom-form-builder.service';
import { UntypedFormGroup, Validators } from '@angular/forms';
import { CheckableSettings, CheckedState, KENDO_TREEVIEW, TreeItem, TreeItemDragEvent } from '@progress/kendo-angular-treeview';
import dayjs from 'dayjs';
import { InfoType } from '../main-contract.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../../app.config';
import { HesitateDirective } from '../../_shared/directives/hesitate-directive';
import { KENDO_SPLITTER } from '@progress/kendo-angular-layout';

@Component({
  selector: 'app-exhibitb-info',
  templateUrl: './exhibitb-info.component.html',
  styleUrl: './exhibitb-info.component.scss',
  changeDetection: ChangeDetectionStrategy.OnPush,
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON, KENDO_TREEVIEW, KENDO_SPLITTER, HesitateDirective]
})
export class ExhibitBInfoComponent implements OnInit {
  contractId = input.required<number>();
  entityType = input<InfoType>(InfoType.Internal);
  saveClicked = output();
  closeClicked = output();

  private service = inject(ExhibitBInfoService);
  private messageService = inject(MessageService);
  private fb = inject(CustomFormBuilder);
  private ref = inject(ChangeDetectorRef);
  private dialogService = inject(DialogService);
  private notify = inject(NotifyService);


  constructor() {
  }

  util = util;
  icons = util.icons;
  selectedKeys: (number | string)[] = [];
  expandedKeys: (number | string)[] = [];
  checkedKeys: number[] = [];
  checkableSettings: CheckableSettings = {
    checkChildren: true,
    checkParents: true,
    enabled: true,
    mode: 'multiple',
    checkOnClick: false,
  };

  localDetail: MainContractExhibitBInfo;
  localRequiredData: RequiredData;
  exhibitBInfoDetailForm = this.getExhibitBInfoDetailForm();
  exhibitBInfoInitialValues = this.exhibitBInfoDetailForm.value as MainContractExhibitBDetail;
  exhibitBSelectionDetailInitialValues: MainContractExhibitBSelectionDetail = { contractId: 0, exibhitBSelectedIds: [] };
  addNewExhibitBForm = this.getAddNewExhibitBForm();
  addNewExhibitBinitialValues: MainContractExhibitBItem = this.addNewExhibitBForm.value as MainContractExhibitBItem;
  exhibitBItemsToDelete: MainContractExhibitBItem[] = [];
  hasModifyPermission = false;
  refreshRequiredData$ = new BehaviorSubject(null);
  loading$ = new BehaviorSubject<boolean>(true);
  refreshExhibitB$ = new BehaviorSubject<number>(null);
  isExhibitBReloading$ = new BehaviorSubject<boolean>(false);
  inactiveDetailOpened$ = new BehaviorSubject<boolean>(false);
  inactiveSectionOpened$ = new BehaviorSubject<boolean>(false);
  inactiveDateForm: UntypedFormGroup;
  inactiveSectionForm: UntypedFormGroup;
  isAddingSubSection: boolean = false;
  addExhibitBClicked$ = new Subject();
  removeExhibitClicked$ = new Subject();
  toggleInactiveExhibitClicked$ = new Subject<Date>();
  addNewExhibitBOpened: boolean;

  ngOnInit(): void {
    const fb = this.fb;
    this.inactiveDateForm = fb.group({
      inactiveDate: [null],
      fileNameOnDisk: [null]
    });

    this.inactiveSectionForm = fb.group({
      inactiveDate: [null, Validators.required]
    });
    this.refreshExhibitB$.next(this.contractId());
  }

  getAddNewExhibitBForm() {
    const fb = this.fb;
    const fg: util.FormModel<MainContractExhibitBItem> = fb.group({
      id: fb.ctr(null),
      title: fb.ctr(null),
      body: fb.ctr(null),
      ordinal: fb.ctr(null),
      isContractSpecific: fb.ctr(false),
      items: fb.ctr([]),
      parentId: fb.ctr(null),
      inactiveDate: fb.ctr(null),
    });
    return fg;
  }

  getExhibitBInfoDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<MainContractExhibitBDetail> = fb.group({
      title: fb.ctr(''),
      body: fb.ctr(''),
      showInactive: fb.ctr(false),
      items: fb.ctr([]),
      exhibitBAmendmentDate: fb.ctr(null),
    });
    return fg;
  }

  saveExhibitBInfo() {
    this.saveClicked.emit();
  }

  closeExhibitBInfo() {
    this.closeClicked.emit();
  }

  getExhibitBInfo(): MainContractExhibitBDetail {
    const exhibitBItems = this.exhibitBInfoDetailForm.value as MainContractExhibitBDetail;
    return exhibitBItems;
  }

  getExhibitBParentOfSelectedItem(allItems: MainContractExhibitBItem[]): MainContractExhibitBItem {
    const selectedId = this.selectedKeys[0];
    let selectedItem: MainContractExhibitBItem;
    allItems.forEach(topItem => {
      if (topItem.id === selectedId)
        selectedItem = topItem;
      else if (topItem.items.length > 0) {
        topItem.items.forEach(subItem => {
          if (subItem.id === selectedId)
            selectedItem = topItem;
        });
      }
    });
    return selectedItem;
  }

  getExhibitBSelectedItem(): MainContractExhibitBItem {
    const items: MainContractExhibitBItem[] = this.exhibitBInfoDetailForm.get('items').value;
    const selectedId = this.selectedKeys[0];
    let selectedItem: MainContractExhibitBItem;
    items.forEach(topItem => {
      if (topItem.id === selectedId)
        selectedItem = topItem;
      else if (topItem.items.length > 0) {
        topItem.items.forEach(subItem => {
          if (subItem.id === selectedId)
            selectedItem = subItem;
        });
      }
    });
    return selectedItem;
  }

  openAddNewExhibitB(isAddingSubSection: boolean) {
    this.isAddingSubSection = isAddingSubSection;
    this.addNewExhibitBForm.setValue(this.addNewExhibitBinitialValues);
    this.addNewExhibitBOpened = true;
    util.focusInputByClassName('inputFocusTargetAdd', false);
  }

  addNewExhibitBCompleted() {
    const newTitle: string = this.addNewExhibitBForm.get('title').value;
    if (!util.isNullOrWhitespace(newTitle)) {
      this.addNewExhibitBOpened = false;
      this.addExhibitBClicked$.next(null);
    }
    else {
      this.notify.error("Title is required");
      return false;
    }
  }

  removeExhibit() {
    const deleteConfirmSettings: DialogSettings = {
      title: "Please confirm",
      content: "Are you sure you want to remove this item?",
      actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
      cssClass: 'utilPrompt'
    }
    this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
      if (util.getDialogAction(result) === util.dialogAction.Yes)
        this.removeExhibitClicked$.next(null);
    });
  }

  // Custom logic handling Indeterminate state when custom data item property is persisted
  isChecked = (dataItem: object): CheckedState => {
    const exhibitBItem = dataItem as MainContractExhibitBItem;
    if (this.containsItem(exhibitBItem)) {
      return "checked";
    }

    if (this.isIndeterminate(exhibitBItem.items)) {
      return "indeterminate";
    }

    return "none";
  };

  containsItem(item: MainContractExhibitBItem): boolean {
    return this.checkedKeys.indexOf(item["id"]) > -1;
  }

  isIndeterminate(items: MainContractExhibitBItem[] = []): boolean {
    let idx = 0;
    let item;

    while ((item = items[idx])) {
      if (this.isIndeterminate(item.items) || this.containsItem(item)) {
        return true;
      }

      idx += 1;
    }

    return false;
  }

  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  sectionDragEnd(event: TreeItemDragEvent) {
    this.exhibitBInfoDetailForm.get('title').markAsDirty();
  }

  exhibitSelected(treeItem: TreeItem): void {
    if (treeItem?.dataItem !== null) {
      const exhibitBitem: MainContractExhibitBItem = treeItem.dataItem;
      this.exhibitBInfoDetailForm.patchValue({ title: exhibitBitem.title, body: exhibitBitem.body });
    }
    this.exhibitBInfoDetailForm.get('title').enable();
    this.exhibitBInfoDetailForm.get('body').enable();
  }

  isSectionSelected(): boolean {
    const isSelected = util.isNumber(this.selectedKeys[0])
    return isSelected;
  }

  toggleInactiveExhibit() {
    const item: MainContractExhibitBItem = this.getExhibitBSelectedItem();
    this.inactiveSectionForm.reset();
    if (item.inactiveDate !== null)
      this.toggleInactiveExhibitClicked$.next(null);
    else {
      this.inactiveSectionOpened$.next(true);
      this.inactiveSectionForm.patchValue({ inactiveDate: util.currentDate.toDate() });
    }
  }

  onInactiveSectionClosing() {
    this.inactiveSectionForm.reset();
    this.inactiveSectionOpened$.next(false);
  }

  setInactiveSection() {
    if (this.inactiveSectionForm.valid) {
      let inactiveDate: Date = this.inactiveSectionForm.get('inactiveDate').value;
      inactiveDate = dayjs(inactiveDate).startOf('day').toDate();
      this.toggleInactiveExhibitClicked$.next(inactiveDate);
      this.inactiveSectionOpened$.next(false);
    }
    else
      this.notify.error("validation failed");
  }

  getExhibitBSelectionDetail(): MainContractExhibitBSelectionDetail {
    const contractId = this.contractId();
    const detail: MainContractExhibitBSelectionDetail = {
      contractId: contractId,
      exibhitBSelectedIds: this.checkedKeys
    }
    return detail;
  }

  getItemsForDeletion(): MainContractExhibitBItem[] {
    return this.exhibitBItemsToDelete;
  }

  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => this.loading$.next(true)),
    switchMap(refreshType => {
      return combineLatest([this.service.requiredData$, of(refreshType)]);
    }),
    map(([requiredData]) => {
      this.localRequiredData = requiredData;
      this.loading$.next(false);
      return requiredData;
    }),
    tap((requiredData) => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  exhibitBTitleChanged$ = this.exhibitBInfoDetailForm.get('title').valueChanges.pipe(
    filter(() => {
      return !this.loading$.value;
    }),
    tap((title: string) => {
      if (util.isNumber(this.selectedKeys[0])) {
        const selectedItem = this.getExhibitBSelectedItem();
        selectedItem.title = title;
      }
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(1)
  )

  exhibitBBodyChanged$ = this.exhibitBInfoDetailForm.get('body').valueChanges.pipe(
    filter(() => {
      return !this.loading$.value;
    }),
    tap((body: string) => {
      if (util.isNumber(this.selectedKeys[0])) {
        const selectedItem = this.getExhibitBSelectedItem();
        selectedItem.body = body;
      }
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(1)
  )

  addExhibitB$ = this.addExhibitBClicked$.pipe(
    tap(() => {
      this.loading$.next(true);
    }),
    switchMap(() => {
      const items = this.exhibitBInfoDetailForm.get('items').value;
      const parentOfselectedItem = this.isAddingSubSection ? this.getExhibitBParentOfSelectedItem(items) : null;
      const newTitle: string = this.addNewExhibitBForm.get('title').value;
      const newIsContract: boolean = this.addNewExhibitBForm.get('isContractSpecific').value;
      const newId: number = util.getRandomInt();
      const newItem: MainContractExhibitBItem = {
        id: newId, title: newTitle, body: "", ordinal: 90000, inactiveDate: null,
        isContractSpecific: newIsContract,
        items: [],
        parentId: parentOfselectedItem ? parentOfselectedItem.id : null
      }
      return of(newItem);
    }),
    tap((newItem) => {
      const items: MainContractExhibitBItem[] = this.exhibitBInfoDetailForm.value.items;
      if (!this.isAddingSubSection)
        items.push(newItem);
      else {
        const parentOfselectedItem = this.getExhibitBParentOfSelectedItem(items);
        parentOfselectedItem.items.push(newItem);
      }
      this.exhibitBInfoDetailForm.patchValue({ items: items });
    }),
    tap(() => {
      this.notify.success('Section Added');
      this.loading$.next(false);
      util.reloadTreeView(this.isExhibitBReloading$, this.ref);
    }),
    shareReplay(1),
    catchError(err => {
      this.loading$.next(false);
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  removeExhibit$ = this.removeExhibitClicked$.pipe(
    tap(() => {
      this.loading$.next(true);
    }),
    tap(() => {
      const items: MainContractExhibitBItem[] = this.exhibitBInfoDetailForm.value.items;
      const itemSelectedForDeletion = this.getExhibitBSelectedItem();
      if (itemSelectedForDeletion.isContractSpecific === false && this.contractId() != 0) {
        this.exhibitBItemsToDelete.push(itemSelectedForDeletion);
        const newItems = items.filter(item => item.id !== itemSelectedForDeletion.id);
        newItems.forEach(topItem => {
          topItem.items = topItem.items.filter(item => item.id !== itemSelectedForDeletion.id);
        });
        this.exhibitBInfoDetailForm.patchValue({ items: newItems });
      }
      else {
        const newItems = items.filter(item => item.id !== itemSelectedForDeletion.id);
        newItems.forEach(topItem => {
          topItem.items = topItem.items.filter(item => item.id !== itemSelectedForDeletion.id);
        });
        this.exhibitBInfoDetailForm.patchValue({ items: newItems });
      }
    }),
    tap(() => {
      this.exhibitBInfoDetailForm.patchValue({ title: '', body: '' });
      this.loading$.next(false);
      util.reloadTreeView(this.isExhibitBReloading$, this.ref);
      this.selectedKeys = [];
    }),
    shareReplay(1),
    catchError(err => {
      this.loading$.next(false);
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  toggleInactiveExhibit$ = this.toggleInactiveExhibitClicked$.pipe(
    tap((inactiveDate) => {
      this.loading$.next(true);
      const itemToToggle = this.getExhibitBSelectedItem();
      itemToToggle.inactiveDate = inactiveDate;
      this.notify.success('Inactivate Toggled');
      this.loading$.next(false);
      util.reloadTreeView(this.isExhibitBReloading$, this.ref);
    }),
    shareReplay(1),
    catchError(err => {
      this.loading$.next(false);
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  exhibitB$ = this.refreshExhibitB$.pipe(
    tap(() => {
      this.loading$.next(true);
    }),
    switchMap(contractId => {
      return this.service.getExhibitBInfo(contractId);
    }),
    map(detail => {
      util.convertToDates(detail, []);
      this.exhibitBInfoDetailForm.setValue(detail.exhibitBDetail);
      this.checkedKeys = detail.exhibitBSelectionDetail?.exibhitBSelectedIds ?? [];
      return detail;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  loadingComplete$ = zip(this.exhibitB$, this.requiredData$).pipe(
    tap(() => {
      this.loading$.next(false);
    })
  )
}
