import { Component, ChangeDetectionStrategy, ChangeDetectorRef, OnInit, input, output, ViewEncapsulation, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { tap, switchMap, catchError, retry, shareReplay } from 'rxjs/operators';
import { BehaviorSubject, zip } from 'rxjs';
import { MessageService } from '../../_shared/services/message.service';
import { DialogService } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../../_shared/services/notify.service';
import * as util from '../../_shared/utils/util'
import { ActivatedRoute } from '@angular/router';
import { CustomFormBuilder } from '../../_shared/services/custom-form-builder.service';
import { EntityInfoService, MainContractEntityDetail } from './entity-info.service';
import { InfoType } from '../main-contract.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../../app.config';

@Component({
  selector: 'app-entity-info',
  templateUrl: './entity-info.component.html',
  styleUrl: './entity-info.component.scss',
  changeDetection: ChangeDetectionStrategy.OnPush,
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON]
})

export class EntityInfoComponent implements OnInit {
  entityId = input.required<number>();
  entityType = input<InfoType>(InfoType.Internal);
  saveClicked = output();
  closeClicked = output();
  
  private service = inject(EntityInfoService);
  private messageService = inject(MessageService);
  private titleService = inject(Title);
  private fb = inject(CustomFormBuilder);
  private ref = inject(ChangeDetectorRef);
  private dialogService = inject(DialogService);
  private notify = inject(NotifyService);
  private activatedRoute = inject(ActivatedRoute)

  constructor() {

  }

  util = util;
  icons = util.icons;
  hasModifyPermission = false;
  refreshRequiredData$ = new BehaviorSubject(null);
  isCounterparty: boolean = false;
  entityInfoDetailForm = this.getEntityInfoDetailForm();
  loading$ = new BehaviorSubject<boolean>(true);
  refreshEntity$ = new BehaviorSubject<number>(null);

  ngOnInit(): void {
    this.refreshEntity$.next(this.entityId());
  }

  getEntityInfoDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<MainContractEntityDetail> = fb.group({
      companyName: fb.ctr(null),
      country: fb.ctr(null),
      addressLine1: fb.ctr(null),
      addressLine2: fb.ctr(null),
      city: fb.ctr(null),
      state: fb.ctr(null),
      zip: fb.ctr(null),
      dunsNum: fb.ctr(null),
      federalTaxId: fb.ctr(null),
      jurisdiction: fb.ctr(null),
      businessType: fb.ctr(null),
      relationships: fb.ctr(null)
    });
    return fg;
  }

  saveEntityInfo() {
    this.saveClicked.emit();
  }

  closeEntityInfo() {
    this.closeClicked.emit();
  }

  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => this.loading$.next(true)),
    switchMap(() => {
      return this.service.requiredData$;
    }),
    tap(requiredData => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  entity$ = this.refreshEntity$.pipe(
    tap(() => {
      this.loading$.next(true);
    }),
    switchMap(entityId => {
      return this.service.getEntityInfo(entityId);
    }),
    tap(detail => {
      this.entityInfoDetailForm.setValue(detail);
    }),
    shareReplay(1),
    catchError(err => {
      this.loading$.next(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  loadingComplete$ = zip(this.entity$, this.requiredData$).pipe(
    tap(() => {
      this.loading$.next(false);
    })
  )
}
