import { Component, ChangeDetectionStrategy, ChangeDetectorRef, output, input, OnInit, ViewChildren, QueryList, ViewEncapsulation, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { tap, map, catchError, switchMap, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, combineLatest, zip } from 'rxjs';
import { MessageService } from '../../_shared/services/message.service';
import { DialogService } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../../_shared/services/notify.service';
import * as util from '../../_shared/utils/util'
import { CrudeInfoService, MainContractCrudeDetail, RequiredData, MainContractCrudeCalcItem } from './crude-info.service';
import { ActivatedRoute } from '@angular/router';
import { CustomFormBuilder } from '../../_shared/services/custom-form-builder.service';
import { UntypedFormArray, Validators } from '@angular/forms';
import { NumericTextBoxComponent } from '@progress/kendo-angular-inputs';
import { InfoType } from '../main-contract.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../../app.config';
import { HoverClassDirective } from '../../_shared/directives/hover-class-directive';
import { HesitateDirective } from '../../_shared/directives/hesitate-directive';

@Component({
  selector: 'app-crude-info',
  templateUrl: './crude-info.component.html',
  styleUrl: './crude-info.component.scss',
  changeDetection: ChangeDetectionStrategy.OnPush,
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON, HoverClassDirective, HesitateDirective]
})
export class CrudeInfoComponent implements OnInit {
  @ViewChildren('crudeCalcElem') crudeCalcElems: QueryList<NumericTextBoxComponent>;

  contractId = input.required<number>();
  entityType = input<InfoType>(InfoType.Internal);
  saveClicked = output();
  closeClicked = output();

  private service = inject(CrudeInfoService);
  private messageService = inject(MessageService);
  private titleService = inject(Title);
  private fb = inject(CustomFormBuilder);
  private ref = inject(ChangeDetectorRef);
  private dialogService = inject(DialogService);
  private notify = inject(NotifyService);
  private activatedRoute = inject(ActivatedRoute);
  

  constructor() {

  }

  util = util;
  icons = util.icons;
  localRequiredData: RequiredData;
  hasModifyPermission = false;
  crudeInfoDetailForm = this.getCrudeInfoDetailForm();
  crudeInfoInitialValues = this.crudeInfoDetailForm.value as MainContractCrudeDetail;
  crudeInitialValues: MainContractCrudeDetail = this.crudeInfoDetailForm.value as MainContractCrudeDetail;
  crudeCalcItemsFormArray: UntypedFormArray;
  refreshRequiredData$ = new BehaviorSubject(null);
  loading$ = new BehaviorSubject<boolean>(true);
  refreshCrude$ = new BehaviorSubject<number>(null);

  ngOnInit(): void {
    this.refreshCrude$.next(this.contractId());
  }

  getCrudeInfoDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<MainContractCrudeDetail> = fb.group({
      calcItems: fb.arr([])
    });
    return fg;
  }

  getCrudeCalcItemFormGroup(calcItem: MainContractCrudeCalcItem) {
    const fb = this.fb;
    const fg: util.FormModel<MainContractCrudeCalcItem> = fb.group({
      pipeLossAllowance: fb.ctr(calcItem ? calcItem.pipeLossAllowance : 0, Validators.required),
      differential: fb.ctr(calcItem ? calcItem.differential : 0, Validators.required),
      isDiffBeforePipeLoss: fb.ctr(calcItem ? calcItem.isDiffBeforePipeLoss : false, Validators.required),
      pointIds: fb.ctr(calcItem ? calcItem.pointIds : []),
      effectiveDate: fb.ctr(calcItem ? calcItem.effectiveDate : util.currentDate.toDate(), Validators.required)
    });
    return fg;
  }

  saveCrudeInfo() {
    this.saveClicked.emit();
  }

  closeCrudeInfo() {
    this.closeClicked.emit();
  }

  getCrudeInfo(): MainContractCrudeDetail {
    const crudeItems = this.crudeInfoDetailForm.value as MainContractCrudeDetail;
    return crudeItems;
  }

  addCrudeCalcItem() {
    this.crudeCalcItemsFormArray.insert(0, this.getCrudeCalcItemFormGroup(null));
    this.crudeCalcItemsFormArray.markAsDirty();
    setTimeout(() => {
      this.crudeCalcElems.first.focus();
    });
  }

  removeCrudeCalcItem(idx: number) {
    this.crudeCalcItemsFormArray.removeAt(idx);
    this.crudeCalcItemsFormArray.markAsDirty();
  }

  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => this.loading$.next(true)),
    switchMap(refreshType => {
      return combineLatest([this.service.requiredData$, of(refreshType)]);
    }),
    map(([requiredData]) => {
      this.localRequiredData = requiredData;
      this.loading$.next(false);
      return requiredData;
    }),
    tap((requiredData) => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  crude$ = this.refreshCrude$.pipe(
    tap(() => {
      this.loading$.next(true);
    }),
    switchMap(contractId => {
      if (contractId === 0)
        return of(this.crudeInitialValues);
      else
        return this.service.getCrudeInfo(contractId);
    }),
    map(detail => {
      util.convertToDates(detail, []);
      this.crudeCalcItemsFormArray = this.crudeInfoDetailForm.get('calcItems') as UntypedFormArray;
      this.crudeCalcItemsFormArray.clear();
      if (detail.calcItems.length === 0) {
        const newCalcItem: MainContractCrudeCalcItem = { pipeLossAllowance: 0, differential: 0, isDiffBeforePipeLoss: false, effectiveDate: util.currentDate.toDate(), pointIds: [] };
        detail.calcItems.push(newCalcItem);
        detail.calcItems.forEach(calcItem => {
          this.crudeCalcItemsFormArray.push(this.getCrudeCalcItemFormGroup(calcItem));
        });
        this.crudeInfoDetailForm.setValue(detail);
      }
      else {
        detail.calcItems.forEach(calcItem => {
          this.crudeCalcItemsFormArray.push(this.getCrudeCalcItemFormGroup(calcItem));
        });
        this.crudeInfoDetailForm.setValue(detail);
      }
      return detail;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  loadingComplete$ = zip(this.crude$, this.requiredData$).pipe(
    tap(() => {
      this.loading$.next(false);
    })
  )

  filterStates$ = new BehaviorSubject<string>(null)
  states$ = this.filterStates$.pipe(util.filterIdNames(this.loading$, this.requiredData$, 'states'));

  filterCountries$ = new BehaviorSubject<string>(null)
  countries$ = this.filterCountries$.pipe(util.filterIdNames(this.loading$, this.requiredData$, 'countries'));

  filterCrudePoints$ = new BehaviorSubject<string>(null)
  crudePoints$ = this.filterCrudePoints$.pipe(util.filterIdNames(this.loading$, this.requiredData$, 'crudePoints'));
}
